/*
 *   DIS/x : An implementation of the IEEE 1278.1 protocol
 *
 *   Copyright (C) 1996, Riley Rainey (rainey@netcom.com)
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of either:
 *
 *   a) the GNU Library General Public License as published by the Free
 *   Software Foundation; either version 2 of the License, or (at your
 *   option) any later version.  A description of the terms and conditions
 *   of the GLPL may be found in the "COPYING.LIB" file.
 *
 *   b) the "Artistic License" which comes with this Kit.  Information
 *   about this license may be found in the "Artistic" file.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License or the Artistic License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Information describing how to contact the author can be found in the
 *   README file.
 */

/**
 * Interface between the lower layer (see dis module) and upper layer (see disx
 * module).
 * 
 * @file
 * @author Riley Rainey (rainey@netcom.com)
 * @version $Date: 2017/05/19 16:16:54 $
 */

#ifndef DISX_H
#define DISX_H

#include "dis.h"

#ifdef disx_IMPORT
	#define EXTERN
#else
	#define EXTERN extern
#endif

typedef struct {
	dis_Transceiver *xcvr;
	dis_pdu_header hdr;
	dis_simulation_addr id;
	u_short   last_event;
	u_long    last_request;
	u_short   last_entity;
} disx_ApplicationInfo;

/**
 * Establishes the application’s connection to the DIS Network; also set the
 * non-blocking mode on the transceiver socket.
 * @param xcvr DIS transceiver.
 * @param exercise_id DIS exercise number in [0,255].
 * @param site_id Site number in [0,65535].
 * @param application_id The DIS application id in [0,65535].
 * @return Initialized application information structure, or NULL if fails to
 * set the non-blocking mode.
 */
EXTERN disx_ApplicationInfo * disx_initializeApplication(
	dis_Transceiver *xcvr,
	int exercise_id,
	int site_id,
	int application_id);

/**
 * Closes the application context and the underlying DIS transceiver.
 * Does nothing if info is NULL.
 * @param info
 */
EXTERN void disx_closeApplication(disx_ApplicationInfo *info);

/**
 * Returns the next PDU received from the DIS network. If no packet is available,
 * it returns immediately.
 * @param app The application information structure returned by
 * disx_initializeApplication().
 * @param pdu
 * @return True if a DIS PDU is available. False if no more packets are
 * available. Invalid packets are skipped and an error message is logged.
 */
EXTERN int disx_readPDU(disx_ApplicationInfo *app, dis_pdu *pdu);

/**
 * Broadcasts the specified PDU onto the DIS network. It automatically fills in
 * the following fields in the PDU:
 * - protocol_version
 * - exercise_id
 * - protocol_family
 * - length (always filled automatically)
 * @param app The pointer to the application information structure returned by
 * disx_initializeApplication().
 * @param pdu A pointer to the protocol data unit to be transmitted. Certain
 * structure members are automatically inserted when disx_writePDU() is called.
 * Those values are updated within the referenced structure.
 * @return True on success. False if failed sending the packet to at least one
 * destination address. Failing to encode the PDU is fatal.
 */
EXTERN int disx_writePDU(disx_ApplicationInfo *app, dis_pdu *pdu);

/**
 * Returns DIS simulation address in-use by this application.
 * @param app The application information structure returned by
 * disx_initializeApplication().
 * @param p Simulation address.
 */
EXTERN void disx_getSimulationAddress(disx_ApplicationInfo * app,
	dis_simulation_addr * p);

/**
 * Allocate a unique DIS event ID for use by this application.
 * @param app The application information structure returned by
 * disx_initializeApplication().
 * @param event A pointer to an empty dis_event_id structure. It will be filled
 * in by this function.
 * @return Value of “event” that was passed to it.
 */
EXTERN dis_event_id * disx_issueEventID(disx_ApplicationInfo * app,
	dis_event_id * event);

/**
 * Allocate a unique DIS entity ID for use by this application. The entity id
 * returned by the first call will be 0x0000. The number returned on subsequent
 * calls is always one greater than the previous call.
 * @param app The application information structure returned by
 * disx_initializeApplication().
 * @param p A pointer to an empty dis_entity_id structure. It will be filled in
 * by this function.
 * @return Value of “p” that was passed to it.
 */
EXTERN dis_entity_id *disx_issueEntityID(disx_ApplicationInfo * app,
	dis_entity_id * p);

EXTERN dis_request_id disx_issueRequestID(disx_ApplicationInfo * app);

EXTERN int disx_setProtocolVersion(int version);

EXTERN void disx_initializeDatumInfo(dis_datum_spec_record *pd);

#undef EXTERN
#endif
