# -*- coding: utf-8 -*-

# Bluemindo: A really simple but powerful audio player in Python/PyGTK.
# Copyright (C) 2007-2009  Erwan Briand

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation version 3 of the License.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from gobject import timeout_add, markup_escape_text
from threading import Thread
from gettext import gettext as _
from gtk.gdk import Pixbuf as gPixbuf
from gtk.glade import XML as glade_XML
from gtk import (TreeViewColumn, ListStore, TreeStore, CellRendererText,
                 STOCK_DIRECTORY, STOCK_FILE, STOCK_ABOUT, CellRendererPixbuf,
                 ICON_SIZE_MENU, TREE_VIEW_COLUMN_FIXED)
from cPickle import dump, load
from urllib2 import urlopen
from os.path import isfile, join, exists
from os import makedirs, remove

from common.webservices import Shoutcast
from common.functions import Functions

from modules.explorer.playlist import Playlist
from modules.explorer import viewscommon
from common.sqlite import SQLite
unescape = viewscommon.unescape

class WebradiosView(object):
    ref = None
    ref2 = None

    def __new__(cls, *args, **kws):
        # Singleton
        if cls.ref is None:
            cls.ref = super(WebradiosView, cls).__new__(cls, args, kws)
        return cls.ref

    def __init__(self, glade_file, config, userconf):
        # This playlist will not be used
        playlist = Playlist(glade_file, config, userconf)
        glade_file.get_widget('vpaned1').hide()
        glade_file.get_widget('scrolledwindow3').hide()
        glade_file.get_widget('toolbar4').hide()

        # Variables
        self.shoutcast = Shoutcast()
        self.glade_file = glade_file
        self.config = config
        self.userconf = userconf

        self.functions = Functions()

        # Show widgets
        self.glade_file.get_widget('vbox2').show()
        self.glade_file.get_widget('scrolledwindow10').show()
        self.glade_file.get_widget('scrolledwindow11').show()
        self.glade_file.get_widget('toolbar6').show()
        self.glade_file.get_widget('tool-radios-add').hide()
        self.glade_file.get_widget('tool-radios-delete').hide()
        self.glade_file.get_widget('tool-radios-refresh').hide()

        if WebradiosView.ref2 is None: 
            WebradiosView.ref2 = 42 
            self.launch()

    def reload_database(self, force_scan=False):
        pass

    def launch(self):
        # Start GUI
        tview = self.glade_file.get_widget('treeview8')
        tview.connect('row-activated', self.load_radios_category)
        treestore = TreeStore(gPixbuf, str, str)
        tview.set_model(treestore)

        column = TreeViewColumn()
        render_text = CellRendererText()
        render_pixbuf = CellRendererPixbuf()
        column.pack_start(render_pixbuf, expand=False)
        column.add_attribute(render_pixbuf, 'pixbuf', 0)
        column.pack_start(render_text, expand=True)
        column.add_attribute(render_text, 'markup', 1)
        tview.append_column(column)

        # Append
        fav_title = '<b>%s</b>' % markup_escape_text(_('Bookmarks'))
        web_title = '<b>%s</b>' % markup_escape_text(_('World Wide Web'))

        logo = tview.render_icon(STOCK_ABOUT, ICON_SIZE_MENU)
        it_fav = treestore.append(None, (logo, fav_title, 'fav'))
        logo = tview.render_icon(STOCK_DIRECTORY, ICON_SIZE_MENU)
        it_web = treestore.append(None, (logo, web_title, 'web'))

        # Add categories
        pbar = self.glade_file.get_widget('progressbar1')
        pbar.set_text(_('Downloading…'))
        pbar.show()

        thread = Thread(group=None, target=self.add_categories,
                        name='categories', args=(it_web, treestore, tview))
        self.thread_name = thread

        timeout_add(100, self.timeout_progress, pbar)
        thread.start()

        # Radios tree
        tview = self.glade_file.get_widget('treeview9')
        tview.connect('row-activated', self.launch_radio)
        width = tview.get_allocation().width
        self.r_tstore = ListStore(str, str, int, str)
        self.r_tstore_f = self.r_tstore.filter_new(None)
        self.r_tstore_f.set_visible_func(self.filter_visible_radios)
        tview.set_model(self.r_tstore_f)

        column = TreeViewColumn()
        column.pack_start(render_text, expand=True)
        column.add_attribute(render_text, 'markup', 0)
        column.set_title(_('Name'))
        column.set_expand(True)
        column.set_min_width(int(width / 100. * 75.))
        column.set_sizing(TREE_VIEW_COLUMN_FIXED)
        tview.append_column(column)

        column = TreeViewColumn()
        column.pack_start(render_text, expand=False)
        column.add_attribute(render_text, 'markup', 1)
        column.set_title(_('Listener count'))
        column.set_max_width(int(width / 100. * 25.))
        tview.append_column(column)

        # GTK signals
        entry_vi = self.glade_file.get_widget('entry2')
        tool_add = self.glade_file.get_widget('tool-radios-add')
        tool_del = self.glade_file.get_widget('tool-radios-delete')
        tool_ref = self.glade_file.get_widget('tool-radios-refresh')

        entry_vi.connect('key-release-event', self.radios_filter,
                                              self.r_tstore_f)
        tool_add.connect('clicked', self.add_a_radio)
        tool_del.connect('clicked', self.delete_a_radio)
        tool_ref.connect('clicked', self.refresh_a_category)

    def add_a_radio(self, widget):
        """This function add a bookmark radio."""
        txt = '<span size="small">%s</span>'
        newwin = glade_XML(join(self.functions.datadir, 'glade',
                           'mainwindow.glade'), 'dialog1', domain='bluemindo')
        new_win = newwin.get_widget('dialog1')
        new_win.resize(200, 150)
        response = new_win.run()

        title = newwin.get_widget('entry3').get_text()
        url = newwin.get_widget('entry4').get_text()
        if response and title != '' and url != '':
            sql = SQLite()
            sql.execute('insert into radios (name, url) '
                        'values (?, ?)', (title, url))

            fnm = url.split('.')
            ext = fnm[len(fnm)-1]

            self.r_tstore.append((txt % markup_escape_text(title),
                                  txt % markup_escape_text(url),
                                  0, ext))

            sql.close()
        new_win.destroy()

    def delete_a_radio(self, widget):
        """This function deletes a bookmark radio."""
        tview = self.glade_file.get_widget('treeview9')
        tstore = tview.get_model()
        (mod, iter_) = tview.get_selection().get_selected()

        if iter_ is not None:
            title = self.functions.clear_html(unescape(tstore[iter_][0]))
            url = self.functions.clear_html(unescape(tstore[iter_][1]))

            sql = SQLite()
            sql = sql.execute('delete from radios where '
                              'name=? and url=?', (title, url))
            sql.close()
            del mod[iter_]

    def refresh_a_category(self, widget):
        """This function refresh an already downloaded category."""
        val = self.current_category
        genre_hash = self.functions.get_hash(val, 'webradios')
        catfile = join(self.config['__data-dir'], 'webradios', genre_hash)
        if isfile(catfile):
            remove(catfile)

        pbar = self.glade_file.get_widget('progressbar1')
        pbar.set_text(_('Downloading…'))
        pbar.show()

        tview = self.glade_file.get_widget('treeview9')
        treestore = tview.get_model()

        thread = Thread(group=None, target=self.add_radios,
                        name='update_radios', args=(val, treestore, tview))
        self.thread_name = thread

        timeout_add(100, self.timeout_progress, pbar)
        thread.start()

    def timeout_progress(self, pbar):
        if self.thread_name.isAlive():
            pbar.pulse()
            return True
        else:
            pbar.hide()

    def add_categories(self, it_web, treestore, tview):
        """This function add all categories."""
        logo = tview.render_icon(STOCK_FILE, ICON_SIZE_MENU)
        radios_dir = join(self.config['__data-dir'], 'webradios')
        categories_file = join(radios_dir, 'categories')

        if not isfile(categories_file):
            # Create directory
            if not exists(radios_dir):
                makedirs(radios_dir)

            # Get shoutcast list
            categories = self.shoutcast.get_categories()
            dump(categories, open(categories_file, 'w'))
        else:
            categories = load(open(categories_file, 'r'))

        for category in categories:
            treestore.append(it_web, (logo, markup_escape_text(category),
                                            markup_escape_text(category)))

    def load_radios_category(self, tview, path, column):
        """This function handles row activation."""
        (mod, iter_) = tview.get_selection().get_selected()
        iter_has_depth = tview.get_model().iter_depth(iter_)
        val = unescape(mod.get_value(iter_, 2))

        if not iter_has_depth and val == 'fav':
            # GUI update
            self.glade_file.get_widget('tool-radios-add').show()
            self.glade_file.get_widget('tool-radios-delete').show()
            self.glade_file.get_widget('tool-radios-refresh').hide()
            self.r_tstore.clear()

            sql = SQLite()
            radios = sql.execute('select * from radios')

            txt = '<span size="small">%s</span>'
            for radio in radios:
                fnm = radio[1].split('.')
                ext = fnm[len(fnm)-1]

                self.r_tstore.append((txt % markup_escape_text(radio[0]),
                                      txt % markup_escape_text(radio[1]),
                                      0, ext))

            sql.close()
        else:
            self.current_category = val
            # GUI update
            self.glade_file.get_widget('tool-radios-add').hide()
            self.glade_file.get_widget('tool-radios-delete').hide()
            self.glade_file.get_widget('tool-radios-refresh').show()

            # Add radios
            pbar = self.glade_file.get_widget('progressbar1')
            pbar.set_text(_('Downloading…'))
            pbar.show()

            tview = self.glade_file.get_widget('treeview9')
            treestore = tview.get_model()

            thread = Thread(group=None, target=self.add_radios,
                            name='radios', args=(val, treestore, tview))
            self.thread_name = thread

            timeout_add(100, self.timeout_progress, pbar)
            thread.start()

    def add_radios(self, genre, tstore, tview):
        """This function add all channels of a category."""
        txt = '<span size="small">%s</span>'
        genre_hash = self.functions.get_hash(genre, 'webradios')
        radios_file = join(self.config['__data-dir'], 'webradios', genre_hash)

        if not isfile(radios_file):
            # Get shoutcast list
            radios = self.shoutcast.get_radios_by_category(genre)
            dump(radios, open(radios_file, 'w'))
        else:
            radios = load(open(radios_file, 'r'))

        # Sort the radios
        def add_zero(integer):
            integer = '0000000' + str(integer)
            return int(integer)

        radios.sort(lambda x,y:cmp(add_zero(x[2]), add_zero(y[2])))
        radios.reverse()

        # Add radios
        self.r_tstore.clear()
        for radio in radios:
            self.r_tstore.append((
                           txt % markup_escape_text(self.map_str(radio[0])),
                           txt % str(radio[2]), int(radio[1]), 'shoutcast'))

    def launch_radio(self, tview, path, column):
        """This function handles row activation."""
        (mod, iter_) = tview.get_selection().get_selected()
        radio_type = unescape(mod.get_value(iter_, 3))
        radio_id = unescape(str(mod.get_value(iter_, 2)))
        radio_title = unescape(mod.get_value(iter_, 0))
        title = self.map_str(self.functions.clear_html(radio_title), 35)

        if radio_type == 'shoutcast':
            pls = self.shoutcast.get_radio_url(radio_id)
            url = self.shoutcast.get_file_in_pls(pls)
        elif radio_type == 'm3u':
            pls = self.functions.clear_html(unescape(mod.get_value(iter_, 1)))
            url = self.get_url_in_m3u(pls)
        else:
            print '%s is not handled by Bluemindo.' % radio_type
            return

        radio_infos = (title, url)
        self.config['__extensions'].load_event('OnPlayNewRadio', radio_infos)

    def map_str(self, string, limit=70):
        """This function returns the string with a limited size."""
        if len(string) > limit:
            return string[:limit] + '…'
        else:
            return string

    def get_url_in_m3u(self, pls):
        """This function get the URL in a m3u file."""
        urlo = urlopen(pls)
        content = urlo.read().split('\n')
        for line in content:
            if line.startswith('http://'):
                return line

    def filter_visible_radios(self, model, iter_):
        """This function returns True if the row have to be shown."""
        if len(model) > 0:
            search = self.glade_file.get_widget('entry2').get_text()
            searchin = model.get_value(iter_, 0)
            if search and searchin:
                if search.upper() in searchin.upper():
                    return True
            else:
                return True

    def radios_filter(self, entry, event, tstore):
        """This function forces the filter."""
        tstore.refilter()
