//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Sample/ISelectionForm.h
//! @brief     Defines class ISelectionForm, and implements templated functions.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_VIEW_SAMPLE_ISELECTIONFORM_H
#define BORNAGAIN_GUI_VIEW_SAMPLE_ISELECTIONFORM_H

#include "GUI/Model/Descriptor/PolyItem.h"
#include "GUI/View/Base/CustomEventFilters.h"
#include "GUI/View/Sample/SampleEditorController.h"
#include <QComboBox>
#include <QGridLayout>

//! Abstract widget base class to contain a selection, defined by a SelectionDescriptor.
//!
//! This abstract base class contains only the selection combo box and the infrastructure to
//! call the SampleEditorController to change the current selection. A derived class has to
//! implement the showing of the contents (method createContent).
class ISelectionForm : public QWidget {
public:
    virtual void createContent() = 0;

protected:
    ISelectionForm(QWidget* parent, SampleEditorController* ec);

    template <typename T> void initUI(PolyItem<T>& d)
    {
        m_grid_layout = new QGridLayout(this);
        m_grid_layout->setContentsMargins(0, 0, 0, 0);
        m_grid_layout->setSpacing(6);

        m_combo = new QComboBox;
        WheelEventEater::install(m_combo);
        m_combo->addItems(d.menuEntries());
        m_combo->setCurrentIndex(d.currentIndex());
        m_combo->setMaxVisibleItems(m_combo->count());

        QObject::connect(m_combo, &QComboBox::currentIndexChanged, [this, &d](int current) {
            clear();
            d.setCurrentIndex(current);
            createContent();
            emit m_ec->modified();
        });

        m_grid_layout->addWidget(m_combo, 1, 0);
        createContent();
    }

private:
    //! Remove all properties from the layout
    void clear();

protected:
    QGridLayout* m_grid_layout;
    QComboBox* m_combo;
    SampleEditorController* m_ec;
};

#endif // BORNAGAIN_GUI_VIEW_SAMPLE_ISELECTIONFORM_H
