//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Sample/Interface/LayerRoughness.cpp
//! @brief     Implements class LayerRoughness.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "Sample/Interface/LayerRoughness.h"
#include "Base/Py/PyFmt.h"
#include "Base/Util/Assert.h"
#include <gsl/gsl_sf_bessel.h>
#include <numbers>

using std::numbers::pi;

//! Constructor of layer roughness.
//! @param sigma: rms of the roughness in nanometers
//! @param hurstParameter: hurst parameter which describes how jagged the interface,
//! dimensionless [0.0, 1.0], where 0.0 gives more spikes, 1.0 more smoothness
//! @param lateralCorrLength: lateral correlation length of the roughness in nanometers
//! @param profileModel: shape of the interfacial transition region
LayerRoughness::LayerRoughness(double sigma, double hurstParameter, double lateralCorrLength,
                               const RoughnessModel* roughnessModel)
    : m_sigma(sigma)
    , m_hurst_parameter(hurstParameter)
    , m_lateral_corr_length(lateralCorrLength)
{
    setRoughnessModel(roughnessModel);
    validateOrThrow();
}

LayerRoughness::LayerRoughness(double sigma, const RoughnessModel* roughnessModel)
    : LayerRoughness(sigma, 0, 0, roughnessModel)
{
}

LayerRoughness::LayerRoughness()
    : LayerRoughness(0, 0, 0, nullptr)
{
}

LayerRoughness* LayerRoughness::clone() const
{
    return new LayerRoughness(m_sigma, m_hurst_parameter, m_lateral_corr_length,
                              m_roughness_model.get());
}

//! Power spectral density of the surface roughness is a result of two-dimensional
//! Fourier transform of the correlation function of the roughness profile.
//!
//! Based on Palasantzas, Phys Rev B, 48, 14472 (1993)
double LayerRoughness::spectralFunction(const R3 kvec) const
{
    ASSERT(m_validated);
    double H = m_hurst_parameter;
    double clength2 = m_lateral_corr_length * m_lateral_corr_length;
    double Qpar2 = kvec.x() * kvec.x() + kvec.y() * kvec.y();
    return 4.0 * pi * H * m_sigma * m_sigma * clength2 * std::pow(1 + Qpar2 * clength2, -1 - H);
}

//! Correlation function of the roughness profile
double LayerRoughness::corrFunction(const R3 k) const
{
    ASSERT(m_validated);
    double H = m_hurst_parameter;
    double clength = m_lateral_corr_length;
    double R = sqrt(k.x() * k.x() + k.y() * k.y());
    return m_sigma * m_sigma * std::pow(2., 1 - H) / tgamma(H) * std::pow(R / clength, H)
           * gsl_sf_bessel_Knu(H, R / clength);
}

std::string LayerRoughness::pythonConstructor() const
{
    return Py::Fmt::printFunction(
        "LayerRoughness", {{m_sigma, ""}, {m_hurst_parameter, ""}, {m_lateral_corr_length, "nm"}});
}

std::string LayerRoughness::validate() const
{
    std::vector<std::string> errs;
    requestGe0(errs, m_sigma, "sigma");
    if (m_sigma > 0) {
        requestIn(errs, m_hurst_parameter, "hurst", 0, 1);
        requestGe0(errs, m_lateral_corr_length, "lateralCorrLength"); // TODO why not gt0 ?
    }
    if (!errs.empty())
        return jointError(errs);
    m_validated = true;
    return "";
}

void LayerRoughness::setRoughnessModel(const RoughnessModel* roughnessModel)
{
    if (!roughnessModel)
        m_roughness_model = std::make_unique<DefaultRoughness>();
    else
        m_roughness_model.reset(roughnessModel->clone());
}
