//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Sample/Interface/LayerRoughness.h
//! @brief     Defines class LayerRoughness.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_SAMPLE_INTERFACE_LAYERROUGHNESS_H
#define BORNAGAIN_SAMPLE_INTERFACE_LAYERROUGHNESS_H

#include "Sample/Interface/RoughnessModels.h"
#include "Sample/Scattering/ISampleNode.h"
#include <heinz/Vectors3D.h>

//! A roughness of interface between two layers.
//!
//! Based on the article "X-ray reflection and transmission by rough surfaces"
//! by D. K. G. de Boer, Physical Review B 51, 5297 (1995)

class LayerRoughness : public ISampleNode {
public:
    LayerRoughness(double sigma, double hurstParameter = 0, double lateralCorrLength = 0,
                   const RoughnessModel* roughnessModel = nullptr);
    LayerRoughness(double sigma, const RoughnessModel* roughnessModel);
    LayerRoughness();

    LayerRoughness* clone() const override;
    std::string className() const final { return "LayerRoughness"; }
    std::vector<ParaMeta> parDefs() const final
    {
        return {{"Sigma", ""}, {"Hurst", ""}, {"CorrLength", "nm"}};
    }

    //! Returns power spectral density of the surface roughness
    double spectralFunction(R3 kvec) const;

    double corrFunction(R3 k) const;

    //! Sets rms of roughness
    void setSigma(double sigma) { m_sigma = sigma; }
    //! Returns rms of roughness
    double sigma() const { return m_sigma; }

    //! Sets hurst parameter. It describes how jagged the surface is.
    void setHurstParameter(double hurstParameter) { m_hurst_parameter = hurstParameter; }
    //! Returns hurst parameter
    double hurst() const { return m_hurst_parameter; }

    //! Sets lateral correlation length
    void setLatteralCorrLength(double lateralCorrLength)
    {
        m_lateral_corr_length = lateralCorrLength;
    }
    //! Returns lateral correlation length
    double lateralCorrLength() const { return m_lateral_corr_length; }

#ifndef SWIG
    //! Creates the Python constructor of this class
    std::string pythonConstructor() const;

    bool allZero() const
    {
        return sigma() == 0 && hurst() == 0 && lateralCorrLength() == 0;
    }
#endif

    std::string validate() const override;

    void setRoughnessModel(const RoughnessModel* roughnessModel);
    const RoughnessModel* roughnessModel() const
    {
        return m_roughness_model.get();
    }

private:
    double m_sigma;           //!< rms of roughness
    double m_hurst_parameter; //!< Hurst parameter which describes how jagged the interface, 0<H<=1
    double m_lateral_corr_length; //!< lateral correlation length of the roughness
    std::unique_ptr<RoughnessModel> m_roughness_model; // never nullptr
};

#endif // BORNAGAIN_SAMPLE_INTERFACE_LAYERROUGHNESS_H
