/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Pressure    pressure_fl          Pressure on full hybrid levels
      Pressure    pressure_hl          Pressure on half hybrid levels
      Pressure    deltap               Difference of two half hybrid levels
*/

#include <cdi.h>

#include "cdo_options.h"
#include "process_int.h"
#include "cdo_vlist.h"
#include "vertical_interp.h"
#include "stdnametable.h"
#include "util_string.h"
#include "const.h"

void *
Pressure(void *process)
{
  ModelMode mode(ModelMode::UNDEF);
  int nrecs;
  int k;
  int varID, levelID;
  int zaxisIDp, zaxisIDh = -1;
  int nhlevf = 0, nhlevh = 0, nlevel = 0;
  int nvct = 0;
  size_t nmiss;
  int psID = -1, lnpsID = -1;
  char paramstr[32];
  char varname[CDI_MAX_NAME];
  double *pout = nullptr;
  gribcode_t gribcodes;

  cdoInitialize(process);

  // clang-format off
  const auto PRESSURE_FL = cdoOperatorAdd("pressure_fl", 0, 0, nullptr);
  const auto PRESSURE_HL = cdoOperatorAdd("pressure_hl", 0, 0, nullptr);
  const auto DELTAP      = cdoOperatorAdd("deltap",      0, 0, nullptr);
  // clang-format on

  const auto operatorID = cdoOperatorID();

  operatorCheckArgc(0);

  const auto streamID1 = cdoOpenRead(0);
  const auto vlistID1 = cdoStreamInqVlist(streamID1);

  const auto gridsize = vlist_check_gridsize(vlistID1);

  int nhlev;
  Varray<double> vct;
  vlist_read_vct(vlistID1, &zaxisIDh, &nvct, &nhlev, &nhlevf, &nhlevh, vct);

  bool l3Dvars = (zaxisIDh != -1 && gridsize > 0);
  if (!l3Dvars) cdoAbort("No 3D variable with hybrid sigma pressure coordinate found!");

  Varray<double> ps_prog(gridsize);
  Varray<double> deltap(gridsize * nhlevf), full_press(gridsize * nhlevf), half_press(gridsize * nhlevh);

  if (operatorID == PRESSURE_FL || operatorID == DELTAP)
    {
      if (Options::cdoVerbose) cdoPrint("Creating ZAXIS_HYBRID .. (nhlevf=%d)", nhlevf);
      zaxisIDp = zaxisCreate(ZAXIS_HYBRID, nhlevf);
    }
  else
    {
      if (Options::cdoVerbose) cdoPrint("Creating ZAXIS_HYBRID_HALF .. (nhlevh=%d)", nhlevh);
      zaxisIDp = zaxisCreate(ZAXIS_HYBRID_HALF, nhlevh);
    }

  Varray<double> level(nhlevh);
  for (int l = 0; l < nhlevh; l++) level[l] = l + 1;
  zaxisDefLevels(zaxisIDp, level.data());

  zaxisDefVct(zaxisIDp, 2 * nhlevh, vct.data());

  const auto nvars = vlistNvars(vlistID1);

  bool useTable = false;
  for (varID = 0; varID < nvars; varID++)
    {
      const auto tableNum = tableInqNum(vlistInqVarTable(vlistID1, varID));
      if (tableNum > 0 && tableNum != 255)
        {
          useTable = true;
          break;
        }
    }

  if (Options::cdoVerbose && useTable) cdoPrint("Use code tables!");

  for (varID = 0; varID < nvars; varID++)
    {
      const auto zaxisID = vlistInqVarZaxis(vlistID1, varID);
      const auto nlevels = zaxisInqSize(zaxisID);
      const auto instNum = institutInqCenter(vlistInqVarInstitut(vlistID1, varID));
      const auto tableNum = tableInqNum(vlistInqVarTable(vlistID1, varID));

      auto code = vlistInqVarCode(vlistID1, varID);
      auto param = vlistInqVarParam(vlistID1, varID);

      cdiParamToString(param, paramstr, sizeof(paramstr));

      if (useTable)
        {
          if (tableNum == 2)
            {
              mode = ModelMode::WMO;
              wmo_gribcodes(&gribcodes);
            }
          else if (tableNum == 128)
            {
              mode = ModelMode::ECHAM;
              echam_gribcodes(&gribcodes);
            }
          //  KNMI: HIRLAM model version 7.2 uses tableNum=1    (LAMH_D11*)
          //  KNMI: HARMONIE model version 36 uses tableNum=1   (grib*)
          //  (opreational NWP version) KNMI: HARMONIE model version 38 uses
          //  tableNum=253 (grib,grib_md) and tableNum=1 (grib_sfx) (research
          //  version)
          else if (tableNum == 1 || tableNum == 253)
            {
              mode = ModelMode::HIRLAM;
              hirlam_harmonie_gribcodes(&gribcodes);
            }
        }
      else
        {
          mode = ModelMode::ECHAM;
          echam_gribcodes(&gribcodes);
        }

      if (Options::cdoVerbose)
        {
          vlistInqVarName(vlistID1, varID, varname);
          cdoPrint("Mode = %d  Center = %d TableNum =%d Code = %d Param = %s Varname = %s varID = %d", (int) mode, instNum,
                   tableNum, code, paramstr, varname, varID);
        }

      if (code <= 0)
        {
          vlistInqVarName(vlistID1, varID, varname);

          cstrToLowerCase(varname);

          //                       ECHAM                         ECMWF
          if (cdo_cmpstr(varname, "geosp") || cdo_cmpstr(varname, "z"))
            code = 129;
          else if (cdo_cmpstr(varname, "st") || cdo_cmpstr(varname, "t"))
            code = 130;
          else if (cdo_cmpstr(varname, "aps") || cdo_cmpstr(varname, "sp"))
            code = 134;
          else if (cdo_cmpstr(varname, "ps"))
            code = 134;
          else if (cdo_cmpstr(varname, "lsp") || cdo_cmpstr(varname, "lnsp"))
            code = 152;
          /* else if (cdo_cmpstr(varname, "geopoth") == 0 ) code = 156; */
        }

      if (mode == ModelMode::ECHAM)
        {
          if (code == gribcodes.ps && nlevels == 1)
            psID = varID;
          else if (code == gribcodes.lsp && nlevels == 1)
            lnpsID = varID;
        }
      else if (mode == ModelMode::WMO || mode == ModelMode::HIRLAM)
        {
          if (code == gribcodes.ps && nlevels == 1) psID = varID;
        }
    }

  auto pvarID = lnpsID;
  if (zaxisIDh != -1 && lnpsID != -1)
    {
      const auto gridID = vlistInqVarGrid(vlistID1, lnpsID);
      if (gridInqType(gridID) == GRID_SPECTRAL)
        {
          lnpsID = -1;
          cdoWarning("Spectral LOG(%s) not supported - using %s!", var_stdname(surface_air_pressure),
                     var_stdname(surface_air_pressure));
        }
    }

  if (zaxisIDh != -1 && lnpsID == -1)
    {
      pvarID = psID;
      if (psID == -1) cdoAbort("%s not found!", var_stdname(surface_air_pressure));
    }

  const auto gridID = vlistInqVarGrid(vlistID1, pvarID);
  if (gridInqType(gridID) == GRID_SPECTRAL)
    cdoAbort("%s on spectral representation not supported!", var_stdname(surface_air_pressure));

  Varray<double> array(gridsize);

  const auto vlistID2 = vlistCreate();
  varID = vlistDefVar(vlistID2, gridID, zaxisIDp, TIME_VARYING);
  vlistDefVarParam(vlistID2, varID, cdiEncodeParam(1, 255, 255));
  cdiDefKeyString(vlistID2, varID, CDI_KEY_NAME, "pressure");
  cdiDefKeyString(vlistID2, varID, CDI_KEY_STDNAME, "air_pressure");
  cdiDefKeyString(vlistID2, varID, CDI_KEY_UNITS, "Pa");

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          if (varID == pvarID)
            {
              cdoReadRecord(streamID1, array.data(), &nmiss);
              if (nmiss) cdoAbort("Missing valus unsupported!");
            }
        }

      if (zaxisIDh != -1)
        {
          if (lnpsID != -1)
            for (size_t i = 0; i < gridsize; i++) ps_prog[i] = std::exp(array[i]);
          else if (psID != -1)
            varrayCopy(gridsize, array, ps_prog);

          // check range of ps_prog
          const auto mm = varrayMinMax(ps_prog);
          if (mm.min < MIN_PS || mm.max > MAX_PS)
            cdoWarning("Surface pressure out of range (min=%g max=%g)!", mm.min, mm.max);

          presh(full_press.data(), half_press.data(), vct.data(), ps_prog.data(), nhlevf, gridsize);
        }

      if (operatorID == PRESSURE_FL)
        {
          nlevel = nhlevf;
          pout = full_press.data();
        }
      else if (operatorID == DELTAP)
        {
          nlevel = nhlevf;
          for (k = 0; k < nhlevf; ++k)
            for (size_t i = 0; i < gridsize; ++i)
              deltap[k * gridsize + i] = half_press[(k + 1) * gridsize + i] - half_press[k * gridsize + i];

          pout = deltap.data();
        }
      else if (operatorID == PRESSURE_HL)
        {
          nlevel = nhlevh;
          pout = half_press.data();
        }

      varID = 0;
      for (levelID = 0; levelID < nlevel; levelID++)
        {
          cdoDefRecord(streamID2, varID, levelID);
          const auto offset = levelID * gridsize;
          cdoWriteRecord(streamID2, pout + offset, 0);
        }

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
