#
# Issue.py - issue definition
#
# Copyright (c) 2006-2007 The DITrack Project, www.ditrack.org.
#
# $Id: Issue.py 2605 2008-06-23 03:07:37Z vss $
# $HeadURL: https://svn.xiolabs.com/ditrack/src/tags/0.8/DITrack/DB/Issue.py $
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#  * Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
#  * Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following disclaimer in the documentation
# and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#

import email
import email.Message
import os.path
import re
import sys

# DITrack modules
import DITrack.Common
from DITrack.Logging import DEBUG, get_caller

# Comment file name regular expression.
comment_fname_re = re.compile("^comment(\\d+)$")

class AttachedFile:
    """
    Representation of a file attached to an issue.
    """

    def __init__(self, shortname, path, is_local):
        self.name = shortname
        self.path = path
        self.is_local = is_local

class _Attachments:
    """
    File attachments storage for an issue.
    
    Internal use only.
    """

    def __contains__(self, key):
        """
        Check whether an attachment named KEY exists.
        """
        return key in self._files

    def __getitem__(self, key):
        """
        Returns the attached file object (AttachedFile) by its name.
        """
        return self._files[key]

    def __init__(self):
        self._files = {}

        # A list of attachments (short names) removed from the storage since it
        # was created. It's needed to prevent removing an attachment and adding
        # another one with the same name within a single transaction.
        #
        # Only keys (which are short attachment names) matter.
        self._previously_removed = {}

    def __iter__(self):
        return self._files.iteritems()

    def add(self, fname, is_local, key=None):
        """
        Adds file FNAME (full path) to the list of attachments. KEY is the 
        "short" name used to refer to this attachment. If KEY is None, the base
        name of FNAME (e.g. its last component) is used as a key. The "local"
        flag for the attachment is set according to IS_LOCAL.

        May raise the same exceptions as add_object().
        """

        if key is None:
            key = os.path.basename(fname)

        self.add_object(AttachedFile(key, fname, is_local))

    def add_object(self, af):
        """
        Adds AttachedFile object AF to the storage. AF.name is used as a key.

        May raise the following exceptions:
            BadAttachmentNameError
                Can't use the attachment name since another attachment with
                similar name was removed in the same transaction.

            ValueError
                Another attachment with the same [base] name is already 
                present.

        """

        if af.name in self._files:
            raise ValueError, af.name

        if af.name in self._previously_removed:
            raise DITrack.DB.Exceptions.BadAttachmentNameError(af.name)

        self._files[af.name] = af

    def keys(self):
        """
        Returns a list of the attached file names in alphabetical order.
        """

        keys = self._files.keys()
        keys.sort()
        return keys

    def remove(self, key):
        """
        Removes a file referred to by KEY (short name) from the list of
        attachments. Assertion will fail if the KEY is not actually in the
        list.
        """

        assert key in self._files, key
        assert key not in self._previously_removed, key

        del self._files[key]
        self._previously_removed[key] = True

class Issue:
    """
    A representation of an issue record.
    """

    def __contains__(self, key):
        """
        Checks whether a comment indexed by the string KEY is present in the
        issue.
        """

        present = (key in self.local_names) or (key in self.firm_names)

        if present:
            assert(key in self.comment)

        return present

    def __getitem__(self, key):
        """
        Returns comment object indexed by KEY (which should be a string).
        """

        # XXX: Make sure the key is a string.

        if key not in self:
            raise KeyError

        return self.comment[key]

    def __init__(self):
        # XXX: all other data members should also be prefixed with '_', since
        # they are not publicly available.
        self._attachments = _Attachments()
        self.comment = {}
        self.firm_names = []
        self.local_names = []

    def __len__(self):
        """
        Returns total number of attached comments.
        """

        return len(self.firm_names) + len(self.local_names)

    def add_attachment(self, fname, is_local=True):
        """
        Store a reference to file FNAME as an attachment. This method merely
        adds the file _name_ to the list of attached files, without actually
        copying data or modifying the working copy.

        The attachment's "local" flag is set according to IS_LOCAL.

        XXX: raises the same exceptions as _Attachments.add()
        """

        self._attachments.add(fname, is_local)
        self.update_attachments_header()

    def add_comment(self, comment, is_local, update_info=True):
        """
        Adds comment COMMENT to the list of comments and names it according to
        IS_LOCAL. If UPDATE_INFO is true, updates the issue info afterwards.
        Returns simple comment id as a string.
        """

        DEBUG("Adding comment: is_local=%s, update_info=%s (called from %s)" %
            (is_local, update_info, get_caller()))

        # XXX
        assert(is_local)

        name = DITrack.DB.Common.next_entity_name(self.local_names)
        assert(name not in self.comment)

        self.comment[name] = comment
        self.local_names.append(name)

        if update_info:
            self.update_info()

        return name

    def attachments(self):
        """
        Returns a list of AttachedFile objects for all files attached to the
        issue in alphabetical order of their names.
        """

        return map(lambda x: self._attachments[x], self._attachments.keys())

    def change_due_version(self, version):
        """
        Changes due version of the issue to VERSION. No sanity checks are made.
        """
        self.replace_header("Due-in", version)

    def close(self, resolution):
        """
        Closes the issue with specified RESOLUTION (which should be one of
        "dropped", "fixed" or "invalid"). Assertion will fail if the issue is
        already closed or there is already a "Resolution" header present.

        XXX: we should probably raise an exception, not fire an assert
        XXX: resolution enumeration should be defined elsewhere (in a single
        place for all modules that use it).
        """

        if self.info["Status"] != "open":
            raise DITrack.DB.Exceptions.InconsistentActionError(
                    "The issue is not open"
                )

        self.replace_header("Status", "closed")

        assert "Resolution" not in self.info
        self.info["Resolution"] = resolution


    def comments(self, firm=True, local=True):
        """
        Returns a list of tuples (ID, COMMENT) for all existing comments. IDs
        are strings. First all firm comments go in order, then the local ones.

        FIRM and LOCAL prescribe which comments to include into the result.
        """

        assert firm or local

        if firm:
            # We assume the list is sorted.
            names = self.firm_names
        else:
            names = []

        result = [(id, self.comment[id]) for id in names]

        if local:
            names = self.local_names
            names.sort()

            result.extend([(id, self.comment[id]) for id in names])

        return result

    def create(cls, title, opened_by, opened_on, owned_by, category,
        version_reported, version_due, description):
        """
        Create new issue instance from scratch.
        """

        issue = cls()

        # Create an initial comment.
        comment = Comment.create(
            text=description,
            added_by=opened_by,
            added_on=opened_on,
            delta=[
                ("Opened-by", None, opened_by),
                ("Opened-on", None, opened_on),
                ("Owned-by", None, owned_by),
                ("Title", None, title),
                ("Category", None, category),
                ("Status", None, "open"),
                ("Reported-for", None, version_reported),
                ("Due-in", None, version_due)
            ])

        # Now append the comment to the issue.
        issue.add_comment(comment, is_local=True)

        return issue

    create = classmethod(create)

    def diff(self, other):
        """
        Returns a difference in headers between SELF and the OTHER issue, by
        checking the headers. The changed headers are returned as a list of
        tuples (HEADER, OLD, NEW), where the HEADER is the header name and the
        OLD and NEW are the old and new values respectively (any of these may
        be None, but not both).

        XXX: should be renamed to diff_headers
        """

        delta = []

        for h in self.info.keys():

            if h in other.info:
                if self.info[h] != other.info[h]:
                    delta.append((h, self.info[h], other.info[h]))
            else:
                    delta.append((h, self.info[h], None))

        for h in other.info.keys():

            if h not in self.info:
                delta.append((h, None, other.info[h]))

        return delta

    def diff_attachments(self, another):
        """
        Figure out the difference in attachments between SELF and ANOTHER
        issue.

        If there is no difference, None is returned. Otherwise returns a 
        dictionary with keys "added" and "removed", where corresponding values
        are the lists of added and removed files. The items of the lists are
        instances of AttachedFile objects.
        """

        added = []
        removed = []

        # Note: the contents of attached file can't change. It should be done
        # via delete/add operation.

        for name, attach in self._attachments:
            if name not in another._attachments:
                removed.append(attach)

        for name, attach in another._attachments:
            if name not in self._attachments:
                added.append(attach)

        if (not added) and (not removed):
            return None

        return { "added": added, "removed": removed }

    def get_attachment(self, fname):
        """
        Returns AttachedFile object of an attachment named FNAME. Raises 
        KeyError if there is no such attachment.
        """

        return self._attachments[fname]

    def info_as_strings(self, terminator="\n"):
        """
        Returns a list of strings, representing the issue info, one header per
        line. The list is sorted by header names. The lines are terminated
        with the specified TERMINATOR.
        """

        keys = self.info.keys()
        keys.sort()

        return map(lambda k: "%s: %s%s" % (k, self.info[k], terminator), keys)

    def is_up_to_date(self, path):
        """
        Check if the issue is up to date with respect to its on-disk
        representation at PATH (should point to the issue directory).
        """

        n = int(self.firm_names[-1])
        fname = os.path.join(path, "comment%d" % (n + 1))

        return not os.path.exists(fname)

    def load(cls, path):
        """
        Load an issue from path PATH (should point to a directory).
        """

        DEBUG("Loading an issue from '%s' (called from %s)" %
            (path, get_caller()))

        issue = cls()

        comments = {}
        keys = []

        for fn in os.listdir(path):
            m = comment_fname_re.match(fn)
            if m:
                n = m.group(1)
                fname = os.path.join(path, "comment" + n)
                comments[int(n)] = Comment.load(fname)
                keys.append(int(n))

        keys.sort()

        issue.firm_names = []

        # k is numeric here.
        for k in keys:
            key = "%d" % k
            issue.comment[key] = comments[k]
            issue.firm_names.append(key)

        issue.update_info()

        # Take care of the attachments now.
        # XXX: should be a constant
        if "Attachments" in issue.info:
            # XXX: there should be a function to parse this out
            attachments = issue.info["Attachments"].strip().split()

            for a in attachments:
                # Check that the attachment is really there
                a_path = os.path.join(path, "attaches", a)

                # XXX: this should be an exception
                assert os.path.exists(a_path)

                issue.add_attachment(a_path, is_local=False)

        return issue

    load = classmethod(load)

    def merge_local_comments(self, local):
        """
        Merges all local comments from the LOCAL issue to SELF, retaining
        the names as is. Names collision will raise an assertion fault.

        NB! Does NOT update the info, use the update_info() method for that.
        """

        for name in local.local_names:

            DEBUG("Merging local comment '%s' (called from %s)" %
                (name, get_caller())
            )
            assert name not in self.local_names, name

            self.comment[name] = local.comment[name]

        self.local_names.extend(local.local_names)

        self.local_names.sort()

    def reassign(self, new_owner):
        """
        Reassigns the issue to NEW_OWNER. No checks are made on the validity of
        the user name passed.
        """
        self.replace_header("Owned-by", new_owner)

    def remove_attachment(self, fname):
        """
        Removed FNAME from the list of attached files (no on-disk operations
        are performed) and updates the corresponding header with 
        update_attachments_header(). 

        FNAME is a short name.
        """

        self._attachments.remove(fname)
        self.update_attachments_header()

    def remove_comment(self, name):
        """
        Removes a local comment named NAME from the issue. The comment should
        exist, otherwise KeyError is raised. The info is *not* updated.
        """

        if name not in self.local_names:
            raise KeyError, name

        assert name in self.comment, "name='%s'" % name

        del self.comment[name]

        self.local_names.remove(name)

    def reopen(self):
        """
        Reopens the issue (which must be closed, otherwise the assertion will
        fail). An assertion will also fail if there is no "Resolution" header.
        """

        if self.info["Status"] != "closed":
            raise DITrack.DB.Exceptions.InconsistentActionError, \
                "The issue is not closed"

        # i#95: work around absent resolution header.
        try:
            del self.info["Resolution"]
        except KeyError:
            pass

        self.replace_header("Status", "open")

    def replace_header(self, name, value):
        """
        Replaces the issue header NAME with VALUE. The header should exist.
        """
        del self.info[name]
        self.info[name] = value

    def update_attachments_header(self):
        """
        Updates the 'Attachments' info header with the current status of
        attached files.

        XXX: should be internal
        """

        self.info["Attachments"] = " ".join(self._attachments.keys())

    def update_info(self):
        """
        Goes through all the comments and updates the info and attachments
        list accordingly (applies deltas in sequence). May be used repeatedly.
        """

        self.info = {}

        names = self.local_names
        names.sort()

        names = self.firm_names + names

        DEBUG("Updating the info (called from %s)" % get_caller())

        for name in names:
            DEBUG("Processing comment '%s'" % name)
            self.comment[name].apply_delta(self.info, self._attachments)

        DEBUG("Updated info: %s" % self.info)

    def write_info(self):
        """
        Prints out current issue headers.
        """

        sys.stdout.writelines(self.info_as_strings())

class Comment:
    """
    A representation of a comment record.
    """

    def apply_delta(self, info, attachments):
        """
        Applies own delta to the info dictionary INFO.
        
        ATTACHMENTS is an object representing attached files. It should provide
        methods add() and remove() to apply attachment deltas.

        XXX: should deal with conflicts
        """

        # XXX: should check for duplicate headers in delta

        DEBUG("Applying the delta")

        for header, old, new in self.delta:

            DEBUG("Delta: header='%s' old='%s' new='%s'" % (header, old, new))

            # XXX
            if old:

                assert header in info, "header = '%s'" % header

                assert(info[header] == old)

                if new:
                    info[header] = new
                else:
                    del info[header]

            else:
                assert(new is not None)
                assert(header not in info)

                info[header] = new

        # Process attachments
        if self.attachments_changed():

            added = self.attachments_added()
            removed = self.attachments_removed()

            DEBUG("Processing attachment delta (%d added files)" % len(added))
            for a in added:
                DEBUG("attachment name: %s" % a.name)
                assert a.name not in attachments, \
                    "%s shouldn't be in %s" % (a.name, attachments.keys())
                attachments.add_object(a)

            DEBUG("Processing attachment delta (%d removed files)"
                % len(removed))
            for a in removed:
                DEBUG("attachment name: %s" % a.name)
                assert a.name in attachments, \
                    "%s should be in %s" % (a.name, attachments.keys())
                attachments.remove(a.name)

            # XXX: should make use of update_attachments_header
            info["Attachments"] = " ".join(attachments.keys())

        else:
            DEBUG("Attachment delta is empty")

    def attachments_added(self):
        """
        Returns a list of added attachments for this comment or None if there 
        is none. The returned list is a list of AttachedFile objects.
        """
        if self.attachments_changed():
            return self._attachment_delta["added"]

    def attachments_changed(self):
        """
        Returns True if this comment makes any changes to the attached files.

        Assertions will fail if the data is inconsistent in any way.
        """
        if self._attachment_delta is None:
            return False

        assert "added" in self._attachment_delta
        assert "removed" in self._attachment_delta

        return self._attachment_delta["added"] or \
            self._attachment_delta["removed"]

    def attachments_removed(self):
        """
        Returns a list of removed attachments for this comment or None if there 
        is none. The returned list is a list of AttachedFile objects.
        """
        if self.attachments_changed():
            return self._attachment_delta["removed"]

    def create(cls, text, added_on, added_by, delta, logmsg="",
        attachment_delta=None):

        comment = cls()

        comment.text = text
        comment.added_on = added_on
        comment.added_by = added_by
        comment.delta = delta
        comment.logmsg = logmsg
        comment._attachment_delta = attachment_delta

        return comment

    create = classmethod(create)

    def headers(self):
        """
        Constructs the comment header and returns it as a list of tuples 
        (NAME, VALUE) sorted alphabetically by the header name.
        """

        output = []

        output.append(("Added-on", "%s" % self.added_on))
        output.append(("Added-by", "%s" % self.added_by))

        for header, old, new in self.delta:

            assert((old is not None) or (new is not None))

            old = old or ""
            new = new or ""

            output.append(("DT-Old-%s" % header, old))
            output.append(("DT-New-%s" % header, new))

        output.sort(lambda (a_k, a_v), (b_k, b_v): cmp(a_k, b_k))

        return output

    def header_as_strings(self, terminator="\n"):
        """
        Constructs the comment header and returns it as a list of strings,
        sorted alphabetically by the header name and terminated with
        TERMINATOR.
        """

        return ["%s: %s%s" % (k, v, terminator) for (k, v) in self.headers()]

    def load(cls, path):
        """
        Load a comment from path PATH.
        """

        f = open(path)
        data = email.message_from_file(f)
        f.close()

        # XXX: we should also handle user headers here (somehow).

        # To ignore case difference, put all headers in our own map.
        header = {}
        orig = {}
        for h in data.keys():
            header[h.lower()] = data[h]
            orig[h.lower()] = h

        delta = []

        for h in header.keys():
            h = h.lower()

            if h.startswith("dt-old-"):

                name = h[7:]
                nh = "dt-new-" + name
                assert(nh in header)

                if not (data[h] or data[nh]):
                    continue

                delta.append((orig[h][7:], data[h], data[nh]))

        return Comment.create(
            text=data.get_payload(),
            added_on=data["Added-on"],
            added_by=data["Added-by"],
            delta=delta
            )

    load = classmethod(load)

    def write(self, dest=sys.stdout, headers_only=False, display_headers=True):
        """
        Write out the comment into file object DEST.
        """

        if display_headers:
            dest.writelines(self.header_as_strings())

        if not headers_only:

            if display_headers:
                dest.write("\n")

            dest.write("%s" % self.text)
