/*
 * Licensed to Elasticsearch under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.elasticsearch.gateway.local;

import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.cluster.routing.*;
import org.elasticsearch.common.settings.ImmutableSettings;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.test.ElasticsearchTestCase;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

public class PriorityComparatorTests extends ElasticsearchTestCase {

    public void testPriorityComparatorSort() {
        RoutingNodes.UnassignedShards shards = new RoutingNodes.UnassignedShards(null);
        int numIndices = randomIntBetween(3, 99);
        IndexMeta[] indices = new IndexMeta[numIndices];
        final Map<String, IndexMeta> map = new HashMap<>();

        for (int i = 0; i < indices.length; i++) {
            if (frequently()) {
                indices[i] = new IndexMeta("idx_2015_04_" + String.format(Locale.ROOT, "%02d", i), randomIntBetween(1, 1000), randomIntBetween(1, 10000));
            } else { // sometimes just use defaults
                indices[i] = new IndexMeta("idx_2015_04_" +  String.format(Locale.ROOT, "%02d", i));
            }
            map.put(indices[i].name, indices[i]);
        }
        int numShards = randomIntBetween(10, 100);
        for (int i = 0; i < numShards; i++) {
            IndexMeta indexMeta = randomFrom(indices);
            shards.add(new MutableShardRouting(new ImmutableShardRouting(indexMeta.name, randomIntBetween(1, 5), null, null, null,
                    randomBoolean(), ShardRoutingState.UNASSIGNED, randomIntBetween(0, 100), new UnassignedInfo(randomFrom(UnassignedInfo.Reason.values()), "foobar"))));
        }
        shards.sort(new PriorityComparator() {
            @Override
            protected Settings getIndexSettings(String index) {
                IndexMeta indexMeta = map.get(index);
                return indexMeta.settings;
            }
        });
        ShardRouting previous = null;
        for (ShardRouting routing : shards) {
            if (previous != null) {
                IndexMeta prevMeta = map.get(previous.getIndex());
                IndexMeta currentMeta = map.get(routing.getIndex());
                if (prevMeta.priority == currentMeta.priority) {
                    if (prevMeta.creationDate == currentMeta.creationDate) {
                        if (prevMeta.name.equals(currentMeta.name) == false) {
                            assertTrue("indexName mismatch, expected:" + currentMeta.name + " after " + prevMeta.name + " " + prevMeta.name.compareTo(currentMeta.name), prevMeta.name.compareTo(currentMeta.name) > 0);
                        }
                    } else {
                        assertTrue("creationDate mismatch, expected:" + currentMeta.creationDate + " after " + prevMeta.creationDate, prevMeta.creationDate > currentMeta.creationDate);
                    }
                } else {
                    assertTrue("priority mismatch, expected:" +  currentMeta.priority + " after " + prevMeta.priority, prevMeta.priority > currentMeta.priority);
                }
            }
            previous = routing;
        }
    }

    private static class IndexMeta {
        final String name;
        final int priority;
        final long creationDate;
        final Settings settings;

        private IndexMeta(String name) { // default
            this.name = name;
            this.priority = 1;
            this.creationDate = -1;
            this.settings = ImmutableSettings.EMPTY;
        }

        private IndexMeta(String name, int priority, long creationDate) {
            this.name = name;
            this.priority = priority;
            this.creationDate = creationDate;
            this.settings = ImmutableSettings.builder().put(IndexMetaData.SETTING_CREATION_DATE, creationDate)
                    .put(IndexMetaData.SETTING_PRIORITY, priority).build();
        }
    }
}