/*
 * Copyright (C) 2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdbool.h>
#include <stdio.h>

#include "glue.h"

#include "chip_gen_74595.h"

#define CHIP_(x)	chip_gen_74595_ ## x

struct cpssp {
	/* Out */
	struct sig_std_logic *port_q7s;
	struct sig_std_logic *port_q[8];

	/* In */
	bool state_gnd;
	bool state_vcc;
	bool state_shcp;
	bool state_stcp;
	bool state_ds;
	bool state_mr;
	bool state_oe;

	/* State */
	uint8_t state_ff;
	uint8_t state_latch;
};

static void
CHIP_(update_q)(struct cpssp *cpssp)
{
	unsigned int i;

	for (i = 0; i < 8; i++) {
		if (! cpssp->state_oe) {
			sig_std_logic_set(cpssp->port_q[i], cpssp,
					SIG_STD_LOGIC_Z);
		} else if ((cpssp->state_latch >> i) & 1) {
			sig_std_logic_set(cpssp->port_q[i], cpssp,
					SIG_STD_LOGIC_1);
		} else {
			sig_std_logic_set(cpssp->port_q[i], cpssp,
					SIG_STD_LOGIC_0);
		}
	}
}

static void
CHIP_(update_q7s)(struct cpssp *cpssp)
{
	sig_std_logic_set(cpssp->port_q7s, cpssp,
			(cpssp->state_ff >> 7) & 1
			?  SIG_STD_LOGIC_1
			: SIG_STD_LOGIC_0);
}

static void
CHIP_(ds_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_ds = val;
}

static void
CHIP_(stcp_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	if (! cpssp->state_stcp
	 && val) {
		cpssp->state_latch = cpssp->state_ff;

		CHIP_(update_q)(cpssp);
	}
	cpssp->state_stcp = val;
}

static void
CHIP_(shcp_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	if (! cpssp->state_stcp
	 && val) {
		cpssp->state_ff = (cpssp->state_ff << 1) | cpssp->state_ds;

		CHIP_(update_q7s)(cpssp);
	}
	cpssp->state_stcp = val;
}

static void
CHIP_(n_oe_set)(void *_cpssp, unsigned int n_val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_oe = ! n_val;

	CHIP_(update_q)(cpssp);
}

static void
CHIP_(n_mr_set)(void *_cpssp, unsigned int n_val)
{
	struct cpssp *cpssp = _cpssp;

	if (! n_val) {
		cpssp->state_ff = 0x00;
		cpssp->state_latch = 0x00;

		CHIP_(update_q)(cpssp);
		CHIP_(update_q7s)(cpssp);
	}
	cpssp->state_mr = ! n_val;
}

static void
CHIP_(gnd_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_gnd = val;
}

static void
CHIP_(vcc_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_vcc = val;
}

void *
CHIP_(create)(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_std_logic *port_gnd,
	struct sig_std_logic *port_vcc,
	struct sig_std_logic *port_n_mr,
	struct sig_std_logic *port_n_oe,
	struct sig_std_logic *port_shcp,
	struct sig_std_logic *port_stcp,
	struct sig_std_logic *port_ds,
	struct sig_std_logic *port_q7s,
	struct sig_std_logic *port_q0,
	struct sig_std_logic *port_q1,
	struct sig_std_logic *port_q2,
	struct sig_std_logic *port_q3,
	struct sig_std_logic *port_q4,
	struct sig_std_logic *port_q5,
	struct sig_std_logic *port_q6,
	struct sig_std_logic *port_q7
)
{
	static const struct sig_std_logic_funcs gnd_funcs = {
		.boolean_or_set = CHIP_(gnd_set),
	};
	static const struct sig_std_logic_funcs vcc_funcs = {
		.boolean_or_set = CHIP_(vcc_set),
	};
	static const struct sig_std_logic_funcs n_mr_funcs = {
		.boolean_or_set = CHIP_(n_mr_set),
	};
	static const struct sig_std_logic_funcs n_oe_funcs = {
		.boolean_or_set = CHIP_(n_oe_set),
	};
	static const struct sig_std_logic_funcs shcp_funcs = {
		.boolean_or_set = CHIP_(shcp_set),
	};
	static const struct sig_std_logic_funcs stcp_funcs = {
		.boolean_or_set = CHIP_(stcp_set),
	};
	static const struct sig_std_logic_funcs ds_funcs = {
		.boolean_or_set = CHIP_(ds_set),
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	cpssp->state_ff = 0;
	cpssp->state_latch = 0;

	/* Out */
	cpssp->port_q7s = port_q7s;
	sig_std_logic_connect_out(port_q7s, cpssp, SIG_STD_LOGIC_0);
	cpssp->port_q[0] = port_q0;
	sig_std_logic_connect_out(port_q0, cpssp, SIG_STD_LOGIC_0);
	cpssp->port_q[1] = port_q1;
	sig_std_logic_connect_out(port_q1, cpssp, SIG_STD_LOGIC_0);
	cpssp->port_q[2] = port_q2;
	sig_std_logic_connect_out(port_q2, cpssp, SIG_STD_LOGIC_0);
	cpssp->port_q[3] = port_q3;
	sig_std_logic_connect_out(port_q3, cpssp, SIG_STD_LOGIC_0);
	cpssp->port_q[4] = port_q4;
	sig_std_logic_connect_out(port_q4, cpssp, SIG_STD_LOGIC_0);
	cpssp->port_q[5] = port_q5;
	sig_std_logic_connect_out(port_q5, cpssp, SIG_STD_LOGIC_0);
	cpssp->port_q[6] = port_q6;
	sig_std_logic_connect_out(port_q6, cpssp, SIG_STD_LOGIC_0);
	cpssp->port_q[7] = port_q7;
	sig_std_logic_connect_out(port_q7, cpssp, SIG_STD_LOGIC_0);

	/* In */
	cpssp->state_gnd = 0;
	sig_std_logic_connect_in(port_gnd, cpssp, &gnd_funcs);
	cpssp->state_vcc = 0;
	sig_std_logic_connect_in(port_vcc, cpssp, &vcc_funcs);
	cpssp->state_mr = 1;
	sig_std_logic_connect_in(port_n_mr, cpssp, &n_mr_funcs);
	cpssp->state_oe = 1;
	sig_std_logic_connect_in(port_n_oe, cpssp, &n_oe_funcs);
	cpssp->state_shcp = 0;
	sig_std_logic_connect_in(port_shcp, cpssp, &shcp_funcs);
	cpssp->state_stcp = 0;
	sig_std_logic_connect_in(port_stcp, cpssp, &stcp_funcs);
	cpssp->state_ds = 0;
	sig_std_logic_connect_in(port_ds, cpssp, &ds_funcs);

	CHIP_(update_q)(cpssp);
	CHIP_(update_q7s)(cpssp);

	return cpssp;
}

void
CHIP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
CHIP_(suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
CHIP_(resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fp);
}
