/*
 * Architecture for Philips pcx8582X-2 eeprom
 * 256 * 8 Bit CMOS EEPROM with i2c-bus interface
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifdef STATE

struct {
	/** data of the chip */
	unsigned char data[256];
	/** device code:
	 *  first four bits hardwired to 1010
	 *  then a2, a1, a0 (last bit read/write) */
	unsigned char device_code;
	/** offset counter in data register, auto_increment when reading */
	unsigned char counter;
	/** state of i2c bus. true means the next byte read written is the 
	 *  word address. */
	bool is_address_phase;

	/* TODO potyra porst not yet handled:
	 *      - V(DD) power 5v DC, V(SS) (negative supply voltage?)
	 *      - A0, A1, A2 (these should be input pins)
	 *      - PTC out (programming time control output)
	 */
} NAME;

#endif /* STATE */

#ifdef BEHAVIOR

#include <string.h>

static inline void
NAME_(stop_transaction)(struct cpssp *cpssp)
{
	cpssp->NAME.is_address_phase = true;
}


static inline void
NAME_(read_byte)(struct cpssp *cpssp, unsigned char *val)
{
	if (cpssp->NAME.is_address_phase) {
		*val = cpssp->NAME.counter;
		cpssp->NAME.is_address_phase = false;
		return;
	}

	*val = cpssp->NAME.data[cpssp->NAME.counter];
	cpssp->NAME.counter++;
}

static inline bool
NAME_(write_byte)(struct cpssp *cpssp, unsigned char val)
{
	if (cpssp->NAME.is_address_phase) {
		cpssp->NAME.counter = val;
		cpssp->NAME.is_address_phase = false;
		return true;
	}

	cpssp->NAME.data[cpssp->NAME.counter] = val;

	/* TODO page write */
	return true;
}

static inline bool
NAME_(ack_addr)(struct cpssp *cpssp, unsigned char addr)
{
	if ((cpssp->NAME.device_code & 0xFE) == (addr & 0xFE)) {
		return true;
	}
	
	return false;
}

static inline void
NAME_(a0_set)(struct cpssp *cpssp, unsigned int val)
{
	cpssp->NAME.device_code &= ~(1 << (0 + 1));
	cpssp->NAME.device_code |= val << (0 + 1);
}

static inline void
NAME_(a1_set)(struct cpssp *cpssp, unsigned int val)
{
	cpssp->NAME.device_code &= ~(1 << (1 + 1));
	cpssp->NAME.device_code |= val << (1 + 1);
}

static inline void
NAME_(a2_set)(struct cpssp *cpssp, unsigned int val)
{
	cpssp->NAME.device_code &= ~(1 << (2 + 1));
	cpssp->NAME.device_code |= val << (2 + 1);
}


/** initialize the eeprom
 *  @param cpssp component's configuration.
 *  @param initial_contents optional byte array containing the initial 
 *         contents. passing a NULL pointer will cause the eeprom to
 *         contain only 0s in the beginning.
 *  @param device_code contains the lower three bits of the device address,
 *         which will get define A2, A1 and A0 respectively.
 */
static void
NAME_(init)(
	struct cpssp *cpssp, 
	const char initial_contents[256]
)
{
	cpssp->NAME.device_code = 0xa0;
	cpssp->NAME.counter = 0;
	cpssp->NAME.is_address_phase = true;

	if (initial_contents != NULL) {
		memcpy(cpssp->NAME. data, initial_contents, 
			sizeof(cpssp->NAME.data));
	} else {
		memset(cpssp->NAME.data, 0, sizeof(cpssp->NAME.data));
	}
}

#endif /* BEHAVIOR */
