/*
 * Copyright (C) 2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifdef INCLUDE

#include "softfloat.h"

#endif /* INCLUDE */
#ifdef STATE

struct {
	uint32_t fifo[64];
	int count;
	int in;
	int out;

	/* Opcode Register */
	uint16_t opcode;

	/* Control Register */
	uint8_t x;
	uint8_t rc;	/* Rounding Control */
	uint8_t pc;	/* Precision Control */
	uint8_t pm;	/* Mask: Precision */
	uint8_t um;	/* Mask: Underrun */
	uint8_t om;	/* Mask: Overrun */
	uint8_t zm;	/* Mask: Division by Zero */
	uint8_t dm;	/* Mask: Denormalized */
	uint8_t im;	/* Mask: Invalid Operation */

	/* Status Register */
	uint8_t c0;
	uint8_t c1;
	uint8_t c2;
	uint8_t c3;
	/* uint8_t es; Bit derived from other bits. */
	uint8_t sf;
	uint8_t pe;	/* Error: Precision */
	uint8_t ue;	/* Error: Underrun */
	uint8_t oe;	/* Error: Overrun */
	uint8_t ze;	/* Error: Division by Zero */
	uint8_t de;	/* Error: Denormalized */
	uint8_t ie;	/* Error: Invalid Operation */
	uint8_t top;

	/* Tag Register */
	uint8_t used[8];

	/* Data Register */
	float80 st[8];
} NAME;

#endif /* STATE */
#ifdef EXPORT

/*forward*/ static void                                       
NAME_(n_error_set)(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
NAME_(write)(struct cpssp *cpssp, uint32_t addr, uint32_t val);
/*forward*/ static void
NAME_(read)(struct cpssp *cpssp, uint32_t addr, uint32_t *valp);
/*forward*/ static void
NAME_(n_reset_set)(struct cpssp *cpssp, unsigned int n_val);
/*forward*/ static void
NAME_(create)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(destroy)(struct cpssp *cpssp);

#endif /* EXPORT */
#ifdef BEHAVIOR

#define DEBUG_CONTROL_FLOW	0

static int
NAME_(error_update)(struct cpssp *cpssp)
{
	uint8_t es;

	es = 0;
	es |= cpssp->NAME.pe & ~cpssp->NAME.pm;
	es |= cpssp->NAME.ue & ~cpssp->NAME.um;
	es |= cpssp->NAME.oe & ~cpssp->NAME.om;
	es |= cpssp->NAME.ze & ~cpssp->NAME.zm;
	es |= cpssp->NAME.de & ~cpssp->NAME.dm;
	es |= cpssp->NAME.ie & ~cpssp->NAME.im;

	NAME_(n_error_set)(cpssp, ! es);

	return es;
}

#if 0
static void
NAME_(P)(struct cpssp *cpssp)
{
	cpssp->NAME.pe = 1;
	NAME_(error_update)(cpssp);
}

static void
NAME_(U)(struct cpssp *cpssp)
{
	cpssp->NAME.ue = 1;
	NAME_(error_update)(cpssp);
}

static void
NAME_(O)(struct cpssp *cpssp)
{
	cpssp->NAME.oe = 1;
	NAME_(error_update)(cpssp);
}

static void
NAME_(Z)(struct cpssp *cpssp)
{
	cpssp->NAME.ze = 1;
	NAME_(error_update)(cpssp);
}

static void
NAME_(D)(struct cpssp *cpssp)
{
	cpssp->NAME.de = 1;
	NAME_(error_update)(cpssp);
}
#endif

static void
NAME_(IA)(struct cpssp *cpssp)
{
	cpssp->NAME.ie = 1;
	NAME_(error_update)(cpssp);
}

#if 0
static void
NAME_(IS)(struct cpssp *cpssp)
{
	cpssp->NAME.ie = 1;
	NAME_(error_update)(cpssp);
}
#endif

static uint16_t
NAME_(load_control)(struct cpssp *cpssp)
{
	uint16_t word;

	word = 0;
	word |= 0 << 15;
	word |= 0 << 14;
	word |= 0 << 13;
	word |= cpssp->NAME.x << 12;
	word |= cpssp->NAME.rc << 10;
	word |= cpssp->NAME.pc << 8;
	word |= 0 << 7;
	word |= 1 << 6;
	word |= cpssp->NAME.pm << 5;
	word |= cpssp->NAME.um << 4;
	word |= cpssp->NAME.om << 3;
	word |= cpssp->NAME.zm << 2;
	word |= cpssp->NAME.dm << 1;
	word |= cpssp->NAME.im << 0;

	return word;
}

static void
NAME_(store_control)(struct cpssp *cpssp, uint16_t word)
{
	/* Bit 15: Hard-wired 0 */
	/* Bit 14: Hard-wired 0 */
	/* Bit 13: Hard-wired 0 */
	cpssp->NAME.x = (word >> 12) & 0x1;
	cpssp->NAME.rc = (word >> 10) & 0x3;
	cpssp->NAME.pc = (word >> 8) & 0x3;
	/* Bit 7: Hard-wired 0 */
	/* Bit 6: Hard-wired 1 */
	cpssp->NAME.pm = (word >> 5) & 0x1;
	cpssp->NAME.um = (word >> 4) & 0x1;
	cpssp->NAME.om = (word >> 3) & 0x1;
	cpssp->NAME.zm = (word >> 2) & 0x1;
	cpssp->NAME.dm = (word >> 1) & 0x1;
	cpssp->NAME.im = (word >> 0) & 0x1;

	NAME_(error_update)(cpssp);
}

static uint16_t
NAME_(load_status)(struct cpssp *cpssp)
{
	uint16_t word;
	uint8_t es;

	es = 0;
	es |= cpssp->NAME.pe & ~cpssp->NAME.pm;
	es |= cpssp->NAME.ue & ~cpssp->NAME.um;
	es |= cpssp->NAME.oe & ~cpssp->NAME.om;
	es |= cpssp->NAME.ze & ~cpssp->NAME.zm;
	es |= cpssp->NAME.de & ~cpssp->NAME.dm;
	es |= cpssp->NAME.ie & ~cpssp->NAME.im;

	word = 0;
	word |= es << 15; /* Busy */
	word |= cpssp->NAME.c3 << 14;
	word |= cpssp->NAME.top << 11;
	word |= cpssp->NAME.c2 << 10;
	word |= cpssp->NAME.c1 << 9;
	word |= cpssp->NAME.c0 << 8;
	word |= es << 7;
	word |= cpssp->NAME.sf << 6;
	word |= cpssp->NAME.pe << 5;
	word |= cpssp->NAME.ue << 4;
	word |= cpssp->NAME.oe << 3;
	word |= cpssp->NAME.ze << 2;
	word |= cpssp->NAME.de << 1;
	word |= cpssp->NAME.ie << 0;

	return word;
}

static void
NAME_(store_status)(struct cpssp *cpssp, uint16_t word)
{
	/* busy derived from other bits. */
	cpssp->NAME.c3 = (word >> 14) & 0x1;
	cpssp->NAME.top = (word >> 11) & 0x7;
	cpssp->NAME.c2 = (word >> 10) & 0x1;
	cpssp->NAME.c1 = (word >> 9) & 0x1;
	cpssp->NAME.c0 = (word >> 8) & 0x1;
	/* es derived from other bits. */
	cpssp->NAME.sf = (word >> 6) & 0x1;
	cpssp->NAME.pe = (word >> 5) & 0x1;
	cpssp->NAME.ue = (word >> 4) & 0x1;
	cpssp->NAME.oe = (word >> 3) & 0x1;
	cpssp->NAME.ze = (word >> 2) & 0x1;
	cpssp->NAME.de = (word >> 1) & 0x1;
	cpssp->NAME.ie = (word >> 0) & 0x1;

	NAME_(error_update)(cpssp);
}

static uint16_t
NAME_(load_tag)(struct cpssp *cpssp)
{
	uint16_t word;
	int i;

	word = 0;
	for (i = 7; 0 <= i; i--) {
		word <<= 2;

		if (! cpssp->NAME.used[i]) {
			word |= 0x3;
		} else if (float80_iszero(cpssp->NAME.st[i])) {
			word |= 0x1;
		} else if (! float80_isnorm(cpssp->NAME.st[i])) {
			word |= 0x2;
		} else {
			word |= 0x0;
		}
	}

	return word;
}

static void
NAME_(store_tag)(struct cpssp *cpssp, uint16_t word)
{
	int i;

	for (i = 0; i < 8; i++) {
		cpssp->NAME.used[i] = (word & 0x3) != 0x3;
		word >>= 2;
	}
}

static void
NAME_(store_data)(struct cpssp *cpssp, int i, float80 val)
{
	cpssp->NAME.st[(cpssp->NAME.top + i) % 8] = val;
}

static float80
NAME_(load_data)(struct cpssp *cpssp, int i)
{
	return cpssp->NAME.st[(cpssp->NAME.top + i) % 8];
}

static uint16_t
NAME_(load_mem_m16bit)(struct cpssp *cpssp)
{
	uint16_t val;

	val = (uint16_t ) cpssp->NAME.fifo[cpssp->NAME.count];

	cpssp->NAME.count++;

	return val;
}

static void
NAME_(store_mem_m16bit)(struct cpssp *cpssp, uint16_t val)
{
	cpssp->NAME.fifo[cpssp->NAME.count] = (0xffff << 16) | val;

	cpssp->NAME.count++;
}

static int16_t
NAME_(load_mem_m16int)(struct cpssp *cpssp)
{
	int16_t val;

	val = *(int16_t *) &cpssp->NAME.fifo[cpssp->NAME.count];

	cpssp->NAME.count++;

	return val;
}

static void
NAME_(store_mem_m16int)(struct cpssp *cpssp, int16_t val)
{
	*(int16_t *) &cpssp->NAME.fifo[cpssp->NAME.count] = val;

	cpssp->NAME.count++;
}

static int32_t
NAME_(load_mem_m32int)(struct cpssp *cpssp)
{
	int32_t val;

	val = *(int32_t *) &cpssp->NAME.fifo[cpssp->NAME.count];

	cpssp->NAME.count++;

	return val;
}

static void
NAME_(store_mem_m32int)(struct cpssp *cpssp, int32_t val)
{
	*(int32_t *) &cpssp->NAME.fifo[cpssp->NAME.count] = val;

	cpssp->NAME.count++;
}

static int64_t
NAME_(load_mem_m64int)(struct cpssp *cpssp)
{
	int64_t val;

	val = *(int64_t *) &cpssp->NAME.fifo[cpssp->NAME.count];

	cpssp->NAME.count += 2;

	return val;
}

static void
NAME_(store_mem_m64int)(struct cpssp *cpssp, int64_t val)
{
	*(int64_t *) &cpssp->NAME.fifo[cpssp->NAME.count] = val;

	cpssp->NAME.count += 2;
}

static float32
NAME_(load_mem_m32fp)(struct cpssp *cpssp)
{
	float32 val;

	val = *(float32 *) &cpssp->NAME.fifo[cpssp->NAME.count];

	cpssp->NAME.count++;

	return val;
}

static void
NAME_(store_mem_m32fp)(struct cpssp *cpssp, float32 val)
{
	*(float32 *) &cpssp->NAME.fifo[cpssp->NAME.count] = val;

	cpssp->NAME.count++;
}

static float64
NAME_(load_mem_m64fp)(struct cpssp *cpssp)
{
	float64 val;

	val = *(float64 *) &cpssp->NAME.fifo[cpssp->NAME.count];

	cpssp->NAME.count += 2;

	return val;
}

static void
NAME_(store_mem_m64fp)(struct cpssp *cpssp, float64 val)
{
	*(float64 *) &cpssp->NAME.fifo[cpssp->NAME.count] = val;

	cpssp->NAME.count += 2;
}

static float80
NAME_(load_mem_m80fp)(struct cpssp *cpssp)
{
	float80 val;

	val = *(float80 *) &cpssp->NAME.fifo[cpssp->NAME.count];

	cpssp->NAME.count += 3;

	return val;
}

static void
NAME_(store_mem_m80fp)(struct cpssp *cpssp, float80 val)
{
	*(float80 *) &cpssp->NAME.fifo[cpssp->NAME.count] = val;

	cpssp->NAME.count += 3;
}

static void
NAME_(load_mem_m80bcd)(struct cpssp *cpssp, uint8_t bcd[10])
{
	memcpy(bcd, &cpssp->NAME.fifo[cpssp->NAME.count], 10);

	cpssp->NAME.count += 3;
}

static void
NAME_(store_mem_m80bcd)(struct cpssp *cpssp, uint8_t bcd[10])
{
	memcpy(&cpssp->NAME.fifo[cpssp->NAME.count], bcd, 10);

	cpssp->NAME.count += 3;
}

static void
NAME_(dec)(struct cpssp *cpssp)
{
	cpssp->NAME.top = (cpssp->NAME.top - 1) & 7;

	/* cpssp->NAME.c0 = ?; Undefined */
	cpssp->NAME.c1 = 0; /* Cleared to 0. */
	/* cpssp->NAME.c2 = ?; Undefined */
	/* cpssp->NAME.c3 = ?; Undefined */
}

static void
NAME_(inc)(struct cpssp *cpssp)
{
	cpssp->NAME.top = (cpssp->NAME.top + 1) & 7;

	/* cpssp->NAME.c0 = ?; Undefined */
	cpssp->NAME.c1 = 0; /* Cleared to 0. */
	/* cpssp->NAME.c2 = ?; Undefined */
	/* cpssp->NAME.c3 = ?; Undefined */
}

static void
NAME_(push)(struct cpssp *cpssp, float80 val)
{
	cpssp->NAME.top = (cpssp->NAME.top - 1) & 7;
	cpssp->NAME.used[cpssp->NAME.top] =  1;
	NAME_(store_data)(cpssp, 0, val);
}

static void
NAME_(pop)(struct cpssp *cpssp)
{
	cpssp->NAME.used[cpssp->NAME.top] =  0;
	cpssp->NAME.top = (cpssp->NAME.top + 1) & 7;
}

static uint8_t
NAME_(_fucom)(struct cpssp *cpssp, float80 op1, float80 op2)
{
	if (float80_isunordered(op1, op2)) {
		return (1 << 6) | (1 << 2) | (1 << 0);

	} else if (float80_isless(op1, op2)) {
		return (0 << 6) | (0 << 2) | (1 << 0);

	} else if (float80_isless(op2, op1)) {
		return (0 << 6) | (0 << 2) | (0 << 0);

	} else {
		/* op1 == op2 */
		return (1 << 6) | (0 << 2) | (0 << 0);
	}
}

static void
NAME_(fucom)(struct cpssp *cpssp, float80 op1, float80 op2)
{
	uint8_t res;

	res = NAME_(_fucom)(cpssp, op1, op2);

	cpssp->NAME.c3 = (res >> 6) & 1;
	cpssp->NAME.c2 = (res >> 2) & 1;
	cpssp->NAME.c0 = (res >> 0) & 1;
}

static void
NAME_(fcom)(struct cpssp *cpssp, float80 op1, float80 op2)
{
	uint8_t res;

	res = NAME_(_fucom)(cpssp, op1, op2);

	if (res == 0b1101) {
		NAME_(IA)(cpssp);
	}

	cpssp->NAME.c3 = (res >> 6) & 1;
	cpssp->NAME.c2 = (res >> 2) & 1;
	cpssp->NAME.c1 = 0;
	cpssp->NAME.c0 = (res >> 0) & 1;
}

#if defined CONFIG_CPU && 80686 <= CONFIG_CPU
static void
NAME_(fucomi)(struct cpssp *cpssp, float80 op1, float80 op2)
{
	uint8_t res;

	res = NAME_(_fucom)(cpssp, op1, op2);

	NAME_(core_set_eflags)(cpssp, res);
}

static void
NAME_(fcomi)(struct cpssp *cpssp, float80 op1, float80 op2)
{
	uint8_t res;

	res = NAME_(_fucom)(cpssp, op1, op2);

	if (res == 0b1101) {
		NAME_(IA)(cpssp);
	}

	NAME_(core_set_eflags)(cpssp, res);
}
#endif /* defined CONFIG_CPU && 80686 <= CONFIG_CPU */

static void
NAME_(ftst)(struct cpssp *cpssp, float80 op1)
{
	NAME_(fcom)(cpssp, op1, 0.0L);
}

static void
NAME_(fxam)(struct cpssp *cpssp, float80 op1)
{
	cpssp->NAME.c1 = float80_isneg(op1);

	if (! cpssp->NAME.used[cpssp->NAME.top]) {
		cpssp->NAME.c3 = 1;
		cpssp->NAME.c2 = 0;
		cpssp->NAME.c0 = 1;
	} else if (float80_isnan(op1)) {
		cpssp->NAME.c3 = 0;
		cpssp->NAME.c2 = 0;
		cpssp->NAME.c0 = 1;
	} else if (float80_isinf(op1)) {
		cpssp->NAME.c3 = 0;
		cpssp->NAME.c2 = 1;
		cpssp->NAME.c0 = 1;
	} else if (float80_iszero(op1)) {
		cpssp->NAME.c3 = 1;
		cpssp->NAME.c2 = 0;
		cpssp->NAME.c0 = 0;
	} else if (float80_isnorm(op1)) {
		cpssp->NAME.c3 = 0;
		cpssp->NAME.c2 = 1;
		cpssp->NAME.c0 = 0;
	} else if (float80_isdenorm(op1)) {
		cpssp->NAME.c3 = 1;
		cpssp->NAME.c2 = 1;
		cpssp->NAME.c0 = 0;
	} else {
		assert(0);
	}
}

static float80
NAME_(fchs)(struct cpssp *cpssp, float80 op1)
{
	return float80_chs(cpssp->NAME.rc, cpssp->NAME.pc,
			op1,
			&cpssp->NAME.pe, &cpssp->NAME.ue,
			&cpssp->NAME.oe, &cpssp->NAME.ze,
			&cpssp->NAME.de, &cpssp->NAME.ie);
}

static float80
NAME_(fabs)(struct cpssp *cpssp, float80 op1)
{
	return float80_abs(cpssp->NAME.rc, cpssp->NAME.pc,
			op1,
			&cpssp->NAME.pe, &cpssp->NAME.ue,
			&cpssp->NAME.oe, &cpssp->NAME.ze,
			&cpssp->NAME.de, &cpssp->NAME.ie);
}

static float80
NAME_(frndint)(struct cpssp *cpssp, float80 op1)
{
	return float80_rndint(cpssp->NAME.rc, cpssp->NAME.pc,
			op1,
			&cpssp->NAME.pe, &cpssp->NAME.ue,
			&cpssp->NAME.oe, &cpssp->NAME.ze,
			&cpssp->NAME.de, &cpssp->NAME.ie);
}

static float80
NAME_(fscale)(struct cpssp *cpssp, float80 op1, float80 op2)
{
	return float80_scale(cpssp->NAME.rc, cpssp->NAME.pc,
			op1, op2,
			&cpssp->NAME.pe, &cpssp->NAME.ue,
			&cpssp->NAME.oe, &cpssp->NAME.ze,
			&cpssp->NAME.de, &cpssp->NAME.ie);
}

static float80
NAME_(fprem)(struct cpssp *cpssp, float80 op1, float80 op2)
{
	return float80_fprem(cpssp->NAME.rc, cpssp->NAME.pc,
			op1, op2,
			&cpssp->NAME.pe, &cpssp->NAME.ue,
			&cpssp->NAME.oe, &cpssp->NAME.ze,
			&cpssp->NAME.de, &cpssp->NAME.ie,
			&cpssp->NAME.c0, &cpssp->NAME.c1,
			&cpssp->NAME.c2, &cpssp->NAME.c3);
}

static float80
NAME_(fprem1)(struct cpssp *cpssp, float80 op1, float80 op2)
{
	return float80_fprem1(cpssp->NAME.rc, cpssp->NAME.pc,
			op1, op2,
			&cpssp->NAME.pe, &cpssp->NAME.ue,
			&cpssp->NAME.oe, &cpssp->NAME.ze,
			&cpssp->NAME.de, &cpssp->NAME.ie,
			&cpssp->NAME.c0, &cpssp->NAME.c1,
			&cpssp->NAME.c2, &cpssp->NAME.c3);
}

static float80
NAME_(fsqrt)(struct cpssp *cpssp, float80 op1)
{
	return float80_sqrt(cpssp->NAME.rc, cpssp->NAME.pc,
			op1,
			&cpssp->NAME.pe, &cpssp->NAME.ue,
			&cpssp->NAME.oe, &cpssp->NAME.ze,
			&cpssp->NAME.de, &cpssp->NAME.ie);
}

static float80
NAME_(fptan)(struct cpssp *cpssp, float80 op1)
{
	return float80_tan(cpssp->NAME.rc, cpssp->NAME.pc,
			op1,
			&cpssp->NAME.pe, &cpssp->NAME.ue,
			&cpssp->NAME.oe, &cpssp->NAME.ze,
			&cpssp->NAME.de, &cpssp->NAME.ie);
}

static float80
NAME_(fpatan)(struct cpssp *cpssp, float80 op1, float80 op2)
{
	return float80_atan(cpssp->NAME.rc, cpssp->NAME.pc,
			op1, op2,
			&cpssp->NAME.pe, &cpssp->NAME.ue,
			&cpssp->NAME.oe, &cpssp->NAME.ze,
			&cpssp->NAME.de, &cpssp->NAME.ie);
}

static float80
NAME_(fsin)(struct cpssp *cpssp, float80 op1)
{
	return float80_sin(cpssp->NAME.rc, cpssp->NAME.pc,
			op1,
			&cpssp->NAME.pe, &cpssp->NAME.ue,
			&cpssp->NAME.oe, &cpssp->NAME.ze,
			&cpssp->NAME.de, &cpssp->NAME.ie);
}

static float80
NAME_(fcos)(struct cpssp *cpssp, float80 op1)
{
	return float80_cos(cpssp->NAME.rc, cpssp->NAME.pc,
			op1,
			&cpssp->NAME.pe, &cpssp->NAME.ue,
			&cpssp->NAME.oe, &cpssp->NAME.ze,
			&cpssp->NAME.de, &cpssp->NAME.ie);
}

static void
NAME_(fsincos)(struct cpssp *cpssp, float80 *res1p, float80 *res2p, float80 op1)
{
	float80_sincos(cpssp->NAME.rc, cpssp->NAME.pc,
			res1p, res2p, op1,
			&cpssp->NAME.pe, &cpssp->NAME.ue,
			&cpssp->NAME.oe, &cpssp->NAME.ze,
			&cpssp->NAME.de, &cpssp->NAME.ie);
}

static float80
NAME_(f2xm1)(struct cpssp *cpssp, float80 op1)
{
	return float80_2xm1(cpssp->NAME.rc, cpssp->NAME.pc,
			op1,
			&cpssp->NAME.pe, &cpssp->NAME.ue,
			&cpssp->NAME.oe, &cpssp->NAME.ze,
			&cpssp->NAME.de, &cpssp->NAME.ie);
}

static float80
NAME_(fyl2x)(struct cpssp *cpssp, float80 op1, float80 op2)
{
	return float80_yl2x(cpssp->NAME.rc, cpssp->NAME.pc,
			op1, op2,
			&cpssp->NAME.pe, &cpssp->NAME.ue,
			&cpssp->NAME.oe, &cpssp->NAME.ze,
			&cpssp->NAME.de, &cpssp->NAME.ie);
}

static float80
NAME_(fyl2xp1)(struct cpssp *cpssp, float80 op1, float80 op2)
{
	return float80_yl2xp1(cpssp->NAME.rc, cpssp->NAME.pc,
			op1, op2,
			&cpssp->NAME.pe, &cpssp->NAME.ue,
			&cpssp->NAME.oe, &cpssp->NAME.ze,
			&cpssp->NAME.de, &cpssp->NAME.ie);
}

static float80
NAME_(alu)(struct cpssp *cpssp, uint8_t reg_opex, float80 op1, float80 op2)
{
	float80 res;

	switch (reg_opex) {
	case 0x0: /* fadd */
		res = float80_add(cpssp->NAME.rc, cpssp->NAME.pc,
				op1, op2,
				&cpssp->NAME.pe, &cpssp->NAME.ue,
				&cpssp->NAME.oe, &cpssp->NAME.ze,
				&cpssp->NAME.de, &cpssp->NAME.ie);
		break;
	case 0x1: /* fmul */
		res = float80_mul(cpssp->NAME.rc, cpssp->NAME.pc,
				op1, op2,
				&cpssp->NAME.pe, &cpssp->NAME.ue,
				&cpssp->NAME.oe, &cpssp->NAME.ze,
				&cpssp->NAME.de, &cpssp->NAME.ie);
		break;
	case 0x2: /* fcom */
	case 0x3: /* fcomp */
		NAME_(fcom)(cpssp, op1, op2);
		res = 0.0; /* Not used. */
		break;
	case 0x4: /* fsubr */
		res = float80_sub(cpssp->NAME.rc, cpssp->NAME.pc,
				op1, op2,
				&cpssp->NAME.pe, &cpssp->NAME.ue,
				&cpssp->NAME.oe, &cpssp->NAME.ze,
				&cpssp->NAME.de, &cpssp->NAME.ie);
		break;
	case 0x5: /* fsub */
		res = float80_sub(cpssp->NAME.rc, cpssp->NAME.pc,
				op2, op1,
				&cpssp->NAME.pe, &cpssp->NAME.ue,
				&cpssp->NAME.oe, &cpssp->NAME.ze,
				&cpssp->NAME.de, &cpssp->NAME.ie);
		break;
	case 0x6: /* fdivr */
		res = float80_div(cpssp->NAME.rc, cpssp->NAME.pc,
				op1, op2,
				&cpssp->NAME.pe, &cpssp->NAME.ue,
				&cpssp->NAME.oe, &cpssp->NAME.ze,
				&cpssp->NAME.de, &cpssp->NAME.ie);
		break;
	case 0x7: /* fdiv */
		res = float80_div(cpssp->NAME.rc, cpssp->NAME.pc,
				op2, op1,
				&cpssp->NAME.pe, &cpssp->NAME.ue,
				&cpssp->NAME.oe, &cpssp->NAME.ze,
				&cpssp->NAME.de, &cpssp->NAME.ie);
		break;
	default:
		/* Mustn't happen. */
		assert(0);
	}

	return res;
}

#if defined CONFIG_CPU_CMOV_SUPPORT && CONFIG_CPU_CMOV_SUPPORT
static int
NAME_(test_cc)(struct cpssp *cpssp, uint8_t cond)
{
	uint8_t eflags;
	int ret;
	
	eflags = NAME_(core_get_eflags)(cpssp);

	switch (cond & 0x3) {
	case 0x0: /* b */
		ret = (eflags >> 0) & 1; /* carry flag */
		break;
	case 0x1: /* e */
		ret = (eflags >> 6) & 1; /* zero flag */
		break;
	case 0x2: /* be */
		ret = ((eflags >> 6) & 1) | ((eflags >> 0) & 1); /* zero or carry */
		break;
	case 0x3: /* u */
		ret = (eflags >> 2) & 1; /* parity flag */
		break;
	default:
		assert(0); /* Mustn't happen. */
	}
	ret ^= (cond >> 2) & 1;

	return ret;
}
#endif /* defined CONFIG_CPU_CMOV_SUPPORT && CONFIG_CPU_CMOV_SUPPORT */

static void
NAME_(step)(struct cpssp *cpssp, uint8_t opcode, uint8_t mod, uint8_t reg_opex, uint8_t rm)
{
	uint16_t word;
	int16_t i16;
	int32_t i32;
	int64_t i64;
	float32 f32;
	float64 f64;
	float80 op1;
	float80 op2;
	float80 res;
	float80 res2;
	uint8_t bcd[10];

	if (mod != 0x3) {
#if 0
fprintf(stderr, "FPU %02x %x %x %x %x%x%x%x%x%x\n", 0xd8 | opcode, reg_opex,
		cpssp->NAME.rc, cpssp->NAME.pc,
		cpssp->NAME.pm, cpssp->NAME.um,
		cpssp->NAME.om, cpssp->NAME.zm,
		cpssp->NAME.dm, cpssp->NAME.im);
#endif
		switch ((((opcode >> 0) & 0x7) << 3) | (reg_opex << 0)) {
#define _(x, y) (((x & 7) << 3) | y)
		case _(0xd8, 0x0): /* fadd m32fp */
		case _(0xd8, 0x1): /* fmul m32fp */
		case _(0xd8, 0x2): /* fcom m32fp */
		case _(0xd8, 0x3): /* fcomp m32fp */
		case _(0xd8, 0x4): /* fsub m32fp */
		case _(0xd8, 0x5): /* fsubr m32fp */
		case _(0xd8, 0x6): /* fdiv m32fp */
		case _(0xd8, 0x7): /* fdivr m32fp */

		case _(0xda, 0x0): /* fiadd m32int */
		case _(0xda, 0x1): /* fimul m32int */
		case _(0xda, 0x2): /* ficom m32int */
		case _(0xda, 0x3): /* ficomp m32int */
		case _(0xda, 0x4): /* fisub m32int */
		case _(0xda, 0x5): /* fisubr m32int */
		case _(0xda, 0x6): /* fidiv m32int */
		case _(0xda, 0x7): /* fidivr m32int */

		case _(0xdc, 0x0): /* fadd m64fp */
		case _(0xdc, 0x1): /* fmul m64fp */
		case _(0xdc, 0x2): /* fcom m64fp */
		case _(0xdc, 0x3): /* fcomp m64fp */
		case _(0xdc, 0x4): /* fsub m64fp */
		case _(0xdc, 0x5): /* fsubr m64fp */
		case _(0xdc, 0x6): /* fdiv m64fp */
		case _(0xdc, 0x7): /* fdivr m64fp */

		case _(0xde, 0x0): /* fiadd m16int */
		case _(0xde, 0x1): /* fimul m16int */
		case _(0xde, 0x2): /* ficom m16int */
		case _(0xde, 0x3): /* ficomp m16int */
		case _(0xde, 0x4): /* fisub m16int */
		case _(0xde, 0x5): /* fisubr m16int */
		case _(0xde, 0x6): /* fidiv m16int */
		case _(0xde, 0x7): /* fidivr m16int */
			op1 = NAME_(load_data)(cpssp, 0);
			switch ((opcode >> 0) & 0x7) {
			case 0x0:
				/* fxxx m32fp */
				f32 = NAME_(load_mem_m32fp)(cpssp);
				op2 = float80_from_f32(f32);
				break;
			case 0x2:
				/* fxxx m32int */
				i32 = NAME_(load_mem_m32int)(cpssp);
				op2 = float80_from_i32(i32);
				break;
			case 0x4:
				/* fxxx m64fp */
				f64 = NAME_(load_mem_m64fp)(cpssp);
				op2 = float80_from_f64(f64);
				break;
			case 0x6:
				/* fxxx m16int */
				i16 = NAME_(load_mem_m16int)(cpssp);
				op2 = float80_from_i16(i16);
				break;
			default:
				assert(0); /* Cannot happen. */
			}

			res = NAME_(alu)(cpssp, reg_opex, op1, op2);

			if (! NAME_(error_update)(cpssp)) {
				if (reg_opex != 0x2 /* fcom */
				 && reg_opex != 0x3) { /* fcomp */
					NAME_(store_data)(cpssp, 0, res);
				}
				if (reg_opex == 0x3) { /* fcomp */
					NAME_(pop)(cpssp);
				}
			}
			break;

		case _(0xd9, 0x0): /* fld m32fp */
		case _(0xd9, 0x1): /* (reserved) */
			f32 = NAME_(load_mem_m32fp)(cpssp);
			op1 = float80_from_f32(f32);
			NAME_(push)(cpssp, op1);
			break;

		case _(0xd9, 0x2): /* fst m32fp */
		case _(0xd9, 0x3): /* fstp m32fp */
			op1 = NAME_(load_data)(cpssp, 0);

			f32 = float80_to_f32(cpssp->NAME.rc, op1,
					&cpssp->NAME.pe, &cpssp->NAME.ue,
					&cpssp->NAME.oe, &cpssp->NAME.ze,
					&cpssp->NAME.de, &cpssp->NAME.ie);

			NAME_(store_mem_m32fp)(cpssp, f32);

			if (! NAME_(error_update)(cpssp)) {
				if (reg_opex & 1) {
					NAME_(pop)(cpssp);
				}
			}
			break;

		case _(0xdb, 0x0): /* fild m32int */
		case _(0xdb, 0x1): /* (reserved) */
			i32 = NAME_(load_mem_m32int)(cpssp);
			op1 = float80_from_i32(i32);
			NAME_(push)(cpssp, op1);
			break;

		case _(0xdb, 0x2): /* fist m32int */
		case _(0xdb, 0x3): /* fistp m32int */
			op1 = NAME_(load_data)(cpssp, 0);

			i32 = float80_to_i32(cpssp->NAME.rc, op1,
					&cpssp->NAME.pe, &cpssp->NAME.ue,
					&cpssp->NAME.oe, &cpssp->NAME.ze,
					&cpssp->NAME.de, &cpssp->NAME.ie);

			NAME_(store_mem_m32int)(cpssp, i32);

			if (! NAME_(error_update)(cpssp)) {
				if (reg_opex & 1) {
					NAME_(pop)(cpssp);
				}
			}
			break;

		case _(0xdd, 0x0): /* fld m64fp */
		case _(0xdd, 0x1): /* (reserved) */
			f64 = NAME_(load_mem_m64fp)(cpssp);
			op1 = float80_from_f64(f64);
			NAME_(push)(cpssp, op1);
			break;

		case _(0xdd, 0x2): /* fst m64fp */
		case _(0xdd, 0x3): /* fstp m64fp */
			op1 = NAME_(load_data)(cpssp, 0);

			f64 = float80_to_f64(cpssp->NAME.rc, op1,
					&cpssp->NAME.pe, &cpssp->NAME.ue,
					&cpssp->NAME.oe, &cpssp->NAME.ze,
					&cpssp->NAME.de, &cpssp->NAME.ie);

			NAME_(store_mem_m64fp)(cpssp, f64);

			if (! NAME_(error_update)(cpssp)) {
				if (reg_opex & 1) {
					NAME_(pop)(cpssp);
				}
			}
			break;

		case _(0xdf, 0x0): /* fild m16int */
		case _(0xdf, 0x1): /* (reserved) */
			i16 = NAME_(load_mem_m16int)(cpssp);
			op1 = float80_from_i16(i16);
			NAME_(push)(cpssp, op1);
			break;

		case _(0xdf, 0x2): /* fist m16int */
		case _(0xdf, 0x3): /* fistp m16int */
			op1 = NAME_(load_data)(cpssp, 0);

			i16 = float80_to_i16(cpssp->NAME.rc, op1,
					&cpssp->NAME.pe, &cpssp->NAME.ue,
					&cpssp->NAME.oe, &cpssp->NAME.ze,
					&cpssp->NAME.de, &cpssp->NAME.ie);

			NAME_(store_mem_m16int)(cpssp, i16);

			if (! NAME_(error_update)(cpssp)) {
				if (reg_opex & 1) {
					NAME_(pop)(cpssp);
				}
			}
			break;

		case _(0xd9, 0x4): /* fldenv m14/28byte */
			word = NAME_(load_mem_m16bit)(cpssp);
			NAME_(store_control)(cpssp, word);
			word = NAME_(load_mem_m16bit)(cpssp);
			NAME_(store_status)(cpssp, word);
			word = NAME_(load_mem_m16bit)(cpssp);
			NAME_(store_tag)(cpssp, word);
			break;

		case _(0xd9, 0x5): /* fldcw m2byte */
			word = NAME_(load_mem_m16bit)(cpssp);
			NAME_(store_control)(cpssp, word);
			break;

		case _(0xd9, 0x6): /* fstenv m14/28byte */
			word = NAME_(load_control)(cpssp);
			NAME_(store_mem_m16bit)(cpssp, word);
			word = NAME_(load_status)(cpssp);
			NAME_(store_mem_m16bit)(cpssp, word);
			word = NAME_(load_tag)(cpssp);
			NAME_(store_mem_m16bit)(cpssp, word);
			break;

		case _(0xd9, 0x7): /* fstcw m2byte */
			word = NAME_(load_control)(cpssp);
			NAME_(store_mem_m16bit)(cpssp, word);
			break;

		case _(0xdb, 0x4): /* (reserved) */
			assert(0); /* FIXME */
			break;

		case _(0xdb, 0x5): /* fld m80fp */
			op1 = NAME_(load_mem_m80fp)(cpssp);
			NAME_(push)(cpssp, op1);
			break;

		case _(0xdb, 0x6): /* (reserved) */
			assert(0); /* FIXME */
			break;

		case _(0xdb, 0x7): /* fstp m80fp */
			op1 = NAME_(load_data)(cpssp, 0);
			NAME_(store_mem_m80fp)(cpssp, op1);
			NAME_(pop)(cpssp);
			break;

		case _(0xdd, 0x4): /* frstor m14+80/28+80byte */
			word = NAME_(load_mem_m16bit)(cpssp);
			NAME_(store_control)(cpssp, word);
			word = NAME_(load_mem_m16bit)(cpssp);
			NAME_(store_status)(cpssp, word);
			word = NAME_(load_mem_m16bit)(cpssp);
			NAME_(store_tag)(cpssp, word);

			op1 = NAME_(load_mem_m80fp)(cpssp);
			NAME_(store_data)(cpssp, 0, op1);
			op1 = NAME_(load_mem_m80fp)(cpssp);
			NAME_(store_data)(cpssp, 1, op1);
			op1 = NAME_(load_mem_m80fp)(cpssp);
			NAME_(store_data)(cpssp, 2, op1);
			op1 = NAME_(load_mem_m80fp)(cpssp);
			NAME_(store_data)(cpssp, 3, op1);
			op1 = NAME_(load_mem_m80fp)(cpssp);
			NAME_(store_data)(cpssp, 4, op1);
			op1 = NAME_(load_mem_m80fp)(cpssp);
			NAME_(store_data)(cpssp, 5, op1);
			op1 = NAME_(load_mem_m80fp)(cpssp);
			NAME_(store_data)(cpssp, 6, op1);
			op1 = NAME_(load_mem_m80fp)(cpssp);
			NAME_(store_data)(cpssp, 7, op1);
			break;

		case _(0xdd, 0x5): /* (reserved) */
			assert(0); /* FIXME */
			break;

		case _(0xdd, 0x6): /* fsave m14+80/28+80byte */
			word = NAME_(load_control)(cpssp);
			NAME_(store_mem_m16bit)(cpssp, word);
			word = NAME_(load_status)(cpssp);
			NAME_(store_mem_m16bit)(cpssp, word);
			word = NAME_(load_tag)(cpssp);
			NAME_(store_mem_m16bit)(cpssp, word);

			op1 = NAME_(load_data)(cpssp, 0);
			NAME_(store_mem_m80fp)(cpssp, op1);
			op1 = NAME_(load_data)(cpssp, 1);
			NAME_(store_mem_m80fp)(cpssp, op1);
			op1 = NAME_(load_data)(cpssp, 2);
			NAME_(store_mem_m80fp)(cpssp, op1);
			op1 = NAME_(load_data)(cpssp, 3);
			NAME_(store_mem_m80fp)(cpssp, op1);
			op1 = NAME_(load_data)(cpssp, 4);
			NAME_(store_mem_m80fp)(cpssp, op1);
			op1 = NAME_(load_data)(cpssp, 5);
			NAME_(store_mem_m80fp)(cpssp, op1);
			op1 = NAME_(load_data)(cpssp, 6);
			NAME_(store_mem_m80fp)(cpssp, op1);
			op1 = NAME_(load_data)(cpssp, 7);
			NAME_(store_mem_m80fp)(cpssp, op1);

			NAME_(store_control)(cpssp, 0x037f);
			NAME_(store_status)(cpssp, 0x0000);
			NAME_(store_tag)(cpssp, 0xffff);
			break;

		case _(0xdd, 0x7): /* fsts m2byte */
			word = NAME_(load_status)(cpssp);
			NAME_(store_mem_m16bit)(cpssp, word);
			break;

		case _(0xdf, 0x4): /* fbld m80bcd */
			NAME_(load_mem_m80bcd)(cpssp, bcd);
			op1 = float80_from_bcd(bcd);
			NAME_(push)(cpssp, op1);
			break;

		case _(0xdf, 0x5): /* fild m64int */
			i64 = NAME_(load_mem_m64int)(cpssp);
			op1 = float80_from_i64(i64);
			NAME_(push)(cpssp, op1);
			break;

		case _(0xdf, 0x6): /* fbstp m80bcd */
			op1 = NAME_(load_data)(cpssp, 0);

			float80_to_bcd(bcd, cpssp->NAME.rc, op1,
					&cpssp->NAME.pe, &cpssp->NAME.ue,
					&cpssp->NAME.oe, &cpssp->NAME.ze,
					&cpssp->NAME.de, &cpssp->NAME.ie);

			NAME_(store_mem_m80bcd)(cpssp, bcd);

			if (! NAME_(error_update)(cpssp)) {
				NAME_(pop)(cpssp);
			}
			break;

		case _(0xdf, 0x7): /* fistp m64int */
			op1 = NAME_(load_data)(cpssp, 0);

			i64 = float80_to_i64(cpssp->NAME.rc, op1,
					&cpssp->NAME.pe, &cpssp->NAME.ue,
					&cpssp->NAME.oe, &cpssp->NAME.ze,
					&cpssp->NAME.de, &cpssp->NAME.ie);

			NAME_(store_mem_m64int)(cpssp, i64);

			if (! NAME_(error_update)(cpssp)) {
				NAME_(pop)(cpssp);
			}
			break;

		default:
			assert(0); /* Cannot happen. */
#undef _
		}

	} else {
#if 0
fprintf(stderr, "FPU %04x %x %x %x%x%x%x%x%x\n", 0xd8c0 | (opcode << 8) | (reg_opex << 3),
		cpssp->NAME.rc, cpssp->NAME.pc,
		cpssp->NAME.pm, cpssp->NAME.um,
		cpssp->NAME.om, cpssp->NAME.zm,
		cpssp->NAME.dm, cpssp->NAME.im);
#endif
		switch ((((opcode >> 0) & 0x7) << 3) | (reg_opex << 0)) {
#define _(x)	((((x >> 8) & 0x7) << 3) | ((x >> 3) & 0x7))
		case _(0xd8c0): /* fadd */
		case _(0xd8c8): /* fmul */
		case _(0xd8d0): /* fcom */
		case _(0xd8d8): /* fcomp */
		case _(0xd8e0): /* fsub */
		case _(0xd8e8): /* fsubr */
		case _(0xd8f0): /* fdiv */
		case _(0xd8f8): /* fdivr */
			op1 = NAME_(load_data)(cpssp, 0);
			op2 = NAME_(load_data)(cpssp, rm);

			res = NAME_(alu)(cpssp, reg_opex, op1, op2);

			if (! NAME_(error_update)(cpssp)) {
				if (reg_opex != 0x2 /* fcom */
				 && reg_opex != 0x3) { /* fcomp */
					NAME_(store_data)(cpssp, 0, res);
				}
				if (reg_opex == 0x3) { /* fcomp */
					NAME_(pop)(cpssp);
				}
			}
			break;

		case _(0xd9c0): /* fld */
			op1 = NAME_(load_data)(cpssp, rm);
			NAME_(push)(cpssp, op1);
			break;

		case _(0xd9c8): /* fxch */
			op1 = NAME_(load_data)(cpssp, 0);
			op2 = NAME_(load_data)(cpssp, rm);
			NAME_(store_data)(cpssp, rm, op1);
			NAME_(store_data)(cpssp, 0, op2);
			break;

		case _(0xd9d0):
			switch (rm) {
			case 0x0: /* fnop */
				break;

			case 0x1:
			case 0x2:
			case 0x3:
			case 0x4:
			case 0x5:
			case 0x6:
			case 0x7:
			illegal:;
				fprintf(stderr, "WARNING: Illegal FPU instruction 0x%02x 0x%02x.\n",
						0xd8 | opcode, reg_opex << 3);
				NAME_(IA)(cpssp);
				break;

			default:
				assert(0); /* Cannot happen. */
			}
			break;

		case _(0xd9d8):
			goto illegal;

		case _(0xd9e0):
			switch (rm) {
			case 0x0: /* fchs */
				op1 = NAME_(load_data)(cpssp, 0);

				res = NAME_(fchs)(cpssp, op1);

				NAME_(store_data)(cpssp, 0, res);
				break;

			case 0x1: /* fabs */
				op1 = NAME_(load_data)(cpssp, 0);

				res = NAME_(fabs)(cpssp, op1);

				NAME_(store_data)(cpssp, 0, res);
				break;

			case 0x2:
			case 0x3:
				goto illegal;

			case 0x4: /* ftst */
				op1 = NAME_(load_data)(cpssp, 0);
				NAME_(ftst)(cpssp, op1);
				break;

			case 0x5: /* fxam */
				op1 = NAME_(load_data)(cpssp, 0);
				NAME_(fxam)(cpssp, op1);
				break;

			case 0x6:
			case 0x7:
				goto illegal;

			default:
				assert(0); /* FIXME */
			}
			break;

		case _(0xd9e8):
			switch (rm) {
			case 0x0: /* fld1 */
				res = float80_1();
				break;

			case 0x1: /* fldl2t */
				res = float80_l2t();
				break;

			case 0x2: /* fldl2e */
				res = float80_l2e();
				break;

			case 0x3: /* fldpi */
				res = float80_pi();
				break;

			case 0x4: /* fldlg2 */
				res = float80_lg2();
				break;

			case 0x5: /* fldln2 */
				res = float80_ln2();
				break;

			case 0x6: /* fldz */
				res = float80_0();
				break;

			case 0x7:
				goto illegal;

			default:
				assert(0); /* Cannot happen. */
			}
			NAME_(push)(cpssp, res);
			break;

		case _(0xd9f0):
			switch (rm) {
			case 0x0: /* f2xm1 */
				op1 = NAME_(load_data)(cpssp, 0);

				res = NAME_(f2xm1)(cpssp, op1);

				NAME_(store_data)(cpssp, 0, res);
				break;

			case 0x1: /* fyl2x */
				op1 = NAME_(load_data)(cpssp, 0);
				op2 = NAME_(load_data)(cpssp, 1);

				res = NAME_(fyl2x)(cpssp, op1, op2);

				NAME_(store_data)(cpssp, 1, res);
				NAME_(pop)(cpssp);
				break;

			case 0x2: /* fptan */
				op1 = NAME_(load_data)(cpssp, 0);

				res = NAME_(fptan)(cpssp, op1);

				NAME_(store_data)(cpssp, 0, res);
				NAME_(push)(cpssp, 1.0L);
				break;

			case 0x3: /* fpatan */
				op1 = NAME_(load_data)(cpssp, 0);
				op2 = NAME_(load_data)(cpssp, 1);

				res = NAME_(fpatan)(cpssp, op1, op2);

				NAME_(store_data)(cpssp, 1, res);
				NAME_(pop)(cpssp);
				break;

			case 0x4: /* fxtract */
				assert(0); /* FIXME */

			case 0x5: /* fprem1 */
				op1 = NAME_(load_data)(cpssp, 0);
				op2 = NAME_(load_data)(cpssp, 1);

				res = NAME_(fprem1)(cpssp, op1, op2);

				NAME_(store_data)(cpssp, 0, res);
				break;

			case 0x6: /* fdecstp */
				NAME_(dec)(cpssp);
				break;

			case 0x7: /* fincstp */
				NAME_(inc)(cpssp);
				break;

			default:
				assert(0); /* Cannot happen. */
			}
			break;

		case _(0xd9f8):
			switch (rm) {
			case 0x0: /* fprem */
				op1 = NAME_(load_data)(cpssp, 0);
				op2 = NAME_(load_data)(cpssp, 1);

				res = NAME_(fprem)(cpssp, op1, op2);

				NAME_(store_data)(cpssp, 0, res);
				break;

			case 0x1: /* fyl2xp1 */
				op1 = NAME_(load_data)(cpssp, 0);
				op2 = NAME_(load_data)(cpssp, 1);

				res = NAME_(fyl2xp1)(cpssp, op1, op2);

				NAME_(store_data)(cpssp, 1, res);
				NAME_(pop)(cpssp);
				break;

			case 0x2: /* fsqrt */
				op1 = NAME_(load_data)(cpssp, 0);

				res = NAME_(fsqrt)(cpssp, op1);

				NAME_(store_data)(cpssp, 0, res);
				break;

			case 0x3: /* fsincos */
				op1 = NAME_(load_data)(cpssp, 0);

				NAME_(fsincos)(cpssp, &res, &res2, op1);

				NAME_(store_data)(cpssp, 0, res);
				NAME_(push)(cpssp, res2);
				break;

			case 0x4: /* frndint */
				op1 = NAME_(load_data)(cpssp, 0);

				res = NAME_(frndint)(cpssp, op1);

				NAME_(store_data)(cpssp, 0, res);
				break;

			case 0x5: /* fscale */
				op1 = NAME_(load_data)(cpssp, 0);
				op2 = NAME_(load_data)(cpssp, 1);

				res = NAME_(fscale)(cpssp, op1, op2);

				NAME_(store_data)(cpssp, 0, res);
				break;

			case 0x6: /* fsin */
				op1 = NAME_(load_data)(cpssp, 0);

				res = NAME_(fsin)(cpssp, op1);

				NAME_(store_data)(cpssp, 0, res);
				break;

			case 0x7: /* fcos */
				op1 = NAME_(load_data)(cpssp, 0);

				res = NAME_(fcos)(cpssp, op1);

				NAME_(store_data)(cpssp, 0, res);
				break;

			default:
				assert(0); /* Cannot happen. */
			}
			break;

		case _(0xdac0): /* P6: fcmovb */
		case _(0xdac8): /* P6: fcmove */
		case _(0xdad0): /* P6: fcmovbe */
		case _(0xdad8): /* P6: fcmovu */
		case _(0xdbc0): /* P6: fcmovnb */
		case _(0xdbc8): /* P6: fcmovne */
		case _(0xdbd0): /* P6: fcmovnbe */
		case _(0xdbd8): /* P6: fcmovnu */
#if ! defined CONFIG_CPU_CMOV_SUPPORT || ! CONFIG_CPU_CMOV_SUPPORT
			goto illegal; /* FIXME */

#else /* defined CONFIG_CPU_CMOV_SUPPORT && CONFIG_CPU_CMOV_SUPPORT */
			if (NAME_(test_cc)(cpssp, ((opcode & 1) << 2) | reg_opex)) {
				op1 = NAME_(load_data)(cpssp, rm);
				NAME_(store_data)(cpssp, 0, op1);
			}
			break;
#endif /* defined CONFIG_CPU_CMOV_SUPPORT && CONFIG_CPU_CMOV_SUPPORT */

		case _(0xdae0):
			goto illegal;

		case _(0xdae8):
			switch (rm) {
			case 0x0:
				goto illegal;

			case 0x1: /* fucompp */
				op1 = NAME_(load_data)(cpssp, 0);
				op2 = NAME_(load_data)(cpssp, 1);
				NAME_(fucom)(cpssp, op1, op2);
				NAME_(pop)(cpssp);
				NAME_(pop)(cpssp);
				break;

			case 0x2:
			case 0x3:
			case 0x4:
			case 0x5:
			case 0x6:
			case 0x7:
				goto illegal;

			default:
				assert(0); /* Cannot happen. */
			}
			break;

		case _(0xdaf0):
		case _(0xdaf8):
			goto illegal;

		case _(0xdbe0):
			switch (rm) {
			case 0x0: /* 287 only: feni */
				/* NOP */
				break;

			case 0x1: /* 287 only: fdisi */
				/* NOP */
				break;

			case 0x2: /* fnclex */
				cpssp->NAME.pe = 0;
				cpssp->NAME.ue = 0;
				cpssp->NAME.oe = 0;
				cpssp->NAME.ze = 0;
				cpssp->NAME.de = 0;
				cpssp->NAME.ie = 0;
				NAME_(error_update)(cpssp);
				break;

			case 0x3: /* fninit */
				NAME_(store_control)(cpssp, 0x037f);
				NAME_(store_status)(cpssp, 0x0000);
				NAME_(store_tag)(cpssp, 0xffff);
				break;

			case 0x4: /* 287 only: fsetpm */
				/* NOP */
				break;

			case 0x5:
			case 0x6:
			case 0x7:
				goto illegal;

			default:
				assert(0); /* FIXME */
			}
			break;

		case _(0xdbe8): /* fucomi */
		case _(0xdbf0): /* fcomi */
		case _(0xdfe8): /* fucomip */
		case _(0xdff0): /* fcomip */
#if ! defined CONFIG_CPU || CONFIG_CPU < 80686
			goto illegal;
#else /* defined CONFIG_CPU && 80686 <= CONFIG_CPU */
			op1 = NAME_(load_data)(cpssp, 0);
			op2 = NAME_(load_data)(cpssp, rm);

			if (reg_opex == 0x5) {
				NAME_(fucomi)(cpssp, op1, op2);
			} else {
				NAME_(fcomi)(cpssp, op1, op2);
			}

			if (! NAME_(error_update)(cpssp)) {
				if (opcode & 0x4) {
					/* fucomip */
					/* fcomip */
					NAME_(pop)(cpssp);
				}
			}
			break;
#endif /* defined CONFIG_CPU && 80686 <= CONFIG_CPU */

		case _(0xdbf8):
			goto illegal;

		case _(0xdcc0): /* fadd */
		case _(0xdcc8): /* fmul */
		case _(0xdcd0): /* (reserved) */
		case _(0xdcd8): /* (reserved) */
		case _(0xdce0): /* fsubr */
		case _(0xdce8): /* fsub */
		case _(0xdcf0): /* fdivr */
		case _(0xdcf8): /* fdiv */
			op1 = NAME_(load_data)(cpssp, 0);
			op2 = NAME_(load_data)(cpssp, rm);

			res = NAME_(alu)(cpssp, reg_opex, op1, op2);

			if (! NAME_(error_update)(cpssp)) {
				if (reg_opex != 0x2 /* fcom */
				 && reg_opex != 0x3) { /* fcomp */
					NAME_(store_data)(cpssp, rm, res);
				}
				if (reg_opex == 0x3) { /* fcomp */
					NAME_(pop)(cpssp);
				}
			}
			break;

		case _(0xddc0): /* ffree */
		case _(0xddc8): /* ffreep (undocumented) */
			cpssp->NAME.used[rm] = 0;
			if (reg_opex & 1) {
				NAME_(pop)(cpssp);
			}
			break;

		case _(0xddd0): /* fst */
		case _(0xddd8): /* fstp */
			op1 = NAME_(load_data)(cpssp, 0);
			NAME_(store_data)(cpssp, rm, op1);
			if (reg_opex & 1) {
				NAME_(pop)(cpssp);
			}
			break;

		case _(0xdde0): /* fucom */
		case _(0xdde8): /* fucomp */
			op1 = NAME_(load_data)(cpssp, 0);
			op2 = NAME_(load_data)(cpssp, rm);
			NAME_(fucom)(cpssp, op1, op2);
			if (reg_opex & 1) {
				NAME_(pop)(cpssp);
			}
			break;

		case _(0xddf0): /* (reserved) */
		case _(0xddf8): /* (reserved) */
			goto illegal;

		case _(0xdec0): /* faddp */
		case _(0xdec8): /* fmulp */
		case _(0xded0): /* (reserved) */
		case _(0xded8): /* fcompp / (reserved) */
		case _(0xdee0): /* fsubrp */
		case _(0xdee8): /* fsubp */
		case _(0xdef0): /* fdivrp */
		case _(0xdef8): /* fdivp */
			op1 = NAME_(load_data)(cpssp, 0);
			op2 = NAME_(load_data)(cpssp, rm);

			res = NAME_(alu)(cpssp, reg_opex, op1, op2);

			if (! NAME_(error_update)(cpssp)) {
				if (reg_opex != 0x2 /* fcomp */
				 && reg_opex != 0x3) { /* fcompp */
					NAME_(store_data)(cpssp, rm, res);
				}
				if (reg_opex == 0x3) { /* fcompp */
					NAME_(pop)(cpssp);
				}
				NAME_(pop)(cpssp);
			}
			break;

		case _(0xdfc0):
		case _(0xdfc8):
		case _(0xdfd0):
		case _(0xdfd8):
			goto illegal;

		case _(0xdfe0):
			switch (rm) {
			case 0x0:
				/* fsts %ax */
				word = NAME_(load_status)(cpssp);
				NAME_(store_mem_m16bit)(cpssp, word);
				break;

			case 0x1:
			case 0x2:
			case 0x3:
			case 0x4:
			case 0x5:
			case 0x6:
			case 0x7:
				goto illegal;

			default:
				assert(0); /* Cannot happen. */
			}
			break;

		case _(0xdff8):
			goto illegal;

		default:
			assert(0); /* Cannot happen. */
#undef _
		}
	}
}

static void
NAME_(write)(struct cpssp *cpssp, uint32_t addr, uint32_t val)
{
	if (! ((addr >> 2) & 1)) {
		/* Write command. */
		uint8_t mod;
		uint8_t op;

		mod = (val >> 6) & 0x3;
		op = (((val >> 8) & 0x7) << 3) | (((val >> 3) & 0x7) << 0);

		cpssp->NAME.opcode = val;

		if (mod != 3) {
			switch (op) {
			case (0x0 << 3) | 0x0:
			case (0x0 << 3) | 0x1:
			case (0x0 << 3) | 0x2:
			case (0x0 << 3) | 0x3:
			case (0x0 << 3) | 0x4:
			case (0x0 << 3) | 0x5:
			case (0x0 << 3) | 0x6:
			case (0x0 << 3) | 0x7:
				/* fxxx m32fp */
				cpssp->NAME.in = 1;
				cpssp->NAME.out = 0;
				break;

			case (0x2 << 3) | 0x0:
			case (0x2 << 3) | 0x1:
			case (0x2 << 3) | 0x2:
			case (0x2 << 3) | 0x3:
			case (0x2 << 3) | 0x4:
			case (0x2 << 3) | 0x5:
			case (0x2 << 3) | 0x6:
			case (0x2 << 3) | 0x7:
				/* fxxx m32int */
				cpssp->NAME.in = 1;
				cpssp->NAME.out = 0;
				break;

			case (0x4 << 3) | 0x0:
			case (0x4 << 3) | 0x1:
			case (0x4 << 3) | 0x2:
			case (0x4 << 3) | 0x3:
			case (0x4 << 3) | 0x4:
			case (0x4 << 3) | 0x5:
			case (0x4 << 3) | 0x6:
			case (0x4 << 3) | 0x7:
				/* fxxx m64fp */
				cpssp->NAME.in = 2;
				cpssp->NAME.out = 0;
				break;

			case (0x6 << 3) | 0x0:
			case (0x6 << 3) | 0x1:
			case (0x6 << 3) | 0x2:
			case (0x6 << 3) | 0x3:
			case (0x6 << 3) | 0x4:
			case (0x6 << 3) | 0x5:
			case (0x6 << 3) | 0x6:
			case (0x6 << 3) | 0x7:
				/* fxxx m16int */
				cpssp->NAME.in = 1;
				cpssp->NAME.out = 0;
				break;

			case (0x1 << 3) | 0x0: /* fld m32fp */
			case (0x1 << 3) | 0x1: /* (reserved) */
				cpssp->NAME.in = 1;
				cpssp->NAME.out = 0;
				break;

			case (0x1 << 3) | 0x2: /* fst m32fp */
			case (0x1 << 3) | 0x3: /* fstp m32fp */
				cpssp->NAME.in = 0;
				cpssp->NAME.out = 1;
				break;

			case (0x3 << 3) | 0x0: /* fild m32int */
			case (0x3 << 3) | 0x1: /* (reserved) */
				cpssp->NAME.in = 1;
				cpssp->NAME.out = 0;
				break;

			case (0x3 << 3) | 0x2: /* fist m32int */
			case (0x3 << 3) | 0x3: /* fistp m32int */
				cpssp->NAME.in = 0;
				cpssp->NAME.out = 1;
				break;

			case (0x5 << 3) | 0x0: /* fld m64fp */
			case (0x5 << 3) | 0x1: /* (reserved) */
				cpssp->NAME.in = 2;
				cpssp->NAME.out = 0;
				break;

			case (0x5 << 3) | 0x2: /* fst m64fp */
			case (0x5 << 3) | 0x3: /* fstp m64fp */
				cpssp->NAME.in = 0;
				cpssp->NAME.out = 2;
				break;

			case (0x7 << 3) | 0x0: /* fild m16int */
			case (0x7 << 3) | 0x1: /* (reserved) */
				cpssp->NAME.in = 1;
				cpssp->NAME.out = 0;
				break;

			case (0x7 << 3) | 0x2: /* fist m16int */
			case (0x7 << 3) | 0x3: /* fistp m16int */
				cpssp->NAME.in = 0;
				cpssp->NAME.out = 1;
				break;

			case (0x1 << 3) | 0x4: /* fldenv m14/28byte */
				cpssp->NAME.in = 3;
				cpssp->NAME.out = 0;
				break;

			case (0x1 << 3) | 0x5: /* fldcw m2byte */
				cpssp->NAME.in = 1;
				cpssp->NAME.out = 0;
				break;

			case (0x1 << 3) | 0x6: /* fstenv m14/28byte */
				cpssp->NAME.in = 0;
				cpssp->NAME.out = 3;
				break;

			case (0x1 << 3) | 0x7: /* fstcw m2byte */
				cpssp->NAME.in = 0;
				cpssp->NAME.out = 1;
				break;

			case (0x3 << 3) | 0x4: /* (reserved) */
				assert(0); /* FIXME */
				break;

			case (0x3 << 3) | 0x5: /* fld m80fp */
				cpssp->NAME.in = 3;
				cpssp->NAME.out = 0;
				break;

			case (0x3 << 3) | 0x6: /* (reserved) */
				assert(0); /* FIXME */
				break;

			case (0x3 << 3) | 0x7: /* fistp m80fp */
				cpssp->NAME.in = 0;
				cpssp->NAME.out = 3;
				break;

			case (0x5 << 3) | 0x4: /* frstor m14+80/28+80byte */
				cpssp->NAME.in = 3 + 8 * 3;
				cpssp->NAME.out = 0;
				break;

			case (0x5 << 3) | 0x5: /* (reserved) */
				assert(0); /* FIXME */
				break;

			case (0x5 << 3) | 0x6: /* fsave m14+80/28+80byte */
				cpssp->NAME.in = 0;
				cpssp->NAME.out = 3 + 8 * 3;
				break;

			case (0x5 << 3) | 0x7: /* fsts m2byte */
				cpssp->NAME.in = 0;
				cpssp->NAME.out = 1;
				break;

			case (0x7 << 3) | 0x4: /* fbld m80bcd */
				cpssp->NAME.in = 3;
				cpssp->NAME.out = 0;
				break;

			case (0x7 << 3) | 0x5: /* fild m64int */
				cpssp->NAME.in = 2;
				cpssp->NAME.out = 0;
				break;

			case (0x7 << 3) | 0x6: /* fbstp m80bcd */
				cpssp->NAME.in = 0;
				cpssp->NAME.out = 3;
				break;

			case (0x7 << 3) | 0x7: /* fistp m64int */
				cpssp->NAME.in = 0;
				cpssp->NAME.out = 2;
				break;

			default:
				assert(0); /* Cannot happen. */
			}

		} else {
			if (op == ((0x7 << 3) | 0x4)) {
				/* fsts %ax */
				cpssp->NAME.in = 0;
				cpssp->NAME.out = 1;
			} else {
				cpssp->NAME.in = 0;
				cpssp->NAME.out = 0;
			}
		}

		cpssp->NAME.count = 0;

	} else {
		/* Write data. */
		assert(0 < cpssp->NAME.in);
		cpssp->NAME.fifo[cpssp->NAME.count] = val;
		cpssp->NAME.count++;
		cpssp->NAME.in--;
	}

	if (cpssp->NAME.in == 0) {
		/* Execute command. */
		if (DEBUG_CONTROL_FLOW) {
			fprintf(stderr, "%s: opcode=%02x %d %d %d\n", __FUNCTION__,
					0xd8 | ((cpssp->NAME.opcode >> 8) & 0x7),
					(cpssp->NAME.opcode >> 6) & 0x3,
					(cpssp->NAME.opcode >> 3) & 0x7,
					(cpssp->NAME.opcode >> 0) & 0x7);
		}

		cpssp->NAME.count = 0;
		NAME_(step)(cpssp,
				(cpssp->NAME.opcode >> 8) & 0x7,
				(cpssp->NAME.opcode >> 6) & 0x3,
				(cpssp->NAME.opcode >> 3) & 0x7,
				(cpssp->NAME.opcode >> 0) & 0x7);
		cpssp->NAME.count = 0;
	}
}

static void
NAME_(read)(struct cpssp *cpssp, uint32_t addr, uint32_t *valp)
{
	if (! ((addr >> 2) & 1)) {
		/* Read status. */
		*valp = 0;
	} else {
		/* Read data. */
		assert(0 < cpssp->NAME.out);
		*valp = cpssp->NAME.fifo[cpssp->NAME.count];
		cpssp->NAME.count++;
		cpssp->NAME.out--;
	}
}

static void
NAME_(n_reset_set)(struct cpssp *cpssp, unsigned int n_val)
{
	cpssp->NAME.top = 0;
	cpssp->NAME.rc = 0; /* Correct? FIXME */
	cpssp->NAME.pc = 3; /* Correct? FIXME */
	cpssp->NAME.pm = 1; /* Correct? FIXME */
	cpssp->NAME.um = 1; /* Correct? FIXME */
	cpssp->NAME.om = 1; /* Correct? FIXME */
	cpssp->NAME.zm = 1; /* Correct? FIXME */
	cpssp->NAME.dm = 1; /* Correct? FIXME */
	cpssp->NAME.im = 1; /* Correct? FIXME */
	/* ... FIXME */
}

static void
NAME_(create)(struct cpssp *cpssp)
{
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#undef DEBUG_CONTROL_FLOW

#endif /* BEHAVIOR */
