import argparse

from twisted.internet.defer import inlineCallbacks

from juju.control.utils import get_environment
from juju.lib.format import YAMLFormat
from juju.state.service import ServiceStateManager


def configure_subparser(subparsers):
    sub_parser = subparsers.add_parser(
        "get",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        help=config_get.__doc__,
        description=command.__doc__)

    sub_parser.add_argument(
        "--environment", "-e",
        help="Environment to utilize.")

    sub_parser.add_argument(
        "--schema", "-s", action="store_true", default=False,
        help="Display the schema only")

    sub_parser.add_argument(
        "service_name",
        help="The name of the service to retrieve settings for")

    return sub_parser


def command(options):
    """Get service config options.

    Charms may define dynamic options which may be tweaked at
    deployment time, or over the lifetime of the service.  This
    command allows display the current value of these settings
    in yaml format.

    $ juju get wordpress

    {'service': 'wordpress',
     'charm': 'local:series/wordpress-3',
     'settings': {'blog-title': {
                    'description': 'A descriptive title used for the blog.',
                    'type': 'string',
                    'value': 'Hello World'}}},
    """
    environment = get_environment(options)

    return config_get(environment,
                      options.service_name,
                      options.schema)


@inlineCallbacks
def config_get(environment, service_name, display_schema):
    """Get service settings.
    """
    provider = environment.get_machine_provider()
    client = yield provider.connect()
    try:
        # Get the service
        service_manager = ServiceStateManager(client)
        service = yield service_manager.get_service_state(service_name)

        # Retrieve schema
        charm = yield service.get_charm_state()
        schema = yield charm.get_config()
        schema_dict = schema.as_dict()
        display_dict = {"service": service.service_name,
                        "charm": (yield service.get_charm_id()),
                        "settings": schema_dict}

        # Get current settings
        settings = yield service.get_config()
        settings = dict(settings.items())

        # Merge current settings into schema/display dict
        for k, v in schema_dict.items():
            # Display defaults for unset values.
            if k in settings:
                v['value'] = settings[k]
            else:
                v['value'] = "-Not set-"

            if 'default' in v:
                if v['default'] == settings[k]:
                    v['default'] = True
                else:
                    del v['default']

        print YAMLFormat().format(display_dict)
    finally:
        yield client.close()
