/*
 * This program source code file is part of KiCad, a free EDA CAD application.
 *
 * Copyright (C) 2015-2018 KiCad Developers, see AUTHORS.txt for contributors.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you may find one here:
 * http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 * or you may search the http://www.gnu.org website for the version 2 license,
 * or you may write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 */

#include <sch_edit_frame.h>
#include <sch_component.h>
#include <invoke_sch_dialog.h>
#include <dialog_rescue_each_base.h>
#include <kiface_i.h>
#include <class_library.h>
#include <class_libentry.h>
#include <set>
#include <vector>
#include <project_rescue.h>
#include <eeschema_config.h>

class DIALOG_RESCUE_EACH: public DIALOG_RESCUE_EACH_BASE
{
public:
    /**
     * This dialog asks the user which rescuable, cached parts he wants to rescue.
     *
     * Any rejects will be pruned from aCandidates.
     *
     * @param aParent - the SCH_EDIT_FRAME calling this
     * @param aRescuer - the active RESCUER instance
     * @param aAskShowAgain - if true, a "Never Show Again" button will be included
     */
    DIALOG_RESCUE_EACH( SCH_EDIT_FRAME* aParent, RESCUER& aRescuer, bool aAskShowAgain );

    ~DIALOG_RESCUE_EACH();

private:
    SCH_EDIT_FRAME* m_Parent;
    wxConfigBase*   m_Config;
    RESCUER*        m_Rescuer;
    bool            m_AskShowAgain;

    bool TransferDataToWindow() override;
    bool TransferDataFromWindow() override;
    void PopulateConflictList();
    void PopulateInstanceList();
    void OnConflictSelect( wxDataViewEvent& event ) override;
    void OnNeverShowClick( wxCommandEvent& event ) override;
    void OnCancelClick( wxCommandEvent& event ) override;
    void OnHandleCachePreviewRepaint( wxPaintEvent& aRepaintEvent ) override;
    void OnHandleLibraryPreviewRepaint( wxPaintEvent& aRepaintEvent ) override;
    void renderPreview( LIB_PART* aComponent, int aUnit, wxPanel* panel );
};


DIALOG_RESCUE_EACH::DIALOG_RESCUE_EACH( SCH_EDIT_FRAME* aParent, RESCUER& aRescuer,
                                        bool aAskShowAgain )
    : DIALOG_RESCUE_EACH_BASE( aParent ),
      m_Parent( aParent ),
      m_Rescuer( &aRescuer ),
      m_AskShowAgain( aAskShowAgain )
{
    m_Config = Kiface().KifaceSettings();
    m_stdButtonsOK->SetDefault();

    // Set the info message, customized to include the proper suffix.
    wxString info =
        _( "This schematic was made using older symbol libraries which may break the "
           "schematic.  Some symbols may need to be linked to a different symbol name.  "
           "Some symbols may need to be \"rescued\" (copied and renamed) into a new library.\n\n"
           "The following changes are recommended to update the project." );
    m_htmlPrompt->AppendToPage( info );

    // wxDataViewListCtrl seems to do a poor job of laying itself out so help it along here.
    wxString header = _( "Accept" );
    wxFont font = m_ListOfConflicts->GetFont();

    font.MakeBold();

    wxClientDC dc( this );

    dc.SetFont( font );

    int padding = 30;
    int width = dc.GetTextExtent( header ).GetWidth();

    m_ListOfConflicts->AppendToggleColumn( header, wxDATAVIEW_CELL_ACTIVATABLE, width,
                                           wxALIGN_CENTER );

    header = _( "Symbol Name" );
    width = dc.GetTextExtent( header ).GetWidth() + padding;
    m_ListOfConflicts->AppendTextColumn( header, wxDATAVIEW_CELL_INERT, width );

    header = _( "Action Taken" );
    width = dc.GetTextExtent( header ).GetWidth() + padding;
    m_ListOfConflicts->AppendTextColumn( header, wxDATAVIEW_CELL_INERT, width );

    header = _( "Reference" );
    width = dc.GetTextExtent( header ).GetWidth() + padding;
    m_ListOfInstances->AppendTextColumn( header, wxDATAVIEW_CELL_INERT, width );

    header = _( "Value" );
    width = dc.GetTextExtent( header ).GetWidth() + padding;
    m_ListOfInstances->AppendTextColumn( header, wxDATAVIEW_CELL_INERT, width );

    m_componentViewOld->SetLayoutDirection( wxLayout_LeftToRight );
    m_componentViewNew->SetLayoutDirection( wxLayout_LeftToRight );

    Layout();
    SetSizeInDU( 280, 240 );

    // Make sure the HTML window is large enough. Some fun size juggling and
    // fudge factors here but it does seem to work pretty reliably.
    auto info_size = m_htmlPrompt->GetTextExtent( info );
    auto prompt_size = m_htmlPrompt->GetSize();
    auto font_size = m_htmlPrompt->GetTextExtent( "X" );
    auto approx_info_height = ( 2 * info_size.x / prompt_size.x ) * font_size.y;
    m_htmlPrompt->SetSizeHints( 2 * prompt_size.x / 3, approx_info_height );
    Layout();
    GetSizer()->SetSizeHints( this );
    SetSizeInDU( 280, 240 );
    Center();
}


DIALOG_RESCUE_EACH::~DIALOG_RESCUE_EACH()
{
}


bool DIALOG_RESCUE_EACH::TransferDataToWindow()
{
    if( !wxDialog::TransferDataToWindow() )
        return false;

    PopulateConflictList();
    PopulateInstanceList();

    if( !m_AskShowAgain )
        m_btnNeverShowAgain->Hide();

    return true;
}


void DIALOG_RESCUE_EACH::PopulateConflictList()
{
    wxVector<wxVariant> data;
    for( RESCUE_CANDIDATE& each_candidate : m_Rescuer->m_all_candidates )
    {
        data.clear();
        data.push_back( wxVariant( true ) );
        data.push_back( each_candidate.GetRequestedName() );
        data.push_back( each_candidate.GetActionDescription() );

        m_ListOfConflicts->AppendItem( data );
    }

    if( !m_Rescuer->m_all_candidates.empty() )
    {
        // Select the first choice
        m_ListOfConflicts->SelectRow( 0 );
    }
}


void DIALOG_RESCUE_EACH::PopulateInstanceList()
{
    m_ListOfInstances->DeleteAllItems();

    int row = m_ListOfConflicts->GetSelectedRow();

    if( row == wxNOT_FOUND )
        row = 0;

    RESCUE_CANDIDATE& selected_part = m_Rescuer->m_all_candidates[row];

    wxVector<wxVariant> data;
    int count = 0;

    for( SCH_COMPONENT* each_component : *m_Rescuer->GetComponents() )
    {
        if( each_component->GetLibId().Format() != UTF8( selected_part.GetRequestedName() ) )
            continue;

        SCH_FIELD* valueField = each_component->GetField( 1 );

        data.clear();
        data.push_back( each_component->GetRef( & m_Parent->GetCurrentSheet() ) );
        data.push_back( valueField ? valueField->GetText() : wxT( "" ) );
        m_ListOfInstances->AppendItem( data );
        count++;
    }

    m_titleInstances->SetLabelText( wxString::Format(
                    _( "Instances of this symbol (%d items):" ), count ) );
}


void DIALOG_RESCUE_EACH::OnHandleCachePreviewRepaint( wxPaintEvent& aRepaintEvent )
{
    int row = m_ListOfConflicts->GetSelectedRow();

    if( row == wxNOT_FOUND )
        row = 0;

    RESCUE_CANDIDATE& selected_part = m_Rescuer->m_all_candidates[row];

    if( selected_part.GetCacheCandidate() )
        renderPreview( selected_part.GetCacheCandidate(), 0, m_componentViewOld );
}


void DIALOG_RESCUE_EACH::OnHandleLibraryPreviewRepaint( wxPaintEvent& aRepaintEvent )
{
    int row = m_ListOfConflicts->GetSelectedRow();

    if( row == wxNOT_FOUND )
        row = 0;

    RESCUE_CANDIDATE& selected_part = m_Rescuer->m_all_candidates[row];

    if( selected_part.GetLibCandidate() )
        renderPreview( selected_part.GetLibCandidate(), 0, m_componentViewNew );
}


// Render the preview in our m_componentView. If this gets more complicated, we should
// probably have a derived class from wxPanel; but this keeps things local.
// Call it only from a Paint Event, because we are using a wxPaintDC to draw the component
void DIALOG_RESCUE_EACH::renderPreview( LIB_PART* aComponent, int aUnit, wxPanel* aPanel )
{
    wxPaintDC dc( aPanel );
    wxColour bgColor = m_Parent->GetDrawBgColor().ToColour();

    dc.SetBackground( wxBrush( bgColor ) );
    dc.Clear();

    if( aComponent == NULL )
        return;

    if( aUnit <= 0 )
        aUnit = 1;

    const wxSize dc_size = dc.GetSize();
    dc.SetDeviceOrigin( dc_size.x / 2, dc_size.y / 2 );

    // Find joint bounding box for everything we are about to draw.
    EDA_RECT bBox = aComponent->GetUnitBoundingBox( aUnit, /* deMorganConvert */ 1 );
    const double xscale = (double) dc_size.x / bBox.GetWidth();
    const double yscale = (double) dc_size.y / bBox.GetHeight();
    const double scale  = std::min( xscale, yscale ) * 0.85;

    dc.SetUserScale( scale, scale );

    wxPoint offset = - bBox.Centre();

    // Avoid rendering when either dimension is zero
    int width, height;

    dc.GetSize( &width, &height );
    if( !width || !height )
        return;

    aComponent->Draw( NULL, &dc, offset, aUnit, 1, PART_DRAW_OPTIONS::Default() );
}


void DIALOG_RESCUE_EACH::OnConflictSelect( wxDataViewEvent& aEvent )
{
    // wxformbuilder connects this event to the _dialog_, not the data view.
    // Make sure the correct item triggered it, otherwise we trigger recursively
    // and get a stack overflow.
    if( aEvent.GetEventObject() != m_ListOfConflicts )
        return;

    PopulateInstanceList();

    m_componentViewOld->Refresh();
    m_componentViewNew->Refresh();
}


bool DIALOG_RESCUE_EACH::TransferDataFromWindow()
{
    if( !wxDialog::TransferDataFromWindow() )
        return false;

    for( size_t index = 0; index < m_Rescuer->GetCandidateCount(); ++index )
    {
        wxVariant val;
        m_ListOfConflicts->GetValue( val, index, 0 );
        bool rescue_part = val.GetBool();

        if( rescue_part )
            m_Rescuer->m_chosen_candidates.push_back( &m_Rescuer->m_all_candidates[index] );
    }
    return true;
}


void DIALOG_RESCUE_EACH::OnNeverShowClick( wxCommandEvent& aEvent )
{
    wxMessageDialog dlg( m_Parent,
                _(  "Stop showing this tool?\n"
                    "No changes will be made.\n\n"
                    "This setting can be changed from the \"Symbol Libraries\" dialog,\n"
                    "and the tool can be activated manually from the \"Tools\" menu." ),
            _( "Rescue Symbols" ), wxYES_NO | wxNO_DEFAULT | wxICON_QUESTION );
    int resp = dlg.ShowModal ();

    if( resp == wxID_YES )
    {
        m_Config->Write( RescueNeverShowEntry, true );
        m_Rescuer->m_chosen_candidates.clear();
        Close();
    }
}


void DIALOG_RESCUE_EACH::OnCancelClick( wxCommandEvent& aEvent )
{
    m_Rescuer->m_chosen_candidates.clear();
    DIALOG_RESCUE_EACH_BASE::OnCancelClick( aEvent );
}


int InvokeDialogRescueEach( SCH_EDIT_FRAME* aCaller, RESCUER& aRescuer, bool aAskShowAgain )
{
    DIALOG_RESCUE_EACH dlg( aCaller, aRescuer, aAskShowAgain );
    return dlg.ShowModal();
}
