/***********************************************************************************

    Copyright (C) 2007-2024 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifndef LIFEOGRAPH_WIDGET_FILTER_HEADER
#define LIFEOGRAPH_WIDGET_FILTER_HEADER


#include <gtkmm.h>

#include "../diaryelements/diarydata.hpp"
#include "../diaryelements/entry.hpp"
#include "../diaryelements/filter.hpp"
#include "src/helpers.hpp"
#include "src/widgets/widget_selector.hpp"
#include "widget_datepicker.hpp"
#include "widget_entrypicker.hpp"
#include "widget_picker.hpp"


namespace LIFEO
{

// FORWARD DECLARATIONS
class WidgetFilterPicker;

// UI FOR EDITING A FILTERER STACK =================================================================
class FiltererUI : public Gtk::Box
{
    public:
        FiltererUI( FiltererContainer*, Filterer* );
        virtual ~FiltererUI() {}

        Gtk::ToggleButton*      m_TB_not;

    protected:
};

class FiltererStatusUI : public FiltererStatus, public FiltererUI
{
    public:
        static Ustring          get_label() { return _( "Status" ); }

        FiltererStatusUI( Diary*, FiltererContainer*, ElemStatus = STATUS_DEFAULT );

        void                    update_state();

    protected:
        Gtk::ToggleButton*      m_ChB_not_todo;
        Gtk::ToggleButton*      m_ChB_open;
        Gtk::ToggleButton*      m_ChB_progressed;
        Gtk::ToggleButton*      m_ChB_done;
        Gtk::ToggleButton*      m_ChB_canceled;
};

class FiltererSizeUI : public FiltererSize, public FiltererUI
{
    public:
        static Ustring          get_label() { return _( "Size" ); }

        FiltererSizeUI( Diary*, FiltererContainer*, char = VT::SO::CHAR_COUNT::C,
                        double = -1, bool = false, double = -1, bool = false );

    protected:
        Gtk::DropDown*          m_DD_type;
        EntryClear*             m_E_range_b;
        EntryClear*             m_E_range_e;
        Gtk::Label*             m_L_incl_b;
        Gtk::Label*             m_L_incl_e;
};

class FiltererFavoriteUI : public FiltererFavorite, public FiltererUI
{
    public:
        static Ustring          get_label() { return _( "Favorite" ); }

        FiltererFavoriteUI( Diary*, FiltererContainer* );

    protected:
};

class FiltererTrashedUI : public FiltererTrashed, public FiltererUI
{
    public:
        static Ustring          get_label() { return _( "Trashed" ); }

        FiltererTrashedUI( Diary*, FiltererContainer* );

    protected:
};

class FiltererUnitUI : public FiltererUnit, public FiltererUI
{
    public:
        static Ustring          get_label() { return _( "Has Unit" ); }

        FiltererUnitUI( Diary*, FiltererContainer*, const Ustring& unit = "");

    protected:
        EntryClear*             m_E_unit;
};

class FiltererIsUI : public FiltererIs, public FiltererUI
{
    public:
        static Ustring          get_label() { return _( "Is Entry" ); }

        FiltererIsUI( Diary*, FiltererContainer*, DEID = DEID_UNSET );

    protected:
        WidgetEntryPicker*      m_W_entry_picker;
};

class FiltererDescendantOfUI : public FiltererDescendantOf, public FiltererUI
{
    public:
        static Ustring          get_label() { return _( "Descendant" ); }

        FiltererDescendantOfUI( Diary*, FiltererContainer*, DEID = DEID_UNSET,
                                bool = false );

    protected:
        WidgetEntryPicker*      m_W_entry_picker;
        Gtk::CheckButton*       m_ChB_or_itself;
};

class FiltererTaggedByUI : public FiltererTaggedBy, public FiltererUI
{
    public:
        static Ustring          get_label() { return _( "Has Tag" ); }

        FiltererTaggedByUI( Diary*, FiltererContainer*, Entry* = nullptr );

    protected:
        WidgetEntryPicker*      m_W_entry_picker;
};

class FiltererSubtaggedByUI : public FiltererSubtaggedBy, public FiltererUI
{
    public:
        static Ustring          get_label() { return _( "Has SubTag" ); }

        FiltererSubtaggedByUI( Diary*, FiltererContainer*,
                               Entry* = nullptr, Entry* = nullptr,
                               char = '=', int = VT::LAST );

    protected:
        WidgetEntryPicker*      m_WEP_tag_p;
        WidgetEntryPicker*      m_WEP_tag_c;
        Gtk::ComboBoxText*      m_CB_type;
        Gtk::DropDown*          m_DD_rel;
};

class FiltererTagValueUI : public FiltererTagValue, public FiltererUI
{
    public:
        static Ustring          get_label() { return _( "Tag Value" ); }

        FiltererTagValueUI( Diary*, FiltererContainer*, Entry* = nullptr, int = VT::REALIZED,
                            double = Constants::INFINITY_MNS, bool = false,
                            double = Constants::INFINITY_PLS, bool = false );

    protected:
        WidgetEntryPicker*      m_W_entry_picker;
        Gtk::ComboBoxText*      m_CB_type;
        EntryClear*             m_E_range_b;
        EntryClear*             m_E_range_e;
        Gtk::Label*             m_L_incl_b;
        Gtk::Label*             m_L_incl_e;
};

class FiltererThemeUI : public FiltererTheme, public FiltererUI
{
    public:
        static Ustring          get_label() { return _( "Has Theme" ); }

        FiltererThemeUI( Diary*, FiltererContainer*, Theme* = nullptr );

    protected:
        WidgetPicker< Theme >*  m_W_theme_picker;
};

class FiltererBetweenDatesUI : public FiltererBetweenDates, public FiltererUI
{
    public:
        static Ustring          get_label() { return _( "Between Dates" ); }

        FiltererBetweenDatesUI( Diary*, FiltererContainer*,
                                DateV = Date::NOT_SET, bool = false,
                                DateV = Date::NOT_SET, bool = false );

    protected:
        WidgetDatePicker*       m_W_date_picker_b;
        WidgetDatePicker*       m_W_date_picker_e;
        Gtk::Label*             m_L_incl_b;
        Gtk::Label*             m_L_incl_e;
};

class FiltererBetweenEntriesUI : public FiltererBetweenEntries, public FiltererUI
{
    public:
        static Ustring          get_label() { return _( "Between Entries" ); }

        FiltererBetweenEntriesUI( Diary*, FiltererContainer*, Entry* = nullptr, bool = false,
                                                              Entry* = nullptr, bool = false );

    protected:
        WidgetEntryPicker*      m_WEP_bgn;
        WidgetEntryPicker*      m_WEP_end;
        Gtk::Label*             m_L_incl_b;
        Gtk::Label*             m_L_incl_e;
};

class FiltererCompletionUI : public FiltererCompletion, public FiltererUI
{
    public:
        static Ustring          get_label() { return _( "Completion" ); }

        FiltererCompletionUI( Diary*, FiltererContainer*, double = 0.0, double = 100.0 );

    protected:
        Gtk::Scale*             m_Sc_bgn;
        Gtk::Scale*             m_Sc_end;
};

class FiltererContainsTextUI : public FiltererContainsText, public FiltererUI
{
    public:
        static Ustring          get_label() { return _( "Contains Text" ); }

        FiltererContainsTextUI( Diary*, FiltererContainer*, const Ustring& = "",
                                                            bool = false, bool = false );

    protected:
        EntryClear*             m_E_text;
        Gtk::ToggleButton*      m_TB_match_case;
        Gtk::ComboBoxText*      m_CB_extent;
};

class FiltererChildFilterUI : public FiltererChildFilter, public FiltererUI
{
    public:
        static Ustring          get_label() { return _( "Child Filter" ); }

        FiltererChildFilterUI( Diary*, FiltererContainer*, Filter* = nullptr );

    protected:
        WidgetFilterPicker*     m_WFP_filter;
};

class FiltererIsImageUI : public FiltererIsImage, public FiltererUI
{
    public:
        static Ustring          get_label() { return _( "Is Image" ); }

        FiltererIsImageUI( Diary*, FiltererContainer* );

    protected:
};

class FiltererHasCoordsUI : public FiltererHasCoords, public FiltererUI
{
    public:
        static Ustring          get_label() { return _( "Has Coordinates" ); }

        FiltererHasCoordsUI( Diary*, FiltererContainer* );

    protected:
};

class FiltererTitleStyleUI : public FiltererTitleStyle, public FiltererUI
{
    public:
        static Ustring          get_label() { return _( "Title Style" ); }

        FiltererTitleStyleUI( Diary*, FiltererContainer*, char = VT::ETS::NAME_ONLY::C );

    protected:
        Gtk::DropDown*          m_DD_title_style;
};

class FiltererHeadingLevelUI : public FiltererHeadingLevel, public FiltererUI
{
    public:
        static Ustring          get_label() { return _( "Heading Level" ); }

        FiltererHeadingLevelUI( Diary*, FiltererContainer*, int = ( 1 << 1 ) );

    protected:
        void                    handle_change( int, bool );

        Gtk::ToggleButton*      m_TB_H0;
        Gtk::ToggleButton*      m_TB_H1;
        Gtk::ToggleButton*      m_TB_H2;
        Gtk::ToggleButton*      m_TB_H3;
};

// BASIC FILTERERS =================================================================================
class FiltererEqualsUI : public FiltererEquals, public FiltererUI
{
    public:
        static Ustring          get_label() { return _( "Equals" ); }

        FiltererEqualsUI( Diary*, FiltererContainer*, char = '=', double = 0.0 );

    protected:
        Gtk::DropDown*          m_DD_relation;
        Gtk::Entry*             m_E_value;
};

// WIDGET FILTER ===================================================================================
class WidgetFilter : public FiltererContainer, public FiltererUI
{
    public:
        static Ustring          get_label() { return _( "Subgroup" ); }

        WidgetFilter( Diary* diary, WidgetFilter* ctr, int objects = FOC::NOTHING )
        :   FiltererContainer( diary, ctr, ctr && !ctr->is_or() ), FiltererUI( ctr, this ),
            m_target_obj_classes( ctr ? ( ctr->m_target_obj_classes & FOC::ALL_REAL ) : objects )
        { init(); }
        // ~WidgetFilter();

        void                    clear_pipeline() override;

        void                    set_from_string( const Ustring& str ) override
        {
            m_F_emit_signals = false;
            FiltererContainer::set_from_string( str );
            m_F_emit_signals = true;
        }

        void                    set_last_filterer_not() override
        {
            FiltererContainer::set_last_filterer_not();
            Lifeograph::START_INTERNAL_OPERATIONS();
            dynamic_cast< FiltererUI* >( m_pipeline.back() )->m_TB_not->set_active( true );
            Lifeograph::FINISH_INTERNAL_OPERATIONS();
        }

        void                    toggle_logic() override;
        void                    update_logic_label() override;

        void                    add_filterer_status( ElemStatus es ) override
        { add_filterer< FiltererStatusUI, ElemStatus >( es ); }
        void                    add_filterer_size( int type,
                                                   double range_b, bool f_incl_b,
                                                   double range_e, bool f_incl_e ) override
        { add_filterer< FiltererSizeUI, int, double, bool, double, bool >(
                type, range_b, f_incl_b, range_e, f_incl_e ); }

        void                    add_filterer_favorite() override
        { add_filterer< FiltererFavoriteUI >(); }
        void                    add_filterer_trashed() override
        { add_filterer< FiltererTrashedUI >(); }
        void                    add_filterer_unit( const Ustring& unit ) override
        { add_filterer< FiltererUnitUI, const Ustring& >( unit ); }
        void                    add_filterer_is( DEID id ) override
        { add_filterer< FiltererIsUI, DEID >( id ); }
        void                    add_filterer_descendant_of( DEID id, bool F_or_itself ) override
        { add_filterer< FiltererDescendantOfUI, DEID, bool >( id, F_or_itself ); }
        void                    add_filterer_tagged_by( Entry* tag ) override
        { add_filterer< FiltererTaggedByUI, Entry* >( tag ); }
        void                    add_filterer_tag_value( Entry* tag, int type,
                                                        double range_b, bool f_incl_b,
                                                        double range_e, bool f_incl_e ) override
        { add_filterer< FiltererTagValueUI, Entry*, int, double, bool, double, bool >(
                tag, type, range_b, f_incl_b, range_e, f_incl_e ); }
        void                    add_filterer_subtagged_by( Entry* tag_p, Entry* tag_c,
                                                           char rel, int type ) override
        { add_filterer< FiltererSubtaggedByUI, Entry*, Entry*, char, int >(
                tag_p, tag_c, rel, type ); }
        void                    add_filterer_theme( Theme* theme ) override
        { add_filterer< FiltererThemeUI, Theme* >( theme ); }
        void                    add_filterer_between_dates( DateV date_b, bool f_incl_b,
                                                            DateV date_e, bool f_incl_e ) override
        { add_filterer< FiltererBetweenDatesUI, DateV, bool, DateV, bool >(
                date_b, f_incl_b, date_e, f_incl_e ); }

        void                    add_filterer_between_entries( Entry* entry_b, bool f_incl_b,
                                                          Entry* entry_e, bool f_incl_e ) override
        { add_filterer< FiltererBetweenEntriesUI, Entry*, bool, Entry*, bool >(
                entry_b, f_incl_b, entry_e, f_incl_e ); }

        void                    add_filterer_completion( double bgn, double end ) override
        { add_filterer< FiltererCompletionUI, double, double >( bgn, end ); }

        void                    add_filterer_contains_text( const Ustring& text,
                                                            bool cs, bool no ) override
        { add_filterer< FiltererContainsTextUI, const Ustring&, bool, bool >( text, cs, no ); }

        void                    add_filterer_child_filter( Filter* filter ) override
        { add_filterer< FiltererChildFilterUI, Filter* >( filter ); }

        void                    add_filterer_is_image() override
        { add_filterer< FiltererIsImageUI >(); }

        void                    add_filterer_has_coords() override
        { add_filterer< FiltererHasCoordsUI >(); }

        void                    add_filterer_title_style( char title_style ) override
        { add_filterer< FiltererTitleStyleUI >( title_style ); }

        void                    add_filterer_heading_level( int h_levels ) override
        { add_filterer< FiltererHeadingLevelUI >( h_levels ); }

        void                    add_filterer_equals( char relation, double value ) override
        { add_filterer< FiltererEqualsUI, double >( relation, value ); }

        FiltererContainer*      add_filterer_subgroup() override
        { return add_filterer< WidgetFilter >(); }

        void                    remove_filterer( Filterer* ) override;

        void                    update_add_menu();

        void                    update_state() override;

        void                    set_enabled( bool );

        SignalVoid              signal_changed()
        { return m_Sg_changed; }

        void                    set_target_obj_classes( int objcls )
        {
            m_target_obj_classes = objcls;
            update_add_menu();
        }

    protected:
        template< class T >
        void                    add_filterer_to_menu()
        {
            // if( !( T::m_obj_classes & m_obj_classes ) ) return;
            // NOTE:do not restrict, allow interchangability

            if( T::m_obj_classes & m_target_obj_classes )
                m_WS_add->insert_elem( T::get_label(),
                                       [ this ]( int ){ add_filterer< T >(); } );
            else // denote that it will be unused in the current context:
                m_WS_add->insert_elem( "*" + T::get_label(),
                                       [ this ]( int ){ add_filterer< T >(); } );
        }

        template< class T, typename... Args >
        T*                      add_filterer( Args... args )
        {
            T* filterer = Gtk::manage( new T( m_p2diary, this, args... ) );
            m_pipeline.push_back( filterer );
            filterer->insert_before( *m_Bx_contents, *m_WS_add );
            if( m_F_emit_signals )
                update_state();
            return filterer;
        }

        void                    init();

        Gtk::Box*               m_Bx_header;
        Gtk::Box*               m_Bx_contents;
        Gtk::Label*             m_L_logic;
        WidgetSelector*         m_WS_add;

        int                     m_target_obj_classes;
        // this flag is used to disable signals during internal updates
        bool                    m_F_emit_signals  { true };

        SignalVoid              m_Sg_changed;
};

} // end of namespace LIFEO

#endif
