/*
 * MIT License
 *
 * Copyright (c) 2002-2021 Mikko Tommila
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package org.apfloat.spi;

import org.apfloat.ApfloatRuntimeException;

/**
 * Number Theoretic Transform (NTT) strategy.
 * An implementing class could be e.g.
 *
 * <ul>
 *   <li>Fast Number Theoretic Transform (FNT)</li>
 *   <li>"Four-step" FNT</li>
 *   <li>"Two-pass" mass storage FNT</li>
 *   <li>Winograd Fourier Transform Algorithm (WFTA)</li>
 * </ul>
 * <p>
 *
 * Note: an implementing transformation class is required only to be able
 * to perform an inverse transform on data transformed by the same class,
 * not by any other class.
 *
 * @version 1.0
 * @author Mikko Tommila
 */

public interface NTTStrategy
{
    /**
     * Perform a forward transform on the data.<p>
     *
     * Multiple moduli can be used, if the convolution algorithm
     * uses the Chinese Remainder Theorem to calculate the final
     * result.
     *
     * @param dataStorage The data to be transformed.
     * @param modulus Number of modulus to use (in case the transform supports multiple moduli).
     */

    public void transform(DataStorage dataStorage, int modulus)
        throws ApfloatRuntimeException;

    /**
     * Perform an inverse transform on the data.<p>
     *
     * Multiple moduli can be used, if the convolution algorithm
     * uses the Chinese Remainder Theorem to calculate the final
     * result.
     *
     * @param dataStorage The data to be transformed.
     * @param modulus Number of modulus to use (in case the transform supports multiple moduli).
     * @param totalTransformLength Total transform length; the final result elements are divided by this value.
     */

    public void inverseTransform(DataStorage dataStorage, int modulus, long totalTransformLength)
        throws ApfloatRuntimeException;

    /**
     * Return the supported transform length for the specified data size.
     *
     * @param size Length of the data to be transformed.
     *
     * @return Length of the transform needed by this transform.
     */

    public long getTransformLength(long size);
}
