/*--------------------------------------------------------------------------+
$Id: StringUndoStackBaseTest.java 26283 2010-02-18 11:18:57Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.string;

import static edu.tum.cs.commons.string.StringUtils.CR;
import static edu.tum.cs.commons.string.StringUtils.EMPTY_STRING;
import junit.framework.TestCase;

/**
 * Test for {@link StringUndoStackBaseTest}.
 * 
 * @see UndoStack
 * 
 * @author deissenb
 * @author $Author: juergens $
 * @version $Rev: 26283 $
 * @levd.rating GREEN Hash: 36AF0FC20BEA721E9BC87D5737798CFB
 */
public class StringUndoStackBaseTest extends TestCase {
	/** Stack under test. */
	private UndoStack stack;

	/** Test initialization. */
	public void testInit() {
		stack = new UndoStack("Test");
		checkStackState(false, false, "Test", 0);
	}

	/** Test initialization with empty string. */
	public void testInitEmpty() {
		stack = new UndoStack(EMPTY_STRING);
		checkStackState(false, false, EMPTY_STRING, 0);
	}

	/** Test adding to stack. */
	public void testAdd() {
		testInit();
		stack.addLine("Test2");
		checkStackState(true, false, "Test" + CR + "Test2", 1);
	}

	/** Test undo. */
	public void testUndo() {
		testAdd();
		stack.undo();
		checkStackState(false, true, "Test", 2);
	}

	/** Test redo. */
	public void testRedo() {
		testUndo();
		stack.redo();
		checkStackState(true, false, "Test" + CR + "Test2", 3);
	}

	/** Test multiple adds followed by undos and redos. */
	public void testMultipleOperations() {

		int n = 100;

		String[] lines = StringUtils.generateStringArray(n, 100, 1);

		stack = new UndoStack(EMPTY_STRING);
		for (int i = 0; i < n; i++) {
			stack.addLine(lines[i]);
			checkStackState(true, false, concat(lines, i + 1), i + 1);
		}

		for (int i = lines.length; i > 0; i--) {
			stack.undo();
			checkStackState(i > 1, true, concat(lines, i - 1), 2 * n - i + 1);
		}

		for (int i = 0; i < lines.length; i++) {
			stack.redo();
			checkStackState(true, i < n - 1, concat(lines, i + 1), n * 2 + i
					+ 1);
		}
	}

	/**
	 * Concatenate the first i lines of the array using the line terminator as
	 * separator.
	 */
	private static String concat(String[] lines, int i) {
		String[] result = new String[i];
		System.arraycopy(lines, 0, result, 0, i);
		return StringUtils.concat(result, CR);
	}

	/**
	 * Check state of the stack.
	 * 
	 * @param canUndo
	 *            expected value for {@link StringUndoStackBase#canUndo()}
	 * @param canRedo
	 *            expected value for {@link StringUndoStackBase#canRedo()}
	 * @param model
	 *            expected stack content
	 * @param changeCount
	 *            expected number of changes to the stack
	 */
	private void checkStackState(boolean canUndo, boolean canRedo,
			String model, int changeCount) {
		assertEquals(canUndo, stack.canUndo());
		assertEquals(canRedo, stack.canRedo());
		assertEquals(model, stack.getModel());
		assertEquals(changeCount, stack.getStackChangeCount());
	}
}