/*--------------------------------------------------------------------------+
$Id: PerformanceMonitorTest.java 26283 2010-02-18 11:18:57Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.system;

import java.util.ArrayList;

import junit.framework.TestCase;

/**
 * Test for {@link PerformanceMonitor}
 * <p>
 * Makes sure that {@link PerformanceMonitor#start()} can only be called in
 * state {@link EPerformanceMonitorState#NOT_RUN} and that
 * {@link PerformanceMonitor#stop()} can only be called in state
 * {@link EPerformanceMonitorState#RUNNING}.
 * 
 * @author Elmar Juergens
 * @author $Author: juergens $
 * 
 * @version $Revision: 26283 $
 * @levd.rating GREEN Hash: 34BAFEF03C515478FF50DE6D2620DDBA
 * 
 */
public class PerformanceMonitorTest extends TestCase {

	/** {@link PerformanceMonitor} under test */
	PerformanceMonitor monitor;

	/** Starts the performance monitor under test */
	@Override
	protected void setUp() throws Exception {
		super.setUp();

		// Use factory method to test that as well
		monitor = PerformanceMonitor.create();
	}

	/**
	 * Makes sure that the {@link PerformanceMonitor} doesn't return entirely
	 * nonsense values
	 */
	public void testSmoke() {
		// Create some objects to consume space and time
		for (int i = 0; i < 100000; i++) {
			new ArrayList<String>(100);
		}

		// stop monitor
		monitor.stop();

		// perform sanity checks on measured values
		assertTrue(monitor.getMaxMemUsageInBytes() > 0);
		assertTrue(monitor.getMilliseconds() > 0);

		assertTrue(monitor.getMaxMemUsageInBytes() > monitor
				.getMaxMemUsageInKBs());
		assertTrue(monitor.getMilliseconds() > monitor.getSeconds());
		
		assertTrue(monitor.getDeltaMemUsageInBytes() >= 0);
		assertTrue(monitor.getDeltaMemUsageInKBs() >= 0);
		assertTrue(monitor.getDeltaMemUsageInBytes() <= monitor.getMaxMemUsageInBytes());
		assertTrue(monitor.getDeltaMemUsageInKBs() <= monitor.getMaxMemUsageInKBs());
	}

	/**
	 * Makes sure that start cannot be called in state
	 * {@link EPerformanceMonitorState#RUNNING}
	 */
	public void testCannotStartTwice() {
		try {
			// start for the second time (first in setup)
			monitor.start();
			fail("It should not be possible to start a monitor twice");
		} catch (IllegalStateException e) {
			// Nothing to do
		}
	}

	/**
	 * Makes sure that stop cannot be called in state
	 * {@link EPerformanceMonitorState#NOT_RUN}
	 */
	public void testCannotStopBeforeStart() {
		try {
			// stop before starting
			new PerformanceMonitor(false).stop();
			fail("It should not be possible to stop the PerformanceMonitor before it gets started");
		} catch (IllegalStateException e) {
			// Nothing to do
		}
	}

	/**
	 * Makes sure that stop cannot be called in state
	 * {@link EPerformanceMonitorState#FINISHED}
	 */
	public void testCannotStopTwice() {
		try {
			// stop before starting
			monitor.stop();
			monitor.stop();
			fail("It should not be possible to stop the PerformanceMonitor more than once");
		} catch (IllegalStateException e) {
			// Nothing to do
		}
	}

	/**
	 * Makes sure that start cannot be called in state
	 * {@link EPerformanceMonitorState#FINISHED}
	 */
	public void testCannotRestartPerformanceMonitor() {
		try {
			// stop before starting
			monitor.stop();
			monitor.start();
			fail("It should not be possible to restart the PerformanceMonitor more than once");
		} catch (IllegalStateException e) {
			// Nothing to do
		}
	}

}