/*
 * @(#)PostCompileClass.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.compiler;

import java.io.IOException;
import java.io.OutputStream;

import net.sourceforge.groboutils.codecoverage.v2.IAnalysisModule;
import net.sourceforge.groboutils.codecoverage.v2.datastore.AnalysisModuleSet;
import net.sourceforge.groboutils.codecoverage.v2.datastore.ClassRecord;
import net.sourceforge.groboutils.codecoverage.v2.datastore.IClassMetaDataWriter;
import net.sourceforge.groboutils.codecoverage.v2.datastore.IMetaDataWriter;


/**
 * This is the main entry-point for performing the post-compilation of
 * class files.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:25 $
 * @since     December 16, 2002
 */
public class PostCompileClass
{
    private IMetaDataWriter out;
    private AnalysisModuleSet amSet;
    private ParseCoverageLogger pcl;
    
    
    /**
     * @param writer meta-data output for the analysis generated for
     *    class post-compiled; must not be <tt>null</tt>.
     * @param modules a list of all analysis modules to use when post-compiling
     *    each class; must not be <tt>null</tt>.
     * @exception IllegalArgumentException if either <tt>writer</tt> or
     *    <tt>modules</tt> is <tt>null</tt>.
     */
    public PostCompileClass( IMetaDataWriter writer,
            IAnalysisModule[] modules )
    {
        this( new ParseCoverageLogger(), writer, modules );
    }
    
    
    /**
     * @param pcl the parser for a specific CoverageLogger (may be null).
     * @param writer meta-data output for the analysis generated for
     *    class post-compiled; must not be <tt>null</tt>.
     * @param modules a list of all analysis modules to use when post-compiling
     *    each class; must not be <tt>null</tt>.
     * @exception IllegalArgumentException if either <tt>writer</tt> or
     *    <tt>modules</tt> is <tt>null</tt>.
     *
     * @since December 28, 2002
     */
    public PostCompileClass( ParseCoverageLogger pcl, IMetaDataWriter writer,
            IAnalysisModule[] modules )
    {
        if (writer == null || modules == null || pcl == null)
        {
            throw new IllegalArgumentException("no null args");
        }
        
        this.out = writer;
        this.amSet = new AnalysisModuleSet( modules );
        this.pcl = pcl;
    }
    
    
    /**
     * Performs the post-compilation for the given class file for
     * all the internal analysis modules, and will write the new class file to
     * the given stream.
     */
    public void postCompile( String filename, byte[] classFile,
            OutputStream classOut )
            throws IOException
    {
        if (this.pcl == null)
        {
            throw new IllegalStateException( "pcl is null." );
        }
        ModifiedClass mc = new ModifiedClass( this.pcl, filename, classFile );
        
        int amCount = amSet.getAnalysisModuleCount();
        ModifiedMethod[] mmL = mc.getMethods();
        for (short moduleIndex = 0; moduleIndex < amCount; ++moduleIndex)
        {
            IAnalysisModule am = this.amSet.getAnalysisModuleAt( moduleIndex );
            ClassRecord cr = mc.createClassRecord( this.amSet );
            
            for (int methodI = 0; methodI < mmL.length; ++methodI)
            {
                DefaultMethodCode dmc = new DefaultMethodCode( moduleIndex,
                    mmL[ methodI ], cr );
                am.analyze( dmc );
            }
            
            // record this class record
            IClassMetaDataWriter cmdw = this.out.getClassWriter( am );
            try
            {
                cmdw.writeClassRecord( cr );
            }
            finally
            {
                cmdw.close();
            }
        }
        
        byte[] modClassFile = mc.getModifiedClass();
        classOut.write( modClassFile );
    }
}

