/*
 * @(#)ExtensionFilenameFilterUTest.java
 *
 * Copyright (C) 2001,,2003 2002 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.util.io.v1;

import java.io.File;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

/**
 * 
 *
 * @author   Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @since    0.9.1d Alpha (sometime in 2001)
 * @version  $Date: 2003/02/10 22:52:45 $
 */
public class ExtensionFilenameFilterUTest extends TestCase
{
    //-------------------------------------------------------------------------
    // Global constructs that only depend upon the class name
    private static final Class THIS_CLASS = ExtensionFilenameFilterUTest.class;
    
    public ExtensionFilenameFilterUTest( String name )
    {
        super( name );
    }
    
    
    //-------------------------------------------------------------------------
    // Test methods
    
    public void testInstantiate()
    {
        createFilter();
        createFilter( "" );
        createFilter( "a" );
        createFilter( ".a" );
        createFilter( ".A" );
        createFilter( true );
        createFilter( false );
        createFilter( new String[] { "" } );
        createFilter( new String[] { "B", "A" } );
        createFilter( "", true );
        createFilter( "", false );
        createFilter( "A", true );
        createFilter( "A", false );
        createFilter( new String[] { "A", "a" }, true );
        createFilter( new String[] { "A", "a" }, false );
    }

    public void testInstantiateBad1()
    {
        try
        {
            createFilter( (String)null );
            fail("did not throw an IllegalArgumentException." );
        }
        catch (IllegalArgumentException iae)
        { }
    }
    
    public void testInstantiateBad2()
    {
        try
        {
            createFilter( (String[])null );
            fail("did not throw an IllegalArgumentException." );
        }
        catch (IllegalArgumentException iae)
        { }
    }
    
    public void testInstantiateBad3()
    {
        try
        {
            createFilter( new String[] { null } );
            fail("did not throw an IllegalArgumentException." );
        }
        catch (IllegalArgumentException iae)
        { }
    }
    
    public void testInstantiateBad4()
    {
        try
        {
            createFilter( new String[] { "", null } );
            fail("did not throw an IllegalArgumentException." );
        }
        catch (IllegalArgumentException iae)
        { }
    }
    
    public void testInstantiateBad5()
    {
        try
        {
            createFilter( new String[] { "a", null } );
            fail("did not throw an IllegalArgumentException." );
        }
        catch (IllegalArgumentException iae)
        { }
    }
    
    public void testInstantiateBad6()
    {
        try
        {
            createFilter( (String)null, true );
            fail("did not throw an IllegalArgumentException." );
        }
        catch (IllegalArgumentException iae)
        { }
    }
    
    public void testInstantiateBad7()
    {
        try
        {
            createFilter( (String[])null, false );
            fail("did not throw an IllegalArgumentException." );
        }
        catch (IllegalArgumentException iae)
        { }
    }
    
    public void testDefaultDirectorySetting()
    {
        assertTrue(
            "Must allow directories by default.",
            createFilter().allowsDirectories() );
    }
    
    public void testNullAccept1()
    {
        try
        {
            createFilter( "a" ).accept( null, null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException e)
        { }
    }
    
    public void testNullAccept2()
    {
        try
        {
            createFilter( "a" ).accept( new File( "b" ), null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException e)
        { }
    }
    
    
    public void testEmptyAccept()
    {
        noMatchFile( createFilter(), new File( "a" ) );
    }
    
    
    public void testAcceptCaseSensitive()
    {
        matchFile( createFilter( "A", false ), new File( "2/A" ) );
        matchFile( createFilter( "A", false ), new File( "2/.A" ) );
        matchFile( createFilter( "A", false ), new File( "2/asdfA" ) );
        matchFile( createFilter( "A", false ), new File( "2/asdf.A" ) );
        matchFile( createFilter( ".a", false ), new File( "2/b.a" ) );
        matchFile( createFilter( ".a", false ), new File( "2/.a" ) );
        matchFile( createFilter( ".a", false ), new File( "2/asdf.a" ) );
        matchFile( createFilter( ".a", false ), new File( "2/asdf.a.a" ) );
        matchFile( createFilter( ".abc", false ), new File( "2/asdf.a.abc" ) );
        
        noMatchFile( createFilter( ".a", false ), new File( "" ) );
        noMatchFile( createFilter( ".a", false ), new File( "4/asdf" ) );
        noMatchFile( createFilter( ".a", false ), new File( "4/asdf.A" ) );
        noMatchFile( createFilter( ".a", false ), new File( "4/.A" ) );
        noMatchFile( createFilter( ".a", false ), new File( "4/.a.A" ) );
        noMatchFile( createFilter( ".a", false ), new File( "4/asdf.a.A" ) );
        noMatchFile( createFilter( ".a", false ), new File( "4/asdf.a." ) );
        noMatchFile( createFilter( ".ABC", false ), new File( "4/asdf.a.A" ) );
        noMatchFile( createFilter( ".ABC", false ), new File( "4/asdf.a.AB" ) );
        noMatchFile( createFilter( ".ABC", false ), new File( "4/asdf.a.AC" ) );
    }
    
    
    
    public void testAcceptCaseInsensitive()
    {
        matchFile( createFilter( "A" ), new File( "A" ) );
        matchFile( createFilter( "A" ), new File( "asdfA" ) );
        matchFile( createFilter( "A" ), new File( "asdf.A" ) );
        matchFile( createFilter( ".a" ), new File( "b.a" ) );
        matchFile( createFilter( ".a" ), new File( ".a" ) );
        matchFile( createFilter( ".a" ), new File( "asdf.a" ) );
        matchFile( createFilter( ".a" ), new File( "asdf.a.a" ) );
        matchFile( createFilter( ".a" ), new File( "asdf.A" ) );
        matchFile( createFilter( ".a" ), new File( ".A" ) );
        matchFile( createFilter( ".a" ), new File( ".a.A" ) );
        matchFile( createFilter( ".a" ), new File( "asdf.a.A" ) );

        matchFile( createFilter( "A" ), new File( "B", "A" ) );
        matchFile( createFilter( "A" ), new File( "B", "asdfA" ) );
        matchFile( createFilter( "A" ), new File( "B", "asdf.A" ) );
        matchFile( createFilter( ".a" ), new File( "a.b", "b.a" ) );
        matchFile( createFilter( ".a" ), new File( "a.b", ".a" ) );
        matchFile( createFilter( ".a" ), new File( "a.b", "asdf.a" ) );
        matchFile( createFilter( ".a" ), new File( "a.b", "asdf.a.a" ) );
        matchFile( createFilter( ".a" ), new File( "a.b", "asdf.A" ) );
        matchFile( createFilter( ".a" ), new File( "a.b", ".A" ) );
        matchFile( createFilter( ".a" ), new File( "a.b", ".a.A" ) );
        matchFile( createFilter( ".a" ), new File( "a.b", "asdf.a.A" ) );
        
        matchFile( createFilter( "A", true ), new File( "A" ) );
        matchFile( createFilter( "A", true ), new File( "asdfA" ) );
        matchFile( createFilter( "A", true ), new File( "asdf.A" ) );
        matchFile( createFilter( ".a", true ), new File( "b.a" ) );
        matchFile( createFilter( ".a", true ), new File( ".a" ) );
        matchFile( createFilter( ".a", true ), new File( "asdf.a" ) );
        matchFile( createFilter( ".a", true ), new File( "asdf.a.a" ) );
        matchFile( createFilter( ".a", true ), new File( "asdf.A" ) );
        matchFile( createFilter( ".a", true ), new File( ".A" ) );
        matchFile( createFilter( ".a", true ), new File( ".a.A" ) );
        matchFile( createFilter( ".a", true ), new File( "asdf.a.A" ) );
        
        noMatchFile( createFilter( ".a" ), new File( "" ) );
        noMatchFile( createFilter( ".a" ), new File( "asdf" ) );
        noMatchFile( createFilter( ".a" ), new File( "asdf.a." ) );
        noMatchFile( createFilter( ".a" ), new File( "asdf.aa" ) );
        noMatchFile( createFilter( ".ABC" ), new File( "asdf.a.A" ) );
        noMatchFile( createFilter( ".ABC" ), new File( "asdf.a.AB" ) );
        noMatchFile( createFilter( ".ABC" ), new File( "asdf.a.AC" ) );
        noMatchFile( createFilter( ".ABC" ), new File( "asdf.a.ACB" ) );
        noMatchFile( createFilter( ".ABC" ), new File( "asdf.a.ABCD" ) );
        noMatchFile( createFilter( ".a", true ), new File( "" ) );
        noMatchFile( createFilter( ".a", true ), new File( "asdf" ) );
        noMatchFile( createFilter( ".a", true ), new File( "asdf.a." ) );
        noMatchFile( createFilter( ".ABC", true ), new File( "asdf.a.A" ) );
        noMatchFile( createFilter( ".ABC", true ), new File( "asdf.a.AB" ) );
        noMatchFile( createFilter( ".ABC", true ), new File( "asdf.a.AC" ) );
        noMatchFile( createFilter( ".ABC", true ), new File( "asdf.a.ACB" ) );
        noMatchFile( createFilter( ".ABC", true ), new File( "asdf.a.ABCD" ) );
        
        noMatchFile( createFilter( ".a" ), new File( "a.a", "asdf" ) );
        noMatchFile( createFilter( ".a" ), new File( "a.a", "asdf.a." ) );
        noMatchFile( createFilter( ".ABC" ), new File( "a.a", "asdf.a.A" ) );
        noMatchFile( createFilter( ".ABC" ), new File( "a.a", "asdf.a.AB" ) );
        noMatchFile( createFilter( ".ABC" ), new File( "a.a", "asdf.a.AC" ) );
        noMatchFile( createFilter( ".a", true ), new File( "" ) );
        noMatchFile( createFilter( ".a", true ), new File( "a.a", "asdf" ) );
        noMatchFile( createFilter( ".a", true ), new File( "a.a", "asdf.a." ) );
        noMatchFile( createFilter( ".ABC", true ), new File( "a.ABC", "asdf.a.A" ) );
        noMatchFile( createFilter( ".ABC", true ), new File( "a.ABC", "asdf.a.AB" ) );
        noMatchFile( createFilter( ".ABC", true ), new File( "a.ABC", "asdf.a.AC" ) );
    }
    
    
    public void testAcceptDirectories()
    {
        matchFile( createFilter( "A" ), new File( "." ) );
        matchFile( createFilter( "A", false ), new File( "." ) );
        matchFile( createFilter( "A", true ), new File( "." ) );
    }
    
    
    public void testDontAcceptDirectories()
    {
        ExtensionFilenameFilter eff = createFilter( "A" );
        eff.allowsDirectories( false );
        noMatchFile( eff, new File( "." ) );
        
        eff = createFilter( "A", true );
        eff.allowsDirectories( false );
        noMatchFile( eff, new File( "." ) );

        eff = createFilter( "A", false );
        eff.allowsDirectories( false );
        noMatchFile( eff, new File( "." ) );
    }
    
    
    
    protected void matchFile( ExtensionFilenameFilter eff, File f )
    {
        File parent = getParentFile( f );
        String name = f.getName();
        assertTrue(
            "Filter must accept parent '"+parent+"' and name '"+name+"'.",
            eff.accept( parent, name ) );
    }
    
    
    protected void noMatchFile( ExtensionFilenameFilter eff, File f )
    {
        File parent = getParentFile( f );
        String name = f.getName();
        assertTrue(
            "Filter must not accept parent '"+parent+"' and name '"+name+"'.",
            !eff.accept( parent, name ) );
    }
    
    
    protected ExtensionFilenameFilter createFilter()
    {
        return new ExtensionFilenameFilter();
    }
    protected ExtensionFilenameFilter createFilter( String s )
    {
        return new ExtensionFilenameFilter( s );
    }
    protected ExtensionFilenameFilter createFilter( String s[] )
    {
        return new ExtensionFilenameFilter( s );
    }
    protected ExtensionFilenameFilter createFilter( boolean b )
    {
        return new ExtensionFilenameFilter( b );
    }
    protected ExtensionFilenameFilter createFilter( String s, boolean b )
    {
        return new ExtensionFilenameFilter( s, b );
    }
    protected ExtensionFilenameFilter createFilter( String s[], boolean b )
    {
        return new ExtensionFilenameFilter( s, b );
    }
    
    
    protected File getParentFile( File f )
    {
        String pname = f.getParent();
        if (pname == null)
        {
            // no parent
            return null;
        }
        return new File( pname );
    }
    
    
    //-------------------------------------------------------------------------
    // Global static methods that don't change.
    
    public static Test suite()
    {
        TestSuite suite = new TestSuite( THIS_CLASS );
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    protected void setUp() throws Exception
    {
        super.setUp();
        
        // set ourself up
    }
    
    
    protected void tearDown() throws Exception
    {
        // tear ourself down
        
        
        super.tearDown();
    }

}
