/*
 * nss_sshsock: NSS module for providing NSS services from a remote
 * ssh server over a SSH socket.
 *
 * Copyright (C) 2011 Scott Balneaves <sbalneav@ltsp.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <nss.h>
#include <pwd.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include "nss_sshsock.h"

/*
 * Needed for getpwent() statefulness.
 */

static FILE *pproc = NULL;

/*
 * buffer_to_pwstruct:
 *
 * Given a buffer containing a single passwd(5) line, populate a
 * password struct.  Note that the ':' in the buffer are converted
 * to '\0' by the call to split.
 */

static enum nss_status
buffer_to_pwstruct (struct passwd *pwstruct, char *buffer)
{
  char **array;
  size_t count;

  if (!pwstruct)
    {
      return NSS_STATUS_UNAVAIL;
    }

  /*
   * split on ':'
   */

  array = split (buffer, ":");

  if (!array)
    {
      /* couldn't split on the buffer */
      return NSS_STATUS_UNAVAIL;
    }

  /*
   * sanity check: we should have 7 fields
   */

  for (count = 0; array[count] != NULL; count++)
    {
    }

  if (count != 7)
    {
      free (array);
      return NSS_STATUS_UNAVAIL;
    }

  /*
   * Populate the pwstruct
   */

  pwstruct->pw_name   = array[0];
  pwstruct->pw_passwd = array[1];
  pwstruct->pw_uid    = (uid_t) atoi (array[2]);
  pwstruct->pw_gid    = (gid_t) atoi (array[3]);
  pwstruct->pw_gecos  = array[4];
  pwstruct->pw_dir    = array[5];
  pwstruct->pw_shell  = array[6];

  free (array);

  return NSS_STATUS_SUCCESS;
}

/*
 * getprocline:
 *
 * Passed a FILE *, gets a line and does some sanity.
 */

static enum nss_status
getprocline (FILE *proc, struct passwd *result, char *buffer, size_t buflen, int *errnop)
{
  int linenl;

  /*
   * Grab our output line.
   */

  linenl = fgets_nonl (buffer, buflen, proc);

  /*
   * Did we get any result back?
   */

  if (*buffer == '\0')
    {
      return NSS_STATUS_NOTFOUND;
    }

  /*
   * Do we have enough space in the buffer we're given to hold the result?
   */

  if (!linenl)
    {
      *errnop = ERANGE;
      return NSS_STATUS_TRYAGAIN;
    }

  return buffer_to_pwstruct (result, buffer);
}

/*
 * search:
 *
 * When passed a command, get the result and populate.
 */

static enum nss_status
search (char *command, struct passwd *result, char *buffer, size_t buflen, int *errnop)
{
  FILE *proc;
  enum nss_status status;

  *errnop = 0;

  proc = sshopen (command);

  if (!proc)
    {
      return NSS_STATUS_UNAVAIL;
    }

  status = getprocline (proc, result, buffer, buflen, errnop);

  sshclose (proc);

  return status;
}

/*
 * _nss_sshsock_getpwuid_r
 *
 * Implement getpwuid() functionality.
 */

enum nss_status
_nss_sshsock_getpwuid_r (uid_t uid, struct passwd *result, char *buffer, size_t buflen, int *errnop)
{
  char command[CHUNKSIZ];

  if (uid < MINUID)
    {
      return NSS_STATUS_NOTFOUND;
    }

  snprintf (command, sizeof command, "getent passwd %u", uid);

  return search (command, result, buffer, buflen, errnop);
}

/*
 * _nss_sshsock_getpwnam_r
 *
 * Implement getpwnam() functionality.
 */

enum nss_status
_nss_sshsock_getpwnam_r(const char *name, struct passwd *result, char *buffer, size_t buflen, int *errnop)
{
  char command[CHUNKSIZ];
  enum nss_status status;

  snprintf (command, sizeof command, "getent passwd '%s'", name);

  status = search (command, result, buffer, buflen, errnop);

  if (status == NSS_STATUS_SUCCESS)
    {
      if (result->pw_uid < MINUID)
	{
	  return NSS_STATUS_NOTFOUND;
	}
    }

  return status;
}


/*
 * _nss_sshsock_setpwent
 *
 * Implements setpwent() functionality.
 */

enum nss_status
_nss_sshsock_setpwent (void)
{
  if (pproc)
    {
      sshclose (pproc);
      pproc = NULL;
    }

  pproc = sshopen ("getent passwd");

  if (!pproc)
    {
      return NSS_STATUS_UNAVAIL;
    }

  return NSS_STATUS_SUCCESS;
}

/*
 * _nss_sshsock_getpwent_r
 *
 * Implements getpwent() functionality
 */

enum nss_status
_nss_sshsock_getpwent_r (struct passwd *result, char *buffer, size_t buflen, int *errnop)
{
  enum nss_status status;

  *errnop = 0;

  if (!pproc) /* Got a file to work on? */
    {
      return NSS_STATUS_UNAVAIL;
    }

  while (1)
    {
      status = getprocline (pproc, result, buffer, buflen, errnop);

      if ((status != NSS_STATUS_SUCCESS) ||
	  (result->pw_uid >= MINUID))
        {
          return status;
        }
    }
}

/*
 * _nss_sshsock_endpwent
 *
 * Implements the endpwent() functionality.
 */

enum nss_status
_nss_sshsock_endpwent (void)
{
  if (pproc)
    {
      sshclose (pproc);
      pproc = NULL;
    }

  return NSS_STATUS_SUCCESS;
}
