/*
 * nzb
 *
 * Copyright (C) 2004-2006 Mattias Nordstrom <matta at ftlight net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *
 * Authors:
 *   Mattias Nordstrom <matta at ftlight net>
 *
 * $Id: decoder.cpp,v 1.8 2005/10/16 12:03:31 mnordstr Exp $
 *   This file provides the decoders.
 */


#include "decoder.h"
#include "mainwindow.h"

Decoder::Decoder(NzbList *nzblist, int thread_id, QMutex *file_lock, QObject *parent)
{
	this->nzblist = nzblist;
	this->parent = parent;
	this->thread_id = thread_id;
	this->file_lock = file_lock;
	connect(this, SIGNAL(decodeEvent(QString, int, int)), parent, SLOT(decodeEvent(QString, int, int)));
}

void Decoder::run()
{
	qDebug("Decoder running.");
	emit decodeEvent("Idle", thread_id);
	
	file = 0;
	seg = 0;
	processing = false;
	
	this->exec();
	
	emit decodeEvent("Halted", thread_id);
	qDebug("Decoder done.");
}

void Decoder::stop()
{
	this->exit();
}

void Decoder::processFiles()
{
	if (getNext(&file, &seg)) {
		emit decodeEvent("Decoding ["+QString::number(file+1)+": "+QString::number(seg+1)+"/"+QString::number(nzblist->getFile(file)->getSegments()->size())+"] "+nzblist->getFile(file)->getSegment(seg)->getMsgid(), thread_id);
		
		processing = true;
		//QTimer::singleShot(0, this, SLOT(processFile()));
		processFile();
	}
}

void Decoder::processFile()
{
	if (nzblist->getFile(file)->getEncoding() == NZB_ENC_YENC || nzblist->getFile(file)->getEncoding() == NZB_UNDEF) {
		yDecode(file, seg);
	}
	
	if (nzblist->getFile(file)->getEncoding() == NZB_ENC_UENC || nzblist->getFile(file)->getEncoding() == NZB_UNDEF) {
		if (seg == 0) qDebug("No yEnc data found, trying uudecoding.");
		if (nzblist->getFile(file)->getSegments()->size() == 1) {
			uDecode(file, seg);
		} else {
			if (seg == 0) uDecode(file, seg, true, 1);
			else if (seg == nzblist->getFile(file)->getSegments()->size() - 1) uDecode(file, seg, true, 3);
			else uDecode(file, seg, true, 2);
		}
				
		if (nzblist->getFile(file)->getSegment(seg)->getEncStatus() == NZB_URES_ERROR) {
			nzblist->getFile(file)->getSegment(seg)->setEncStatus(NZB_ENC_ERROR);
		}
	}
			
	nzblist->getFile(file)->getSegment(seg)->setStatus(NZB_DECODED);
	
	processing = false;
	//emit decodeEvent("Idle", thread_id, 1);
	emit decodeEvent("Decoded ["+QString::number(file+1)+": "+QString::number(seg+1)+"/"+QString::number(nzblist->getFile(file)->getSegments()->size())+"] "+nzblist->getFile(file)->getSegment(seg)->getMsgid(), thread_id, 1);
}

bool Decoder::getNext(int *file, int *seg)
{
	if (nzblist->getList()->size() == 0) return false;
	if (processing == true) return false;
	
	file_lock->lock();
	
	for (;;) {
		if (nzblist->getFile(*file)->getSegment(*seg)->getStatus() == NZB_DOWNLOADING || nzblist->getFile(*file)->getSegment(*seg)->getStatus() == NZB_NONE) {
			file_lock->unlock();
			emit decodeEvent("Idle", thread_id, 2);
			return false;
		}
		if (nzblist->getFile(*file)->getSegment(*seg)->getStatus() == NZB_DOWNLOADED) {
			nzblist->getFile(*file)->getSegment(*seg)->setStatus(NZB_DECODING);
			file_lock->unlock();
			return true;
		}
		if (nzblist->getFile(*file)->getSegment(*seg)->getStatus() == NZB_DL_SKIPPED) {
			nzblist->getFile(*file)->getSegment(*seg)->setStatus(NZB_DEC_SKIPPED);
			file_lock->unlock();
			emit decodeEvent("Idle", thread_id, 1);
			return false;
		}
		
		if (nzblist->getFile(*file)->getSegments()->size() == *seg+1) {
			if (nzblist->getList()->size() == *file+1) {
				file_lock->unlock();
				emit decodeEvent("Idle", thread_id, 3);
				return false;
			}
			
			*file = *file + 1;
			*seg = 0;
		} else {
			*seg = *seg + 1;
		}
	}
	
	file_lock->unlock();
	return false;
}

void Decoder::yDecode(int file, int seg)
{
	// This decoder has been written to comply with yEnc revision 1.3
	// http://www.yenc.org/yenc-draft.1.3.txt
	
	int i;
	int crc32;
	QString pcrc32, header;
	QByteArray decdata = "";
	int ybegin, begin, end, pos = 0, pos2 = 0;
	QString *data = nzblist->getFile(file)->getSegment(seg)->getData();
	
	// Initialize struct with defaults.
	nzblist->getFile(file)->getSegment(seg)->setEncStatus(NZB_YRES_ERROR);
	nzblist->getFile(file)->getSegment(seg)->setFilename("");
	nzblist->getFile(file)->getSegment(seg)->setPart(0);
	nzblist->getFile(file)->getSegment(seg)->setSize(0);
	nzblist->getFile(file)->getSegment(seg)->setTotal(0);
	nzblist->getFile(file)->getSegment(seg)->setTotalSize(0);
	nzblist->getFile(file)->getSegment(seg)->setCrc32("");
	nzblist->getFile(file)->getSegment(seg)->setBlockBegin(0);
	nzblist->getFile(file)->getSegment(seg)->setBlockEnd(0);
	
	// Check for yenc encoded data.
	if ((pos = data->indexOf("=ybegin ", 0)) == -1) {
		return;
	}
	
	if (seg == 0) {
		nzblist->getFile(file)->setEncoding(NZB_ENC_YENC);
	}
	
	ybegin = pos;
	pos2 = data->indexOf("\r\n", pos);
	begin = pos2 + 2;
	
	header = data->mid(ybegin, 2048);

	// part keyword, also means that this is a multi-part file.
	if ((pos = header.indexOf(" part=")) != -1 && pos < (begin-ybegin)) {
		pos2 = data->indexOf("\r\n", begin);
		begin = pos2 + 2;
		
		pos2 = data->indexOf(QRegExp("[ \r\n]"), ybegin+pos+1);		
		nzblist->getFile(file)->getSegment(seg)->setPart((data->mid(ybegin + pos + 6, pos2 - (ybegin + pos + 6))).toInt());
	}

	// total keyword.
	if ((pos = header.indexOf(" total=")) != -1 && pos < (begin-ybegin)) {
		pos2 = header.indexOf(QRegExp("[ \r\n]"), pos+1);
		nzblist->getFile(file)->getSegment(seg)->setTotal((header.mid(pos + 7, pos2 - (pos + 7))).toInt());
	}

	// size keyword.
	if ((pos = data->indexOf(" size=", ybegin)) != -1 && pos < begin) {
		pos2 = data->indexOf(QRegExp("[ \r\n]"), pos+1);		
		nzblist->getFile(file)->getSegment(seg)->setTotalSize((data->mid(pos + 6, pos2 - (pos + 6))).toLong());
	}
	
	// name keyword.
	if ((pos = data->indexOf(" name=", ybegin)) != -1 && pos < begin) {
		pos2 = data->indexOf("\r\n", pos);		
		nzblist->getFile(file)->getSegment(seg)->setFilename(data->mid(pos + 6, pos2 - (pos + 6)).remove("\""));
	}
	
	// Multi-part messages have their own =ypart header line.
	if (nzblist->getFile(file)->getSegment(seg)->getPart() != 0) {
		if ((pos = data->indexOf("\r\n", ybegin)) != -1 && pos < begin) {
			if (data->mid(pos + 2, 7) != "=ypart ") {
				return;
			}
			
			ybegin = pos + 2;
			
			// begin keyword.
			if ((pos = data->indexOf(" begin=", ybegin)) != -1 && pos < begin) {
				pos2 = data->indexOf(QRegExp("[ \r\n]"), pos+1);		
				nzblist->getFile(file)->getSegment(seg)->setBlockBegin((data->mid(pos + 7, pos2 - (pos + 7))).toLong());
			}
			
			// end keyword.
			if ((pos = data->indexOf(" end=", ybegin)) != -1 && pos < begin) {
				pos2 = data->indexOf(QRegExp("[ \r\n]"), pos+1);		
				nzblist->getFile(file)->getSegment(seg)->setBlockEnd((data->mid(pos + 5, pos2 - (pos + 5))).toLong());
			}

		} else {
			return;
		}
	}
	
	if ((pos = data->lastIndexOf("=yend ")) == -1) {
		return;
	}
	
	end = pos;

	// size keyword.
	if ((pos = data->indexOf(" size=", end)) != -1) {
		pos2 = data->indexOf(QRegExp("[ \r\n]"), pos+1);
		nzblist->getFile(file)->getSegment(seg)->setSize((data->mid(pos + 6, pos2 - (pos + 6))).toLong());
		decdata.reserve(nzblist->getFile(file)->getSegment(seg)->getSize());
	}
	
	// part keyword.
	if ((pos = data->indexOf(" part=", end)) != -1) {
		pos2 = data->indexOf(QRegExp("[ \r\n]"), pos+1);
		if (nzblist->getFile(file)->getSegment(seg)->getPart() != (data->mid(pos + 6, pos2 - (pos + 6))).toInt()) {
			nzblist->getFile(file)->getSegment(seg)->setEncStatus(NZB_YRES_ERRPART);
			return;
		}
	}
	
	// pcrc32 keyword.
	if ((pos = data->indexOf(" pcrc32=", end)) != -1) {
		pos2 = data->indexOf(QRegExp("[ \r\n]"), pos+1);
		pcrc32 = data->mid(pos + 8, pos2 - (pos + 8));
	}
	
	// crc32 keyword.
	if ((pos = data->indexOf(" crc32=", end)) != -1) {
		pos2 = data->indexOf(QRegExp("[ \r\n]"), pos+1);
		nzblist->getFile(file)->getSegment(seg)->setCrc32(data->mid(pos + 7, pos2 - (pos + 7)));
	}
	
	*data = data->mid(begin, end - begin);
	
	crcInit(&crc32);
	
	unsigned char c;
	
	for (i=0; i<data->length(); i++){
		c = (unsigned char)data->at(i).toAscii();
		if (c == '\r' && data->at(i+1) == '\n') {
			i = i + 2;
			if (i == data->length()) continue;
			c = (unsigned char)data->at(i).toAscii();
		}
		if (c == '=') {
			i++;
			c = (unsigned char)data->at(i).toAscii();
			if (c == 0) continue;
			
			if (c-64 < 0) {
				c = c+256;
			}
			c = c-64;
		}
			
		if (c-42 < 0) {
			c = c+256;
		}
		decdata += c-42;
		crcAdd(&crc32, c-42);
	}
	
	if (pcrc32 != "" && !crc32Check(crc32, pcrc32)) {
		nzblist->getFile(file)->getSegment(seg)->setEncStatus(NZB_YRES_ERRPCRC32);
	} else {
		nzblist->getFile(file)->getSegment(seg)->setEncStatus(NZB_YRES_OK);
	}
	if (decdata.length() != nzblist->getFile(file)->getSegment(seg)->getSize()) {
		nzblist->getFile(file)->getSegment(seg)->setEncStatus(NZB_YRES_ERRSIZE);
	}
	if (nzblist->getFile(file)->getSegment(seg)->getPart() != 0 && (nzblist->getFile(file)->getSegment(seg)->getBlockEnd() - (nzblist->getFile(file)->getSegment(seg)->getBlockBegin()-1)) != nzblist->getFile(file)->getSegment(seg)->getSize()) {
		nzblist->getFile(file)->getSegment(seg)->setEncStatus(NZB_YRES_ERRSIZE);
	}

	nzblist->getFile(file)->getSegment(seg)->setDecoded(decdata);
	nzblist->getFile(file)->getSegment(seg)->setData("");
}

bool Decoder::crc32Check(int crc32, QString c_crc)
{
	unsigned int crc;
	QString calc_crc;
	
	crc = crc32 ^ 0xFFFFFFFFl;
	
	calc_crc = QString("%1").arg(crc, 8, 16, QChar('0'));
	c_crc = QString("%1").arg(c_crc, 8, QChar('0'));
	
	if (c_crc == calc_crc) {
		return true;
	} else {
		qDebug() << "yEnc CRC " << calc_crc << " != " << c_crc << "\n\n";
		return false;
	}
}

/* Single character decode.  */
#define DEC(Char) (((Char) - ' ') & 077)

void Decoder::uDecode(int file, int seg, bool multi, int type)
{
	QString decdata = "";
	int pos = 0, pos2 = 0, begin = 0, end;
	bool badnl = false; // Newlines only \n, not \r\n,
	int nlcount = 2;
	QString *data = nzblist->getFile(file)->getSegment(seg)->getData();
	
	nzblist->getFile(file)->getSegment(seg)->setEncStatus(NZB_URES_ERROR);
	nzblist->getFile(file)->getSegment(seg)->setFilename("");
  
	// Check for uencoded data.
	if (!(multi && type != 1) && (pos = data->indexOf("begin ", 0)) == -1) {
		return;
	}
	
	// Check if yenc data has been missed.
	if (!(multi && type != 1) && (data->indexOf("=ybegin ", 0)) != -1) {
		return;
	}
	
	if (!multi || (multi && type == 1))
	if (data->at(pos+6).isDigit() && data->at(pos+7).isDigit() && data->at(pos+8).isDigit() && data->at(pos+9) == ' ') {
		if ((pos2 = data->indexOf("\r\n", pos+10)) == -1) {
			badnl = true;
			nlcount = 1;
			
			if ((pos2 = data->indexOf("\n", pos+10)) == -1) {
				return;
			}
		}
		
		begin = pos2 + nlcount;
		
		nzblist->getFile(file)->getSegment(seg)->setFilename(data->mid(pos + 10, pos2 - (pos + 10)));
	} else {
		return;
	}

	if (!multi || (multi && type == 3)) {
		if ((pos = data->indexOf("\r\nend\r\n", begin)) == -1 && (pos = data->indexOf("\nend\n", begin)) == -1) {
			return;
		}
		end = pos;
	} else {
		end = data->length() - nlcount;
	}
	
	*data = data->mid(begin, (end + nlcount) - begin);

	if (seg == 0) {
		nzblist->getFile(file)->setEncoding(NZB_ENC_UENC);
	}
	
	// uudecode

	char buf[64];

	pos = 0;
	pos2 = 0;
	int ok = 1;
  
	while ((pos = data->indexOf("\n", pos2)) != -1) {
		if ((data->mid(pos2, pos + 1 - pos2)).length() > 64) {
			// Too large, something's wrong.
			pos2 = pos + 1;
			continue;
		}
		
		strcpy(buf, (data->mid(pos2, pos + 1 - pos2)).toAscii());
		pos2 = pos + 1;
 
		if  (buf[0] > ' ' && buf[0] <= '`') {
			int  cv_len, i;
			register unsigned char * p=(unsigned char*)buf;

			cv_len = DEC(*p++);
			
			/* Actually decode the uue data; ensure characters are in range. */
			if (ok) for (i=0; i<cv_len; i+=3, p+=4) {
				if ( (p[0]<=' ' || p[0]>'`') ||
				(p[1]<=' ' || p[1]>'`') ||
				(p[2]<=' ' || p[2]>'`') ||
				(p[3]<=' ' || p[3]>'`') ) {
					ok = 0;
					break;
				}
	    
				decdata += DEC(*p) << 2 | DEC(p[1]) >> 4;
				decdata += DEC(p[1]) << 4 | DEC(p[2]) >> 2;
				decdata += DEC(p[2]) << 6 | DEC(p[3]);
			}
		
			if (*p != '\r' && *p != '\n' && *p != '\0')
				ok=0;
			else  *p=0;
		}
	}
	
	nzblist->getFile(file)->getSegment(seg)->setData("");
	if (!multi || type == 3) {
		nzblist->getFile(file)->getSegment(seg)->setDecoded(decdata.left(decdata.length() - 2).toAscii()); // This is something fishy that should be fixed.
	} else {
		nzblist->getFile(file)->getSegment(seg)->setDecoded(decdata.toAscii());
	}
	
	nzblist->getFile(file)->getSegment(seg)->setEncStatus(NZB_URES_OK);
}

void Decoder::crcInit(int *crc32)
{
	*crc32 = -1L;
}

void Decoder::crcAdd(int *crc32, int c)
{
	const int crc_tab[256] = {
	0x00000000, 0x77073096, 0xee0e612c, 0x990951ba, 0x076dc419, 0x706af48f, 0xe963a535, 0x9e6495a3,
	0x0edb8832, 0x79dcb8a4, 0xe0d5e91e, 0x97d2d988, 0x09b64c2b, 0x7eb17cbd, 0xe7b82d07, 0x90bf1d91,
	0x1db71064, 0x6ab020f2, 0xf3b97148, 0x84be41de, 0x1adad47d, 0x6ddde4eb, 0xf4d4b551, 0x83d385c7,
	0x136c9856, 0x646ba8c0, 0xfd62f97a, 0x8a65c9ec, 0x14015c4f, 0x63066cd9, 0xfa0f3d63, 0x8d080df5,
	0x3b6e20c8, 0x4c69105e, 0xd56041e4, 0xa2677172, 0x3c03e4d1, 0x4b04d447, 0xd20d85fd, 0xa50ab56b,
	0x35b5a8fa, 0x42b2986c, 0xdbbbc9d6, 0xacbcf940, 0x32d86ce3, 0x45df5c75, 0xdcd60dcf, 0xabd13d59,
	0x26d930ac, 0x51de003a, 0xc8d75180, 0xbfd06116, 0x21b4f4b5, 0x56b3c423, 0xcfba9599, 0xb8bda50f,
	0x2802b89e, 0x5f058808, 0xc60cd9b2, 0xb10be924, 0x2f6f7c87, 0x58684c11, 0xc1611dab, 0xb6662d3d,
	0x76dc4190, 0x01db7106, 0x98d220bc, 0xefd5102a, 0x71b18589, 0x06b6b51f, 0x9fbfe4a5, 0xe8b8d433,
	0x7807c9a2, 0x0f00f934, 0x9609a88e, 0xe10e9818, 0x7f6a0dbb, 0x086d3d2d, 0x91646c97, 0xe6635c01,
	0x6b6b51f4, 0x1c6c6162, 0x856530d8, 0xf262004e, 0x6c0695ed, 0x1b01a57b, 0x8208f4c1, 0xf50fc457,
	0x65b0d9c6, 0x12b7e950, 0x8bbeb8ea, 0xfcb9887c, 0x62dd1ddf, 0x15da2d49, 0x8cd37cf3, 0xfbd44c65,
	0x4db26158, 0x3ab551ce, 0xa3bc0074, 0xd4bb30e2, 0x4adfa541, 0x3dd895d7, 0xa4d1c46d, 0xd3d6f4fb,
	0x4369e96a, 0x346ed9fc, 0xad678846, 0xda60b8d0, 0x44042d73, 0x33031de5, 0xaa0a4c5f, 0xdd0d7cc9,
	0x5005713c, 0x270241aa, 0xbe0b1010, 0xc90c2086, 0x5768b525, 0x206f85b3, 0xb966d409, 0xce61e49f,
	0x5edef90e, 0x29d9c998, 0xb0d09822, 0xc7d7a8b4, 0x59b33d17, 0x2eb40d81, 0xb7bd5c3b, 0xc0ba6cad,
	0xedb88320, 0x9abfb3b6, 0x03b6e20c, 0x74b1d29a, 0xead54739, 0x9dd277af, 0x04db2615, 0x73dc1683,
	0xe3630b12, 0x94643b84, 0x0d6d6a3e, 0x7a6a5aa8, 0xe40ecf0b, 0x9309ff9d, 0x0a00ae27, 0x7d079eb1,
	0xf00f9344, 0x8708a3d2, 0x1e01f268, 0x6906c2fe, 0xf762575d, 0x806567cb, 0x196c3671, 0x6e6b06e7,
	0xfed41b76, 0x89d32be0, 0x10da7a5a, 0x67dd4acc, 0xf9b9df6f, 0x8ebeeff9, 0x17b7be43, 0x60b08ed5,
	0xd6d6a3e8, 0xa1d1937e, 0x38d8c2c4, 0x4fdff252, 0xd1bb67f1, 0xa6bc5767, 0x3fb506dd, 0x48b2364b,
	0xd80d2bda, 0xaf0a1b4c, 0x36034af6, 0x41047a60, 0xdf60efc3, 0xa867df55, 0x316e8eef, 0x4669be79,
	0xcb61b38c, 0xbc66831a, 0x256fd2a0, 0x5268e236, 0xcc0c7795, 0xbb0b4703, 0x220216b9, 0x5505262f,
	0xc5ba3bbe, 0xb2bd0b28, 0x2bb45a92, 0x5cb36a04, 0xc2d7ffa7, 0xb5d0cf31, 0x2cd99e8b, 0x5bdeae1d,
	0x9b64c2b0, 0xec63f226, 0x756aa39c, 0x026d930a, 0x9c0906a9, 0xeb0e363f, 0x72076785, 0x05005713,
	0x95bf4a82, 0xe2b87a14, 0x7bb12bae, 0x0cb61b38, 0x92d28e9b, 0xe5d5be0d, 0x7cdcefb7, 0x0bdbdf21,
	0x86d3d2d4, 0xf1d4e242, 0x68ddb3f8, 0x1fda836e, 0x81be16cd, 0xf6b9265b, 0x6fb077e1, 0x18b74777,
	0x88085ae6, 0xff0f6a70, 0x66063bca, 0x11010b5c, 0x8f659eff, 0xf862ae69, 0x616bffd3, 0x166ccf45,
	0xa00ae278, 0xd70dd2ee, 0x4e048354, 0x3903b3c2, 0xa7672661, 0xd06016f7, 0x4969474d, 0x3e6e77db,
	0xaed16a4a, 0xd9d65adc, 0x40df0b66, 0x37d83bf0, 0xa9bcae53, 0xdebb9ec5, 0x47b2cf7f, 0x30b5ffe9,
	0xbdbdf21c, 0xcabac28a, 0x53b39330, 0x24b4a3a6, 0xbad03605, 0xcdd70693, 0x54de5729, 0x23d967bf,
	0xb3667a2e, 0xc4614ab8, 0x5d681b02, 0x2a6f2b94, 0xb40bbe37, 0xc30c8ea1, 0x5a05df1b, 0x2d02ef8d
	};

	unsigned long ch1,ch2,cc;
	
	cc= (c) & 0x000000ffL;
	ch1=(*crc32 ^ cc) & 0xffL;
	ch1=crc_tab[ch1];
	ch2=(*crc32>>8L) & 0xffffffL;
	*crc32=ch1 ^ ch2;
}
