#ifndef OSMIUM_OSM_TIMESTAMP_HPP
#define OSMIUM_OSM_TIMESTAMP_HPP

/*

This file is part of Osmium (http://osmcode.org/libosmium).

Copyright 2013-2015 Jochen Topf <jochen@topf.org> and others (see README).

Boost Software License - Version 1.0 - August 17th, 2003

Permission is hereby granted, free of charge, to any person or organization
obtaining a copy of the software and accompanying documentation covered by
this license (the "Software") to use, reproduce, display, distribute,
execute, and transmit the Software, and to prepare derivative works of the
Software, and to permit third-parties to whom the Software is furnished to
do so, all subject to the following:

The copyright notices in the Software and this entire statement, including
the above license grant, this restriction and the following disclaimer,
must be included in all copies of the Software, in whole or in part, and
all derivative works of the Software, unless such copies or derivative
works are solely in the form of machine-executable object code generated by
a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.

*/

#include <cstdint>
#include <ctime>
#include <iosfwd>
#include <limits>
#include <stdexcept>
#include <string>
#include <time.h>

#include <osmium/util/compatibility.hpp>

namespace osmium {

    /**
     * A timestamp. Internal representation is an unsigned 32bit integer
     * holding seconds since epoch, so this will overflow in 2038.
     */
    class Timestamp {

        // length of ISO timestamp string yyyy-mm-ddThh:mm:ssZ\0
        static constexpr int timestamp_length = 20 + 1;

        /**
         * The timestamp format for OSM timestamps in strftime(3) format.
         * This is the ISO-Format yyyy-mm-ddThh:mm:ssZ
         */
        static const char* timestamp_format() {
            static const char f[timestamp_length] = "%Y-%m-%dT%H:%M:%SZ";
            return f;
        }

        uint32_t m_timestamp;

    public:

        constexpr Timestamp() noexcept :
            m_timestamp(0) {
        }

        // Not "explicit" so that conversions from time_t work
        // like in node.timestamp(123);
        constexpr Timestamp(time_t timestamp) noexcept :
            m_timestamp(static_cast<uint32_t>(timestamp)) {
        }

        /**
         * Construct timestamp from ISO date/time string.
         * Throws std::invalid_argument, if the timestamp can not be parsed.
         */
        explicit Timestamp(const char* timestamp) {
#ifndef _WIN32
            struct tm tm {
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
            };
            if (strptime(timestamp, timestamp_format(), &tm) == nullptr) {
                throw std::invalid_argument("can't parse timestamp");
            }
            m_timestamp = static_cast<uint32_t>(timegm(&tm));
#else
            struct tm tm;
            int n = sscanf(timestamp, "%4d-%2d-%2dT%2d:%2d:%2dZ", &tm.tm_year, &tm.tm_mon, &tm.tm_mday, &tm.tm_hour, &tm.tm_min, &tm.tm_sec);
            if (n != 6) {
                throw std::invalid_argument("can't parse timestamp");
            }
            tm.tm_year -= 1900;
            tm.tm_mon--;
            tm.tm_wday = 0;
            tm.tm_yday = 0;
            tm.tm_isdst = 0;
            m_timestamp = static_cast<uint32_t>(_mkgmtime(&tm));
#endif
        }

        constexpr time_t seconds_since_epoch() const noexcept {
            return static_cast<time_t>(m_timestamp);
        }

        constexpr operator time_t() const noexcept {
            return static_cast<time_t>(m_timestamp);
        }

        explicit constexpr operator uint32_t() const noexcept {
            return m_timestamp;
        }

        template <typename T>
        void operator+=(T time_difference) noexcept {
            m_timestamp += time_difference;
        }

        template <typename T>
        void operator-=(T time_difference) noexcept {
            m_timestamp -= time_difference;
        }

        /**
         * Return UTC Unix time as string in ISO date/time format.
         */
        std::string to_iso() const {
            std::string s;

            if (m_timestamp != 0) {
                struct tm tm;
                time_t sse = seconds_since_epoch();
#ifndef _MSC_VER
                gmtime_r(&sse, &tm);
#else
                gmtime_s(&tm, &sse);
#endif

                s.resize(timestamp_length);
                /* This const_cast is ok, because we know we have enough space
                in the string for the format we are using (well at least until
                the year will have 5 digits). And by setting the size
                afterwards from the result of strftime we make sure thats set
                right, too. */
                s.resize(strftime(const_cast<char*>(s.c_str()), timestamp_length, timestamp_format(), &tm));
            }

            return s;
        }

    }; // class Timestamp

    inline OSMIUM_CONSTEXPR Timestamp start_of_time() noexcept {
        return Timestamp(1);
    }

    inline OSMIUM_CONSTEXPR Timestamp end_of_time() noexcept {
        return Timestamp(std::numeric_limits<time_t>::max());
    }

    template <typename TChar, typename TTraits>
    inline std::basic_ostream<TChar, TTraits>& operator<<(std::basic_ostream<TChar, TTraits>& out, Timestamp timestamp) {
        out << timestamp.to_iso();
        return out;
    }

} // namespace osmium

#endif // OSMIUM_OSM_TIMESTAMP_HPP
