#include "test_data.hpp"

#include "libslic3r/TriangleMesh.hpp"
#include "libslic3r/GCodeReader.hpp"
#include "libslic3r/Config.hpp"
#include "libslic3r/Print.hpp"
#include "libslic3r/Format/OBJ.hpp"
#include "libslic3r/Format/STL.hpp"

#include <cstdlib>
#include <string>

#include <boost/nowide/cstdio.hpp>
#include <boost/nowide/fstream.hpp>
#include <boost/filesystem.hpp>
#include <boost/regex.hpp>
#include <libslic3r/ModelArrange.hpp>

using namespace std;

namespace Slic3r { namespace Test {

// Mesh enumeration to name mapping
const std::unordered_map<TestMesh, const char*, TestMeshHash> mesh_names {
    std::pair<TestMesh, const char*>(TestMesh::A,						"A"),
    std::pair<TestMesh, const char*>(TestMesh::L,						"L"), 
    std::pair<TestMesh, const char*>(TestMesh::V,						"V"), 
    std::pair<TestMesh, const char*>(TestMesh::_40x10,					"40x10"), 
    std::pair<TestMesh, const char*>(TestMesh::cube_20x20x20,			"cube_20x20x20"), 
    std::pair<TestMesh, const char*>(TestMesh::cube_2x20x10,            "cube_2x20x10"), 
    std::pair<TestMesh, const char*>(TestMesh::sphere_50mm,				"sphere_50mm"), 
    std::pair<TestMesh, const char*>(TestMesh::bridge,					"bridge"), 
    std::pair<TestMesh, const char*>(TestMesh::bridge_with_hole,		"bridge_with_hole"), 
    std::pair<TestMesh, const char*>(TestMesh::cube_with_concave_hole,	"cube_with_concave_hole"),
    std::pair<TestMesh, const char*>(TestMesh::cube_with_hole,			"cube_with_hole"), 
    std::pair<TestMesh, const char*>(TestMesh::gt2_teeth,				"gt2_teeth"), 
    std::pair<TestMesh, const char*>(TestMesh::ipadstand,				"ipadstand"), 
    std::pair<TestMesh, const char*>(TestMesh::overhang,				"overhang"), 
    std::pair<TestMesh, const char*>(TestMesh::pyramid,					"pyramid"), 
    std::pair<TestMesh, const char*>(TestMesh::sloping_hole,			"sloping_hole"), 
    std::pair<TestMesh, const char*>(TestMesh::slopy_cube,				"slopy_cube"), 
    std::pair<TestMesh, const char*>(TestMesh::small_dorito,			"small_dorito"), 
    std::pair<TestMesh, const char*>(TestMesh::step,					"step"), 
    std::pair<TestMesh, const char*>(TestMesh::two_hollow_squares,		"two_hollow_squares")
};

TriangleMesh mesh(TestMesh m)
{
    TriangleMesh mesh;
    switch(m) {
        case TestMesh::cube_20x20x20:
            mesh = Slic3r::make_cube(20, 20, 20);
            break;
        case TestMesh::cube_2x20x10:
            mesh = Slic3r::make_cube(2, 20, 10);
            break;
        case TestMesh::sphere_50mm:
            mesh = Slic3r::make_sphere(50, PI / 243.0);
            break;
        case TestMesh::cube_with_hole:
            mesh = TriangleMesh(
            	{ {0,0,0}, {0,0,10}, {0,20,0}, {0,20,10}, {20,0,0}, {20,0,10}, {5,5,0}, {15,5,0}, {5,15,0}, {20,20,0}, {15,15,0}, {20,20,10}, {5,5,10}, {5,15,10}, {15,5,10}, {15,15,10} },
            	{ {0,1,2}, {2,1,3}, {1,0,4}, {5,1,4}, {6,7,4}, {8,2,9}, {0,2,8}, {10,8,9}, {0,8,6}, {0,6,4}, {4,7,9}, {7,10,9}, {2,3,9}, {9,3,11}, {12,1,5}, {13,3,12}, {14,12,5}, {3,1,12}, {11,3,13}, {11,15,5}, {11,13,15}, {15,14,5}, {5,4,9}, {11,5,9}, {8,13,12}, {6,8,12}, {10,15,13}, {8,10,13}, {15,10,14}, {14,10,7}, {14,7,12}, {12,7,6} });
            break;
        case TestMesh::cube_with_concave_hole:
            mesh = TriangleMesh(
            	{ {-10,-10,-5}, {-10,-10,5}, {-10,10,-5}, {-10,10,5}, {10,-10,-5}, {10,-10,5}, {-5,-5,-5}, {5,-5,-5}, {5,5,-5}, {5,10,-5}, {-5,5,-5}, Vec3f(3.06161699911402e-16f,5,-5), 
                  {5,0,-5}, {0,0,-5}, {10,5,-5}, {5,10,5}, {-5,-5,5}, {5,0,5}, {5,-5,5}, {-5,5,5}, {10,5,5}, {5,5,5}, Vec3f(3.06161699911402e-16f,5,5), {0,0,5} }, 
            	{ {0,1,2}, {2,1,3}, {1,0,4}, {5,1,4}, {6,7,4}, {8,2,9}, {10,2,11}, {11,12,13}, {0,2,10}, {0,10,6}, {0,6,4}, {11,2,8}, {4,7,12}, {4,12,8}, {12,11,8}, {14,4,8}, {2,3,9}, {9,3,15}, {16,1,5}, {17,18,5}, {19,3,16}, {20,21,5}, {18,16,5}, {3,1,16}, {22,3,19}, {21,3,22}, {21,17,5}, {21,22,17}, {21,15,3}, {23,17,22}, {5,4,14}, {20,5,14}, {20,14,21}, {21,14,8}, {9,15,21}, {8,9,21}, {10,19,16}, {6,10,16}, {11,22,19}, {10,11,19}, {13,23,11}, {11,23,22}, {23,13,12}, {17,23,12}, {17,12,18}, {18,12,7}, {18,7,16}, {16,7,6} });
            break;
        case TestMesh::V:
            mesh = TriangleMesh(
            	{ {-14,0,20}, {-14,15,20}, {0,0,0}, {0,15,0}, {-4,0,20}, {-4,15,20}, {5,0,7.14286f}, {10,0,0}, {24,0,20}, {14,0,20}, {10,15,0}, {5,15,7.14286f}, {14,15,20}, {24,15,20} },
				{ {0,1,2}, {2,1,3}, {1,0,4}, {5,1,4}, {4,0,2}, {6,4,2}, {7,6,2}, {8,9,7}, {9,6,7}, {2,3,7}, {7,3,10}, {1,5,3}, {3,5,11}, {11,12,13}, {11,13,3}, {3,13,10}, {5,4,6}, {11,5,6}, {6,9,11}, {11,9,12}, {12,9,8}, {13,12,8}, {8,7,10}, {13,8,10} });
            break;
        case TestMesh::L:
            mesh = TriangleMesh(
            	{ {0,10,0}, {0,10,10}, {0,20,0}, {0,20,10}, {10,10,0}, {10,10,10}, {20,20,0}, {20,0,0}, {10,0,0}, {20,20,10}, {10,0,10}, {20,0,10} },
            	{ {0,1,2}, {2,1,3}, {4,5,1}, {0,4,1}, {0,2,4}, {4,2,6}, {4,6,7}, {4,7,8}, {2,3,6}, {6,3,9}, {3,1,5}, {9,3,5}, {10,11,5}, {11,9,5}, {5,4,10}, {10,4,8}, {10,8,7}, {11,10,7}, {11,7,6}, {9,11,6} });
            break;
        case TestMesh::overhang:
        	mesh = TriangleMesh(
        		{ {1364.68505859375,614.398010253906,20.002498626709}, {1389.68505859375,614.398010253906,20.002498626709}, {1377.18505859375,589.398986816406,20.002498626709}, {1389.68505859375,589.398986816406,20.002498626709}, {1389.68505859375,564.398986816406,20.0014991760254}, {1364.68505859375,589.398986816406,20.002498626709}, {1364.68505859375,564.398986816406,20.0014991760254}, {1360.93505859375,589.398986816406,17.0014991760254}, {1360.93505859375,585.64697265625,17.0014991760254}, {1357.18505859375,564.398986816406,17.0014991760254}, {1364.68505859375,589.398986816406,17.0014991760254}, {1364.68505859375,571.899963378906,17.0014991760254}, {1364.68505859375,564.398986816406,17.0014991760254}, {1348.43603515625,564.398986816406,17.0014991760254}, {1352.80908203125,589.398986816406,17.0014991760254}, {1357.18408203125,589.398986816406,17.0014991760254}, {1357.18310546875,614.398010253906,17.0014991760254}, {1364.68505859375,606.89599609375,17.0014991760254}, {1364.68505859375,614.398010253906,17.0014991760254}, {1352.18603515625,564.398986816406,20.0014991760254}, {1363.65405273438,589.398986816406,23.3004989624023}, {1359.46704101562,589.398986816406,23.3004989624023}, {1358.37109375,564.398986816406,23.3004989624023}, {1385.56103515625,564.398986816406,23.3004989624023}, {1373.06311035156,589.398986816406,23.3004989624023}, {1368.80810546875,564.398986816406,23.3004989624023}, {1387.623046875,589.398986816406,23.3004989624023}, {1387.623046875,585.276000976562,23.3004989624023}, {1389.68505859375,589.398986816406,23.3004989624023}, {1389.68505859375,572.64599609375,23.3004989624023}, {1389.68505859375,564.398986816406,23.3004989624023}, {1367.77709960938,589.398986816406,23.3004989624023}, {1366.7470703125,564.398986816406,23.3004989624023}, {1354.31201171875,589.398986816406,23.3004989624023}, {1352.18603515625,564.398986816406,23.3004989624023}, {1389.68505859375,614.398010253906,23.3004989624023}, {1377.31701660156,614.398010253906,23.3004989624023}, {1381.43908691406,589.398986816406,23.3004989624023}, {1368.80700683594,614.398010253906,23.3004989624023}, {1368.80810546875,589.398986816406,23.3004989624023}, {1356.43908691406,614.398010253906,23.3004989624023}, {1357.40502929688,589.398986816406,23.3004989624023}, {1360.56201171875,614.398010253906,23.3004989624023}, {1348.705078125,614.398010253906,23.3004989624023}, {1350.44506835938,589.398986816406,23.3004989624023}, {1389.68505859375,606.153015136719,23.3004989624023}, {1347.35205078125,589.398986816406,23.3004989624023}, {1346.56005859375,589.398986816406,23.3004989624023}, {1346.56005859375,594.159912109375,17.0014991760254}, {1346.56005859375,589.398986816406,17.0014991760254}, {1346.56005859375,605.250427246094,23.3004989624023}, {1346.56005859375,614.398010253906,23.3004989624023}, {1346.56005859375,614.398010253906,20.8258285522461}, {1346.56005859375,614.398010253906,17.0014991760254}, {1346.56005859375,564.398986816406,19.10133934021}, {1346.56005859375,567.548583984375,23.3004989624023}, {1346.56005859375,564.398986816406,17.0020332336426}, {1346.56005859375,564.398986816406,23.0018501281738}, {1346.56005859375,564.398986816406,23.3004989624023}, {1346.56005859375,575.118957519531,17.0014991760254}, {1346.56005859375,574.754028320312,23.3004989624023} },
				{ {0,1,2}, {2,3,4}, {2,5,0}, {4,6,2}, {2,6,5}, {2,1,3}, {7,8,9}, {10,9,8}, {11,9,10}, {12,9,11}, {9,13,14}, {7,15,16}, {10,17,0}, {10,0,5}, {12,11,6}, {18,16,0}, {6,19,13}, {6,13,9}, {9,12,6}, {17,18,0}, {11,10,5}, {11,5,6}, {14,16,15}, {17,7,18}, {16,18,7}, {14,15,9}, {7,9,15}, {7,17,8}, {10,8,17}, {20,21,22}, {23,24,25}, {26,23,27}, {28,27,23}, {29,28,23}, {30,29,23}, {25,31,32}, {22,33,34}, {35,36,37}, {24,38,39}, {21,40,41}, {38,42,20}, {33,43,44}, {6,4,23}, {6,23,25}, {36,35,1}, {1,0,38}, {1,38,36}, {29,30,4}, {25,32,6}, {40,42,0}, {35,45,1}, {4,3,28}, {4,28,29}, {3,1,45}, {3,45,28}, {22,34,19}, {19,6,32}, {19,32,22}, {42,38,0}, {30,23,4}, {0,16,43}, {0,43,40}, {24,37,36}, {38,24,36}, {24,23,37}, {37,23,26}, {22,32,20}, {20,32,31}, {33,41,40}, {43,33,40}, {45,35,26}, {37,26,35}, {33,44,34}, {44,43,46}, {20,42,21}, {40,21,42}, {31,39,38}, {20,31,38}, {33,22,41}, {21,41,22}, {31,25,39}, {24,39,25}, {26,27,45}, {28,45,27}, {47,48,49}, {47,50,48}, {51,48,50}, {52,48,51}, {53,48,52}, {54,55,56}, {57,55,54}, {58,55,57}, {49,59,47}, {60,56,55}, {59,56,60}, {60,47,59}, {48,53,16}, {56,13,19}, {54,56,19}, {56,59,13}, {59,49,14}, {59,14,13}, {49,48,16}, {49,16,14}, {44,46,60}, {44,60,55}, {51,50,43}, {19,34,58}, {19,58,57}, {53,52,16}, {43,16,52}, {43,52,51}, {57,54,19}, {47,60,46}, {55,58,34}, {55,34,44}, {50,47,46}, {50,46,43} });
            break;
        case TestMesh::_40x10:
            mesh = TriangleMesh(
            	{ {12.8680295944214,29.5799007415771,12}, {11.7364797592163,29.8480796813965,12}, {11.1571502685547,29.5300102233887,12}, {10.5814504623413,29.9830799102783,12}, {10,29.6000003814697,12}, {9.41855144500732,29.9830799102783,12}, {8.84284687042236,29.5300102233887,12}, {8.26351833343506,29.8480796813965,12}, {7.70256900787354,29.3210391998291,12}, {7.13196802139282,29.5799007415771,12}, {6.59579277038574,28.9761600494385,12}, {6.03920221328735,29.1821594238281,12}, {5.53865718841553,28.5003795623779,12}, {5,28.6602592468262,12}, {4.54657793045044,27.9006500244141,12}, {4.02841377258301,28.0212306976318,12}, {3.63402199745178,27.1856994628906,12}, {3.13758301734924,27.2737407684326,12}, {2.81429696083069,26.3659801483154,12}, {2.33955597877502,26.4278793334961,12}, {2.0993549823761,25.4534206390381,12}, {1.64512205123901,25.4950904846191,12}, {1.49962198734283,24.4613399505615,12}, {1.0636739730835,24.4879894256592,12}, {1.02384400367737,23.4042091369629,12}, {0.603073298931122,23.4202003479004,12}, {0.678958415985107,22.2974300384521,12}, {0.269550800323486,22.3061599731445,12}, {0.469994693994522,21.1571502685547,12}, {0.067615881562233,21.1609306335449,12}, {0.399999290704727,20,12}, {0,20,12}, {0.399999290704727,5,12}, {0,5,12}, {0.456633001565933,4.2804012298584,12}, {0.0615576282143593,4.21782684326172,12}, {0.625140011310577,3.5785219669342,12}, {0.244717106223106,3.45491504669189,12}, {0.901369392871857,2.91164398193359,12}, {0.544967114925385,2.73004698753357,12}, {1.27852201461792,2.29618692398071,12}, {0.954914808273315,2.06107401847839,12}, {1.74730801582336,1.74730801582336,12}, {1.46446597576141,1.46446597576141,12}, {2.29618692398071,1.27852201461792,12}, {2.06107401847839,0.954914808273315,12}, {2.91164398193359,0.901369392871857,12}, {2.73004698753357,0.544967114925385,12}, {3.5785219669342,0.625140011310577,12}, {3.45491504669189,0.244717106223106,12}, {4.2804012298584,0.456633001565933,12}, {4.21782684326172,0.0615576282143593,12}, {5,0.399999290704727,12}, {5,0,12}, {19.6000003814697,0.399999290704727,12}, {20,0,12}, {19.6000003814697,20,12}, {20,20,12}, {19.5300102233887,21.1571502685547,12}, {19.9323806762695,21.1609306335449,12}, {19.3210391998291,22.2974300384521,12}, {19.7304496765137,22.3061599731445,12}, {18.9761600494385,23.4042091369629,12}, {19.3969306945801,23.4202003479004,12}, {18.5003795623779,24.4613399505615,12}, {18.9363307952881,24.4879894256592,12}, {17.9006500244141,25.4534206390381,12}, {18.3548793792725,25.4950904846191,12}, {17.1856994628906,26.3659801483154,12}, {17.6604404449463,26.4278793334961,12}, {16.3659801483154,27.1856994628906,12}, {16.862419128418,27.2737407684326,12}, {15.4534196853638,27.9006500244141,12}, {15.9715900421143,28.0212306976318,12}, {14.4613399505615,28.5003795623779,12}, {15,28.6602592468262,12}, {13.4042100906372,28.9761600494385,12}, {13.9608001708984,29.1821594238281,12}, {12.2974300384521,29.3210391998291,12}, {7.13196802139282,29.5799007415771,0}, {8.26351833343506,29.8480796813965,0}, {8.84284687042236,29.5300102233887,0}, {9.41855144500732,29.9830799102783,0}, {10,29.6000003814697,0}, {10.5814504623413,29.9830799102783,0}, {11.1571502685547,29.5300102233887,0}, {11.7364797592163,29.8480796813965,0}, {12.2974300384521,29.3210391998291,0}, {12.8680295944214,29.5799007415771,0}, {13.4042100906372,28.9761600494385,0}, {13.9608001708984,29.1821594238281,0}, {14.4613399505615,28.5003795623779,0}, {15,28.6602592468262,0}, {15.4534196853638,27.9006500244141,0}, {15.9715900421143,28.0212306976318,0}, {16.3659801483154,27.1856994628906,0}, {16.862419128418,27.2737407684326,0}, {17.1856994628906,26.3659801483154,0}, {17.6604404449463,26.4278793334961,0}, {17.9006500244141,25.4534206390381,0}, {18.3548793792725,25.4950904846191,0}, {18.5003795623779,24.4613399505615,0}, {18.9363307952881,24.4879894256592,0}, {18.9761600494385,23.4042091369629,0}, {19.3969306945801,23.4202003479004,0}, {19.3210391998291,22.2974300384521,0}, {19.7304496765137,22.3061599731445,0}, {19.5300102233887,21.1571502685547,0}, {19.9323806762695,21.1609306335449,0}, {19.6000003814697,20,0}, {20,20,0}, {19.6000003814697,0.399999290704727,0}, {20,0,0}, {5,0.399999290704727,0}, {5,0,0}, {4.2804012298584,0.456633001565933,0}, {4.21782684326172,0.0615576282143593,0}, {3.5785219669342,0.625140011310577,0}, {3.45491504669189,0.244717106223106,0}, {2.91164398193359,0.901369392871857,0}, {2.73004698753357,0.544967114925385,0}, {2.29618692398071,1.27852201461792,0}, {2.06107401847839,0.954914808273315,0}, {1.74730801582336,1.74730801582336,0}, {1.46446597576141,1.46446597576141,0}, {1.27852201461792,2.29618692398071,0}, {0.954914808273315,2.06107401847839,0}, {0.901369392871857,2.91164398193359,0}, {0.544967114925385,2.73004698753357,0}, {0.625140011310577,3.5785219669342,0}, {0.244717106223106,3.45491504669189,0}, {0.456633001565933,4.2804012298584,0}, {0.0615576282143593,4.21782684326172,0}, {0.399999290704727,5,0}, {0,5,0}, {0.399999290704727,20,0}, {0,20,0}, {0.469994693994522,21.1571502685547,0}, {0.067615881562233,21.1609306335449,0}, {0.678958415985107,22.2974300384521,0}, {0.269550800323486,22.3061599731445,0}, {1.02384400367737,23.4042091369629,0}, {0.603073298931122,23.4202003479004,0}, {1.49962198734283,24.4613399505615,0}, {1.0636739730835,24.4879894256592,0}, {2.0993549823761,25.4534206390381,0}, {1.64512205123901,25.4950904846191,0}, {2.81429696083069,26.3659801483154,0}, {2.33955597877502,26.4278793334961,0}, {3.63402199745178,27.1856994628906,0}, {3.13758301734924,27.2737407684326,0}, {4.54657793045044,27.9006500244141,0}, {4.02841377258301,28.0212306976318,0}, {5.53865718841553,28.5003795623779,0}, {5,28.6602592468262,0}, {6.59579277038574,28.9761600494385,0}, {6.03920221328735,29.1821594238281,0}, {7.70256900787354,29.3210391998291,0} },
				{ {0,1,2}, {2,1,3}, {2,3,4}, {4,3,5}, {4,5,6}, {6,5,7}, {6,7,8}, {8,7,9}, {8,9,10}, {10,9,11}, {10,11,12}, {12,11,13}, {12,13,14}, {14,13,15}, {14,15,16}, {16,15,17}, {16,17,18}, {18,17,19}, {18,19,20}, {20,19,21}, {20,21,22}, {22,21,23}, {22,23,24}, {24,23,25}, {24,25,26}, {26,25,27}, {26,27,28}, {28,27,29}, {28,29,30}, {30,29,31}, {30,31,32}, {32,31,33}, {32,33,34}, {34,33,35}, {34,35,36}, {36,35,37}, {36,37,38}, {38,37,39}, {38,39,40}, {40,39,41}, {40,41,42}, {42,41,43}, {42,43,44}, {44,43,45}, {44,45,46}, {46,45,47}, {46,47,48}, {48,47,49}, {48,49,50}, {50,49,51}, {50,51,52}, {52,51,53}, {52,53,54}, {54,53,55}, {54,55,56}, {56,55,57}, {56,57,58}, {58,57,59}, {58,59,60}, {60,59,61}, {60,61,62}, {62,61,63}, {62,63,64}, {64,63,65}, {64,65,66}, {66,65,67}, {66,67,68}, {68,67,69}, {68,69,70}, {70,69,71}, {70,71,72}, {72,71,73}, {72,73,74}, {74,73,75}, {74,75,76}, {76,75,77}, {76,77,78}, {78,77,0}, {78,0,2}, {79,80,81}, {81,80,82}, {81,82,83}, {83,82,84}, {83,84,85}, {85,84,86}, {85,86,87}, {87,86,88}, {87,88,89}, {89,88,90}, {89,90,91}, {91,90,92}, {91,92,93}, {93,92,94}, {93,94,95}, {95,94,96}, {95,96,97}, {97,96,98}, {97,98,99}, {99,98,100}, {99,100,101}, {101,100,102}, {101,102,103}, {103,102,104}, {103,104,105}, {105,104,106}, {105,106,107}, {107,106,108}, {107,108,109}, {109,108,110}, {109,110,111}, {111,110,112}, {111,112,113}, {113,112,114}, {113,114,115}, {115,114,116}, {115,116,117}, {117,116,118}, {117,118,119}, {119,118,120}, {119,120,121}, {121,120,122}, {121,122,123}, {123,122,124}, {123,124,125}, {125,124,126}, {125,126,127}, {127,126,128}, {127,128,129}, {129,128,130}, {129,130,131}, {131,130,132}, {131,132,133}, {133,132,134}, {133,134,135}, {135,134,136}, {135,136,137}, {137,136,138}, {137,138,139}, {139,138,140}, {139,140,141}, {141,140,142}, {141,142,143}, {143,142,144}, {143,144,145}, {145,144,146}, {145,146,147}, {147,146,148}, {147,148,149}, {149,148,150}, {149,150,151}, {151,150,152}, {151,152,153}, {153,152,154}, {153,154,155}, {155,154,156}, {155,156,157}, {157,156,79}, {157,79,81}, {57,110,108}, {57,108,59}, {59,108,106}, {59,106,61}, {61,106,104}, {61,104,63}, {63,104,102}, {63,102,65}, {65,102,100}, {65,100,67}, {67,100,98}, {67,98,69}, {69,98,96}, {69,96,71}, {71,96,94}, {71,94,73}, {73,94,92}, {73,92,75}, {75,92,90}, {75,90,77}, {77,90,88}, {77,88,0}, {0,88,86}, {0,86,1}, {1,86,84}, {1,84,3}, {3,84,82}, {3,82,5}, {5,82,80}, {5,80,7}, {7,80,79}, {7,79,9}, {9,79,156}, {9,156,11}, {11,156,154}, {11,154,13}, {13,154,152}, {13,152,15}, {15,152,150}, {15,150,17}, {17,150,148}, {17,148,19}, {19,148,146}, {19,146,21}, {21,146,144}, {21,144,23}, {23,144,142}, {23,142,25}, {25,142,140}, {25,140,27}, {27,140,138}, {27,138,29}, {29,138,136}, {29,136,31}, {33,31,134}, {134,31,136}, {33,134,132}, {33,132,35}, {35,132,130}, {35,130,37}, {37,130,128}, {37,128,39}, {39,128,126}, {39,126,41}, {41,126,124}, {41,124,43}, {43,124,122}, {43,122,45}, {45,122,120}, {45,120,47}, {47,120,118}, {47,118,49}, {49,118,116}, {49,116,51}, {51,116,114}, {51,114,53}, {55,53,112}, {112,53,114}, {57,55,110}, {110,55,112}, {30,135,137}, {30,137,28}, {28,137,139}, {28,139,26}, {26,139,141}, {26,141,24}, {24,141,143}, {24,143,22}, {22,143,145}, {22,145,20}, {20,145,147}, {20,147,18}, {18,147,149}, {18,149,16}, {16,149,151}, {16,151,14}, {14,151,153}, {14,153,12}, {12,153,155}, {12,155,10}, {10,155,157}, {10,157,8}, {8,157,81}, {8,81,6}, {6,81,83}, {6,83,4}, {4,83,85}, {4,85,2}, {2,85,87}, {2,87,78}, {78,87,89}, {78,89,76}, {76,89,91}, {76,91,74}, {74,91,93}, {74,93,72}, {72,93,95}, {72,95,70}, {70,95,97}, {70,97,68}, {68,97,99}, {68,99,66}, {66,99,101}, {66,101,64}, {64,101,103}, {64,103,62}, {62,103,105}, {62,105,60}, {60,105,107}, {60,107,58}, {58,107,109}, {58,109,56}, {30,32,135}, {135,32,133}, {52,113,115}, {52,115,50}, {50,115,117}, {50,117,48}, {48,117,119}, {48,119,46}, {46,119,121}, {46,121,44}, {44,121,123}, {44,123,42}, {42,123,125}, {42,125,40}, {40,125,127}, {40,127,38}, {38,127,129}, {38,129,36}, {36,129,131}, {36,131,34}, {34,131,133}, {34,133,32}, {52,54,113}, {113,54,111}, {54,56,111}, {111,56,109} });
            break;
        case TestMesh::sloping_hole:
            mesh = TriangleMesh(
            	{ {-20,-20,-5}, {-20,-20,5}, {-20,20,-5}, {-20,20,5}, {20,-20,-5}, {20,-20,5}, {4.46294021606445,7.43144989013672,-5}, {20,20,-5}, {-19.1420993804932,0,-5}, {-18.8330993652344,-2.07911992073059,-5}, {-17.9195003509521,-4.06736993789673,-5}, {-16.4412002563477,-5.87785005569458,-5}, {-14.4629001617432,-7.43144989013672,-5}, {-12.0711002349854,-8.66024971008301,-5}, {-9.37016010284424,-9.51056003570557,-5}, {-3.5217399597168,-9.94521999359131,-5}, {-6.4782600402832,-9.94521999359131,-5}, {-0.629840016365051,-9.51056003570557,-5}, {2.07106995582581,-8.66024971008301,-5}, {6.44122982025146,-5.87785005569458,-5}, {4.46294021606445,-7.43144989013672,-5}, {-12.0711002349854,8.66024971008301,-5}, {-9.37016010284424,9.51056003570557,-5}, {7.91947984695435,-4.06736993789673,-5}, {8.83310031890869,-2.07911992073059,-5}, {-6.4782600402832,9.94521999359131,-5}, {-0.629840016365051,9.51056003570557,-5}, {2.07106995582581,8.66024971008301,-5}, {9.14214038848877,0,-5}, {8.83310031890869,2.07911992073059,-5}, {-3.5217399597168,9.94521999359131,-5}, {7.91947984695435,4.06736993789673,-5}, {6.44122982025146,5.87785005569458,-5}, {-14.4629001617432,7.43144989013672,-5}, {-16.4412002563477,5.87785005569458,-5}, {-17.9195003509521,4.06736993789673,-5}, {-18.8330993652344,2.07911992073059,-5}, {20,20,5}, {3.5217399597168,-9.94521999359131,5}, {-8.83310031890869,-2.07911992073059,5}, {-9.14214038848877,0,5}, {-8.83310031890869,2.07911992073059,5}, {6.4782600402832,-9.94521999359131,5}, {-7.91947984695435,4.06736993789673,5}, {-6.44122982025146,5.87785005569458,5}, {-4.46294021606445,7.43144989013672,5}, {-2.07106995582581,8.66024971008301,5}, {0.629840016365051,9.51056003570557,5}, {12.0711002349854,-8.66024971008301,5}, {9.37016010284424,-9.51056003570557,5}, {3.5217399597168,9.94521999359131,5}, {6.4782600402832,9.94521999359131,5}, {9.37016010284424,9.51056003570557,5}, {12.0711002349854,8.66024971008301,5}, {14.4629001617432,7.43144989013672,5}, {16.4412002563477,-5.87785005569458,5}, {14.4629001617432,-7.43144989013672,5}, {16.4412002563477,5.87785005569458,5}, {17.9195003509521,4.06736993789673,5}, {18.8330993652344,-2.07911992073059,5}, {17.9195003509521,-4.06736993789673,5}, {18.8330993652344,2.07911992073059,5}, {19.1420993804932,0,5}, {0.629840016365051,-9.51056003570557,5}, {-2.07106995582581,-8.66024971008301,5}, {-4.46294021606445,-7.43144989013672,5}, {-6.44122982025146,-5.87785005569458,5}, {-7.91947984695435,-4.06736993789673,5} },
            	{ {0,1,2}, {2,1,3}, {1,0,4}, {5,1,4}, {6,2,7}, {0,2,8}, {0,8,9}, {0,9,10}, {0,10,11}, {0,11,12}, {0,12,13}, {0,13,4}, {13,14,4}, {15,4,16}, {17,4,15}, {18,4,17}, {19,4,20}, {18,20,4}, {21,2,22}, {4,19,23}, {4,23,7}, {23,24,7}, {22,2,25}, {26,2,27}, {28,29,7}, {25,2,30}, {29,31,7}, {30,2,26}, {31,32,7}, {27,2,6}, {32,6,7}, {28,7,24}, {33,2,21}, {34,2,33}, {35,2,34}, {36,2,35}, {8,2,36}, {16,4,14}, {2,3,7}, {7,3,37}, {38,1,5}, {3,1,39}, {3,39,40}, {3,40,41}, {42,38,5}, {3,41,43}, {3,43,44}, {37,3,45}, {37,45,46}, {37,46,47}, {48,49,5}, {37,47,50}, {49,42,5}, {37,50,51}, {37,51,52}, {37,52,53}, {37,53,54}, {55,56,5}, {37,54,57}, {37,57,58}, {59,60,5}, {37,58,61}, {37,62,5}, {37,61,62}, {62,59,5}, {60,55,5}, {63,1,38}, {64,1,63}, {65,1,64}, {66,1,65}, {67,1,66}, {39,1,67}, {44,45,3}, {56,48,5}, {5,4,7}, {37,5,7}, {41,40,36}, {36,40,8}, {39,9,40}, {40,9,8}, {43,41,35}, {35,41,36}, {44,43,34}, {34,43,35}, {33,45,44}, {34,33,44}, {21,46,45}, {33,21,45}, {22,47,46}, {21,22,46}, {25,50,47}, {22,25,47}, {30,51,50}, {25,30,50}, {26,52,51}, {30,26,51}, {27,53,52}, {26,27,52}, {6,54,53}, {27,6,53}, {32,57,54}, {6,32,54}, {31,58,57}, {32,31,57}, {29,61,58}, {31,29,58}, {28,62,61}, {29,28,61}, {59,62,28}, {24,59,28}, {60,59,24}, {23,60,24}, {55,60,23}, {19,55,23}, {55,19,56}, {56,19,20}, {56,20,48}, {48,20,18}, {48,18,49}, {49,18,17}, {49,17,42}, {42,17,15}, {42,15,38}, {38,15,16}, {38,16,63}, {63,16,14}, {63,14,64}, {64,14,13}, {64,13,65}, {65,13,12}, {65,12,66}, {66,12,11}, {66,11,67}, {67,11,10}, {67,10,39}, {39,10,9} });
            break;
        case TestMesh::ipadstand:
            mesh = TriangleMesh(
            	{ Vec3f(17.4344673156738,-2.69879599481136e-16,9.5), {14.2814798355103,10,9.5}, {0,0,9.5}, {31.7159481048584,10,9.5}, Vec3f(62.2344741821289,2.06667568800577e-16,20), {31.7159481048584,10,20}, Vec3f(17.4344673156738,-2.69879599481136e-16,20), {62.2344741821289,10,20}, {98.2079696655273,10,0}, Vec3f(98.2079696655273,8.56525380796383e-16,10), {98.2079696655273,0,0}, {98.2079696655273,10,20}, {98.2079696655273,0,20}, Vec3f(81.6609649658203,-4.39753856997999e-16,10), {90.0549850463867,10,10}, {78.5079803466797,10,10}, Vec3f(93.2079696655273,8.56525380796383e-16,10), {14.2814798355103,10,20}, {0,0,20}, Vec3f(87.4344711303711,2.81343962782118e-15,20), {84.2814788818359,10,20}, {0,10,20}, {0,0,0}, {0,10,0}, Vec3f(62.2344741821289,2.06667568800577e-16,30), {66.9609756469727,10,30}, {62.2344741821289,10,30}, Vec3f(70.1139602661133,8.5525763717214e-16,30), {67.7053375244141,10,28.7107200622559}, Vec3f(71.6787109375,1.24046736339707e-15,27.2897701263428) },
				{ {0,1,2}, {1,0,3}, {4,5,6}, {5,4,7}, {8,9,10}, {9,11,12}, {11,9,8}, {13,14,15}, {14,13,16}, {17,2,1}, {2,17,18}, {19,11,20}, {11,19,12}, {17,21,18}, {21,2,18}, {2,21,22}, {22,21,23}, {8,22,23}, {22,8,10}, {24,25,26}, {25,24,27}, {23,1,8}, {1,23,21}, {1,21,17}, {5,15,3}, {15,5,7}, {15,7,28}, {28,7,26}, {28,26,25}, {8,14,11}, {14,8,3}, {3,8,1}, {14,3,15}, {11,14,20}, {26,4,24}, {4,26,7}, {12,16,9}, {16,12,19}, {29,4,13}, {4,29,24}, {24,29,27}, {9,22,10}, {22,9,0}, {0,9,16}, {0,16,13}, {0,13,6}, {6,13,4}, {2,22,0}, {19,14,16}, {14,19,20}, {15,29,13}, {29,25,27}, {25,29,15}, {25,15,28}, {6,3,0}, {3,6,5} });
            break;
        case TestMesh::A:
            mesh = TriangleMesh(
            	{ {513.075988769531,51.6074333190918,36.0009002685547}, {516.648803710938,51.7324333190918,36.0009002685547}, {513.495178222656,51.7324333190918,36.0009002685547}, {489.391204833984,51.4824333190918,24.0011005401611}, {488.928588867188,51.7324333190918,24.0011005401611}, {492.06201171875,51.7324333190918,24.0011005401611}, {496.840393066406,51.2324333190918,24.0011005401611}, {495.195404052734,51.7324333190918,24.0011005401611}, {498.981994628906,51.7324333190918,24.0011005401611}, {506.966613769531,51.6074333190918,24.0011005401611}, {510.342010498047,51.7324333190918,24.0011005401611}, {507.163818359375,51.6074333190918,24.0011005401611}, {512.515380859375,54.7190322875977,36.0009002685547}, {514.161987304688,54.5058326721191,36.0009002685547}, {493.06201171875,54.7190322875977,36.0009002685547}, {495.195404052734,51.7324333190918,36.0009002685547}, {496.195404052734,54.7190322875977,36.0009002685547}, {497.195404052734,57.7058334350586,36.0009002685547}, {500.851989746094,60.2658309936523,36.0009002685547}, {498.915405273438,62.8258323669434,36.0009002685547}, {506.701995849609,62.8258323669434,36.0009002685547}, {503.648590087891,60.2658309936523,36.0009002685547}, {508.381805419922,57.7058334350586,36.0009002685547}, {496.418792724609,60.052433013916,36.0009002685547}, {506.515197753906,72.2124328613281,36.0009002685547}, {502.808807373047,74.5324325561523,36.0009002685547}, {503.781982421875,71.6058349609375,36.0009002685547}, {515.358764648438,55.4658317565918,36.0009002685547}, {499.375183105469,76.9058380126953,36.0009002685547}, {501.168792724609,78.0658340454102,36.0009002685547}, {504.568786621094,78.0658340454102,36.0009002685547}, {506.32861328125,81.599235534668,36.0009002685547}, {502.928588867188,81.599235534668,36.0009002685547}, {499.528594970703,81.599235534668,36.0009002685547}, {498.20361328125,77.8658294677734,36.0009002685547}, {495.195404052734,51.7324333190918,30.0011005401611}, {498.981994628906,51.7324333190918,27.0011005401611}, {506.555206298828,51.7324333190918,33.0009002685547}, {506.555206298828,51.7324333190918,36.0009002685547}, {510.342010498047,51.7324333190918,36.0009002685547}, {512.515380859375,54.7190322875977,24.0011005401611}, {509.361999511719,54.7190322875977,24.0011005401611}, {508.381805419922,57.7058334350586,24.0011005401611}, {506.701995849609,62.8258323669434,24.0011005401611}, {509.188812255859,60.052433013916,24.0011005401611}, {493.06201171875,54.7190322875977,24.0011005401611}, {503.648590087891,60.2658309936523,24.0011005401611}, {500.851989746094,60.2658309936523,24.0011005401611}, {498.915405273438,62.8258323669434,24.0011005401611}, {502.808807373047,62.8258323669434,24.0011005401611}, {491.425201416016,54.5058326721191,24.0011005401611}, {506.421813964844,76.9058380126953,24.0011005401611}, {502.808807373047,74.5324325561523,24.0011005401611}, {504.568786621094,78.0658340454102,24.0011005401611}, {506.32861328125,81.599235534668,24.0011005401611}, {507.618804931641,77.8658294677734,24.0011005401611}, {499.221801757812,72.2124328613281,24.0011005401611}, {501.835388183594,71.6058349609375,24.0011005401611}, {501.168792724609,78.0658340454102,24.0011005401611}, {499.528594970703,81.599235534668,24.0011005401611}, {502.048583984375,79.8324356079102,24.0011005401611}, {490.253601074219,55.4658317565918,24.0011005401611}, {488.928588867188,51.7324333190918,30.0011005401611}, {488.928588867188,51.7324333190918,36.0009002685547}, {490.253601074219,55.4658317565918,31.5009002685547}, {498.20361328125,77.8658294677734,34.5009002685547}, {508.381805419922,57.7058334350586,30.0011005401611}, {505.585388183594,57.7058334350586,27.0011005401611}, {502.788818359375,57.7058334350586,36.0009002685547}, {499.992004394531,57.7058334350586,33.0009002685547}, {509.851989746094,53.2258338928223,33.0009002685547}, {509.361999511719,54.7190322875977,36.0009002685547}, {508.871795654297,56.2124328613281,27.0011005401611}, {496.695404052734,56.2124328613281,33.0009002685547}, {495.695404052734,53.2258338928223,27.0011005401611}, {506.32861328125,81.599235534668,30.0011005401611}, {507.618804931641,77.8658294677734,25.5011005401611}, {515.358764648438,55.4658317565918,34.5009002685547}, {501.228607177734,81.599235534668,33.0009002685547}, {504.628601074219,81.599235534668,27.0011005401611}, {503.781982421875,71.6058349609375,33.0009002685547}, {502.808807373047,74.5324325561523,30.0011005401611}, {498.915405273438,62.8258323669434,30.0011005401611}, {500.861999511719,62.8258323669434,27.0011005401611}, {502.808807373047,62.8258323669434,36.0009002685547}, {504.755187988281,62.8258323669434,33.0009002685547}, {501.835388183594,71.6058349609375,33.0009002685547}, {499.888793945312,65.7524337768555,33.0009002685547}, {499.888793945312,65.7524337768555,36.0009002685547}, {513.128601074219,51.4824333190918,36.0009002685547}, {513.075988769531,51.6074333190918,24.0011005401611}, {516.648803710938,51.7324333190918,24.0011005401611}, {513.128601074219,51.4824333190918,24.0011005401611}, {513.495178222656,51.7324333190918,24.0011005401611}, {506.966613769531,51.6074333190918,36.0009002685547}, {507.163818359375,51.6074333190918,36.0009002685547}, {490.337799072266,51.4824333190918,24.0011005401611}, {489.391204833984,51.4824333190918,36.0009002685547}, {492.06201171875,51.7324333190918,36.0009002685547}, {490.337799072266,51.4824333190918,36.0009002685547}, {513.233764648438,51.2324333190918,24.0011005401611}, {513.233764648438,51.2324333190918,36.0009002685547}, {504.773803710938,51.4824333190918,36.0009002685547}, {504.773803710938,51.4824333190918,24.0011005401611}, {489.266998291016,51.2324333190918,24.0011005401611}, {489.266998291016,51.2324333190918,36.0009002685547}, {490.253601074219,55.4658317565918,25.5011005401611}, {499.528594970703,81.599235534668,30.0011005401611}, {498.20361328125,77.8658294677734,31.5009002685547}, {515.358764648438,55.4658317565918,28.5011005401611}, {515.358764648438,55.4658317565918,25.5011005401611}, {495.246795654297,61.0124320983887,36.0009002685547}, {490.253601074219,55.4658317565918,34.5009002685547}, {490.253601074219,55.4658317565918,36.0009002685547}, {494.228607177734,66.6658325195312,24.0011005401611}, {499.068786621094,67.5192337036133,24.0011005401611}, {498.20361328125,77.8658294677734,25.5011005401611}, {498.20361328125,77.8658294677734,24.0011005401611}, {506.608795166016,67.5192337036133,36.0009002685547}, {509.09521484375,64.7458343505859,36.0009002685547}, {507.618804931641,77.8658294677734,34.5009002685547}, {507.618804931641,77.8658294677734,36.0009002685547}, {510.385406494141,61.0124320983887,24.0011005401611}, {515.358764648438,55.4658317565918,24.0011005401611}, {489.32861328125,47.7324333190918,31.5009002685547}, {492.95361328125,47.7324333190918,33.5634994506836}, {489.32861328125,47.7324333190918,34.5009002685547}, {489.32861328125,47.7324333190918,28.5011005401611}, {489.32861328125,47.7324333190918,25.5011005401611}, {492.95361328125,47.7324333190918,26.4385013580322}, {492.95361328125,47.7324333190918,30.5635013580322}, {492.95361328125,47.7324333190918,32.0634994506836}, {492.95361328125,47.7324333190918,31.3135013580322}, {492.95361328125,47.7324333190918,35.4384994506836}, {489.32861328125,47.7324333190918,36.0009002685547}, {492.95361328125,47.7324333190918,34.3134994506836}, {492.95361328125,47.7324333190918,34.6884994506836}, {492.95361328125,47.7324333190918,27.9385013580322}, {492.95361328125,47.7324333190918,28.6885013580322}, {492.95361328125,47.7324333190918,29.0635013580322}, {489.32861328125,47.7324333190918,24.0011005401611}, {492.95361328125,47.7324333190918,24.5635013580322}, {492.95361328125,47.7324333190918,25.6885013580322}, {492.95361328125,47.7324333190918,25.3135013580322}, {492.95361328125,47.7324333190918,24.1885013580322}, {492.95361328125,47.7324333190918,24.0011005401611}, {513.443786621094,50.7324333190918,24.0011005401611}, {492.95361328125,47.7324333190918,35.8134994506836}, {492.95361328125,47.7324333190918,36.0009002685547}, {513.443786621094,50.7324333190918,36.0009002685547}, {506.350402832031,51.4824333190918,36.0009002685547}, {506.350402832031,51.4824333190918,24.0011005401611}, {492.743804931641,48.2324333190918,24.0011005401611}, {492.638793945312,48.4824333190918,24.0011005401611}, {492.743804931641,48.2324333190918,36.0009002685547}, {492.638793945312,48.4824333190918,36.0009002685547}, {490.089599609375,50.9824333190918,36.0009002685547}, {490.089599609375,50.9824333190918,24.0011005401611}, {510.342010498047,51.7324333190918,30.0011005401611}, {499.068786621094,67.5192337036133,36.0009002685547}, {494.228607177734,66.6658325195312,36.0009002685547}, {499.375183105469,76.9058380126953,24.0011005401611}, {506.421813964844,76.9058380126953,36.0009002685547}, {506.608795166016,67.5192337036133,24.0011005401611}, {505.728607177734,65.7524337768555,24.0011005401611}, {509.09521484375,64.7458343505859,24.0011005401611}, {506.701995849609,62.8258323669434,30.0011005401611}, {505.728607177734,65.7524337768555,27.0011005401611}, {501.835388183594,71.6058349609375,27.0011005401611}, {499.888793945312,65.7524337768555,27.0011005401611}, {494.228607177734,66.6658325195312,30.0011005401611}, {495.553588867188,70.3992309570312,28.5011005401611}, {492.903594970703,62.9324340820312,28.5011005401611}, {495.553588867188,70.3992309570312,31.5009002685547}, {492.903594970703,62.9324340820312,31.5009002685547}, {511.488800048828,66.6658325195312,24.0011005401611}, {511.488800048828,66.6658325195312,30.0011005401611}, {512.778564453125,62.9324340820312,28.5011005401611}, {515.358764648438,55.4658317565918,31.5009002685547}, {507.618804931641,77.8658294677734,31.5009002685547}, {510.198791503906,70.3992309570312,28.5011005401611}, {511.488800048828,66.6658325195312,36.0009002685547}, {512.778564453125,62.9324340820312,31.5009002685547}, {510.198791503906,70.3992309570312,31.5009002685547}, {502.788818359375,57.7058334350586,24.0011005401611}, {497.195404052734,57.7058334350586,30.0011005401611}, {492.903594970703,62.9324340820312,34.5009002685547}, {492.903594970703,62.9324340820312,36.0009002685547}, {495.553588867188,70.3992309570312,24.0011005401611}, {496.725189208984,69.4392318725586,24.0011005401611}, {495.553588867188,70.3992309570312,25.5011005401611}, {495.246795654297,61.0124320983887,24.0011005401611}, {492.903594970703,62.9324340820312,25.5011005401611}, {492.903594970703,62.9324340820312,24.0011005401611}, {495.553588867188,70.3992309570312,36.0009002685547}, {496.725189208984,69.4392318725586,36.0009002685547}, {495.553588867188,70.3992309570312,34.5009002685547}, {510.198791503906,70.3992309570312,36.0009002685547}, {509.002014160156,69.4392318725586,36.0009002685547}, {510.198791503906,70.3992309570312,34.5009002685547}, {512.778564453125,62.9324340820312,25.5011005401611}, {512.778564453125,62.9324340820312,24.0011005401611}, {510.198791503906,70.3992309570312,24.0011005401611}, {509.002014160156,69.4392318725586,24.0011005401611}, {510.198791503906,70.3992309570312,25.5011005401611}, {510.385406494141,61.0124320983887,36.0009002685547}, {512.778564453125,62.9324340820312,34.5009002685547}, {512.778564453125,62.9324340820312,36.0009002685547}, {496.840393066406,51.2324333190918,36.0009002685547}, {498.981994628906,51.7324333190918,36.0009002685547}, {498.981994628906,51.7324333190918,33.0009002685547}, {506.555206298828,51.7324333190918,24.0011005401611}, {506.555206298828,51.7324333190918,27.0011005401611}, {503.82861328125,47.7324333190918,30.7509002685547}, {507.45361328125,47.7324333190918,32.8134994506836}, {503.82861328125,47.7324333190918,33.7509002685547}, {503.82861328125,47.7324333190918,29.2511005401611}, {503.82861328125,47.7324333190918,26.2511005401611}, {507.45361328125,47.7324333190918,27.1885013580322}, {493.921813964844,57.2792320251465,36.0009002685547}, {491.425201416016,54.5058326721191,36.0009002685547}, {497.195404052734,57.7058334350586,24.0011005401611}, {496.418792724609,60.052433013916,24.0011005401611}, {509.188812255859,60.052433013916,36.0009002685547}, {511.675415039062,57.2792320251465,24.0011005401611}, {514.161987304688,54.5058326721191,24.0011005401611}, {507.45361328125,47.7324333190918,34.3134994506836}, {503.82861328125,47.7324333190918,35.2509002685547}, {507.45361328125,47.7324333190918,25.6885013580322}, {503.82861328125,47.7324333190918,24.7511005401611}, {500.20361328125,47.7324333190918,31.6885013580322}, {500.20361328125,47.7324333190918,28.3135013580322}, {500.20361328125,47.7324333190918,30.1885013580322}, {507.45361328125,47.7324333190918,29.8135013580322}, {507.45361328125,47.7324333190918,31.3135013580322}, {507.45361328125,47.7324333190918,30.5635013580322}, {503.82861328125,47.7324333190918,36.0009002685547}, {507.45361328125,47.7324333190918,35.4384994506836}, {507.45361328125,47.7324333190918,35.0634994506836}, {507.45361328125,47.7324333190918,28.6885013580322}, {507.45361328125,47.7324333190918,29.4385013580322}, {503.82861328125,47.7324333190918,24.0011005401611}, {507.45361328125,47.7324333190918,24.5635013580322}, {507.45361328125,47.7324333190918,24.9385013580322}, {500.20361328125,47.7324333190918,34.6884994506836}, {500.20361328125,47.7324333190918,33.1884994506836}, {500.20361328125,47.7324333190918,33.9384994506836}, {500.20361328125,47.7324333190918,25.3135013580322}, {500.20361328125,47.7324333190918,26.8135013580322}, {500.20361328125,47.7324333190918,26.0635013580322}, {500.20361328125,47.7324333190918,30.9385013580322}, {500.20361328125,47.7324333190918,35.0634994506836}, {500.20361328125,47.7324333190918,35.4384994506836}, {500.20361328125,47.7324333190918,29.0635013580322}, {500.20361328125,47.7324333190918,29.4385013580322}, {500.20361328125,47.7324333190918,24.9385013580322}, {500.20361328125,47.7324333190918,24.5635013580322}, {507.45361328125,47.7324333190918,24.1885013580322}, {507.45361328125,47.7324333190918,24.0011005401611}, {513.86376953125,49.7324333190918,24.0011005401611}, {507.45361328125,47.7324333190918,35.8134994506836}, {507.45361328125,47.7324333190918,36.0009002685547}, {513.86376953125,49.7324333190918,36.0009002685547}, {500.20361328125,47.7324333190918,24.1885013580322}, {500.20361328125,47.7324333190918,24.0011005401611}, {502.988800048828,49.7324333190918,24.0011005401611}, {500.20361328125,47.7324333190918,35.8134994506836}, {500.20361328125,47.7324333190918,36.0009002685547}, {502.988800048828,49.7324333190918,36.0009002685547}, {504.755187988281,62.8258323669434,27.0011005401611}, {499.205383300781,51.2324333190918,36.0009002685547}, {498.786193847656,51.1074333190918,36.0009002685547}, {502.358795166016,51.2324333190918,36.0009002685547}, {499.205383300781,51.2324333190918,24.0011005401611}, {502.358795166016,51.2324333190918,24.0011005401611}, {498.786193847656,51.1074333190918,24.0011005401611}, {502.568786621094,50.7324333190918,24.0011005401611}, {505.931213378906,51.3574333190918,24.0011005401611}, {509.503601074219,51.4824333190918,24.0011005401611}, {502.568786621094,50.7324333190918,36.0009002685547}, {505.931213378906,51.3574333190918,36.0009002685547}, {509.503601074219,51.4824333190918,36.0009002685547}, {499.048583984375,50.4824333190918,36.0009002685547}, {492.428588867188,48.9824333190918,36.0009002685547}, {499.048583984375,50.4824333190918,24.0011005401611}, {492.428588867188,48.9824333190918,24.0011005401611}, {506.088806152344,50.9824333190918,24.0011005401611}, {506.036010742188,51.1074333190918,24.0011005401611}, {506.088806152344,50.9824333190918,36.0009002685547}, {506.036010742188,51.1074333190918,36.0009002685547}, {498.891204833984,50.8574333190918,36.0009002685547}, {498.943786621094,50.7324333190918,36.0009002685547}, {498.891204833984,50.8574333190918,24.0011005401611}, {498.943786621094,50.7324333190918,24.0011005401611}, {499.573608398438,49.2324333190918,24.0011005401611}, {499.783813476562,48.7324333190918,24.0011005401611}, {499.573608398438,49.2324333190918,36.0009002685547}, {499.783813476562,48.7324333190918,36.0009002685547}, {506.403594970703,50.2324333190918,24.0011005401611}, {506.298797607422,50.4824333190918,24.0011005401611}, {506.403594970703,50.2324333190918,36.0009002685547}, {506.298797607422,50.4824333190918,36.0009002685547}, {501.228607177734,81.599235534668,27.0011005401611}, {502.928588867188,81.599235534668,24.0011005401611}, {499.2587890625,49.9824333190918,36.0009002685547}, {499.363800048828,49.7324333190918,36.0009002685547}, {499.2587890625,49.9824333190918,24.0011005401611}, {499.363800048828,49.7324333190918,24.0011005401611}, {496.695404052734,56.2124328613281,27.0011005401611}, {496.195404052734,54.7190322875977,24.0011005401611}, {509.851989746094,53.2258338928223,27.0011005401611}, {493.464782714844,51.1074333190918,36.0009002685547}, {493.464782714844,51.1074333190918,24.0011005401611}, {502.768798828125,51.7324333190918,24.0011005401611}, {500.215789794922,51.3574333190918,24.0011005401611}, {497.628601074219,51.2324333190918,24.0011005401611}, {502.768798828125,51.7324333190918,36.0009002685547}, {500.215789794922,51.3574333190918,36.0009002685547}, {497.628601074219,51.2324333190918,36.0009002685547}, {507.033813476562,48.7324333190918,24.0011005401611}, {506.823791503906,49.2324333190918,24.0011005401611}, {507.033813476562,48.7324333190918,36.0009002685547}, {506.823791503906,49.2324333190918,36.0009002685547}, {494.4501953125,51.1074333190918,24.0011005401611}, {494.4501953125,51.1074333190918,36.0009002685547}, {500.807006835938,51.3574333190918,36.0009002685547}, {503.591186523438,51.4824333190918,36.0009002685547}, {503.591186523438,51.4824333190918,24.0011005401611}, {500.807006835938,51.3574333190918,24.0011005401611}, {505.728607177734,65.7524337768555,36.0009002685547}, {505.728607177734,65.7524337768555,33.0009002685547}, {499.221801757812,72.2124328613281,36.0009002685547}, {501.835388183594,71.6058349609375,36.0009002685547}, {506.515197753906,72.2124328613281,24.0011005401611}, {503.781982421875,71.6058349609375,24.0011005401611}, {503.781982421875,71.6058349609375,27.0011005401611}, {499.888793945312,65.7524337768555,24.0011005401611}, {495.695404052734,53.2258338928223,33.0009002685547}, {516.648803710938,51.7324333190918,30.0011005401611}, {498.20361328125,77.8658294677734,28.5011005401611}, {505.585388183594,57.7058334350586,33.0009002685547}, {508.871795654297,56.2124328613281,33.0009002685547}, {499.992004394531,57.7058334350586,27.0011005401611}, {504.628601074219,81.599235534668,33.0009002685547}, {500.861999511719,62.8258323669434,33.0009002685547}, {496.878601074219,74.1324310302734,27.0011005401611}, {496.878601074219,74.1324310302734,33.0009002685547}, {491.57861328125,59.199031829834,27.0011005401611}, {490.253601074219,55.4658317565918,28.5011005401611}, {491.57861328125,59.199031829834,33.0009002685547}, {514.068786621094,59.199031829834,27.0011005401611}, {514.068786621094,59.199031829834,33.0009002685547}, {508.908813476562,74.1324310302734,27.0011005401611}, {507.618804931641,77.8658294677734,28.5011005401611}, {508.908813476562,74.1324310302734,33.0009002685547}, {491.271789550781,50.9824333190918,36.0009002685547}, {490.877807617188,50.9824333190918,36.0009002685547}, {491.271789550781,50.9824333190918,24.0011005401611}, {490.877807617188,50.9824333190918,24.0011005401611}, {495.213806152344,50.9824333190918,36.0009002685547}, {493.636993408203,50.9824333190918,36.0009002685547}, {495.213806152344,50.9824333190918,24.0011005401611}, {493.636993408203,50.9824333190918,24.0011005401611}, {503.985412597656,51.4824333190918,36.0009002685547}, {503.985412597656,51.4824333190918,24.0011005401611}, {511.675415039062,57.2792320251465,36.0009002685547}, {493.921813964844,57.2792320251465,24.0011005401611}, {502.768798828125,51.7324333190918,30.0011005401611}, {506.555206298828,51.7324333190918,30.0011005401611}, {498.981994628906,51.7324333190918,30.0011005401611}, {492.848815917969,50.9824333190918,24.0011005401611}, {492.848815917969,50.9824333190918,36.0009002685547}, {500.861999511719,68.6792297363281,36.0009002685547}, {500.861999511719,68.6792297363281,24.0011005401611}, {496.878601074219,74.1324310302734,24.0011005401611}, {496.878601074219,74.1324310302734,36.0009002685547}, {504.755187988281,68.6792297363281,24.0011005401611}, {504.755187988281,68.6792297363281,36.0009002685547}, {508.908813476562,74.1324310302734,36.0009002685547}, {508.908813476562,74.1324310302734,24.0011005401611}, {505.728607177734,65.7524337768555,30.0011005401611}, {504.755187988281,68.6792297363281,30.0011005401611}, {503.781982421875,71.6058349609375,30.0011005401611}, {500.861999511719,68.6792297363281,30.0011005401611}, {499.888793945312,65.7524337768555,30.0011005401611}, {501.835388183594,71.6058349609375,30.0011005401611}, {491.57861328125,59.199031829834,24.0011005401611}, {491.57861328125,59.199031829834,36.0009002685547}, {514.068786621094,59.199031829834,36.0009002685547}, {514.068786621094,59.199031829834,24.0011005401611}, {511.07861328125,47.7324333190918,34.8759002685547}, {511.07861328125,47.7324333190918,31.8759002685547}, {514.70361328125,47.7324333190918,33.9384994506836}, {511.07861328125,47.7324333190918,25.1261005401611}, {514.70361328125,47.7324333190918,26.0635013580322}, {511.07861328125,47.7324333190918,28.1261005401611}, {502.788818359375,57.7058334350586,30.0011005401611}, {502.048583984375,79.8324356079102,36.0009002685547}, {514.70361328125,47.7324333190918,30.9385013580322}, {511.07861328125,47.7324333190918,30.3759002685547}, {514.70361328125,47.7324333190918,29.0635013580322}, {511.07861328125,47.7324333190918,29.6261005401611}, {496.57861328125,47.7324333190918,31.1259002685547}, {496.57861328125,47.7324333190918,32.6259002685547}, {496.57861328125,47.7324333190918,34.1259002685547}, {496.57861328125,47.7324333190918,28.8761005401611}, {496.57861328125,47.7324333190918,27.3761005401611}, {496.57861328125,47.7324333190918,25.8761005401611}, {496.57861328125,47.7324333190918,29.6261005401611}, {514.70361328125,47.7324333190918,35.4384994506836}, {511.07861328125,47.7324333190918,35.6259002685547}, {514.70361328125,47.7324333190918,24.5635013580322}, {511.07861328125,47.7324333190918,24.3761005401611}, {496.57861328125,47.7324333190918,34.8759002685547}, {496.57861328125,47.7324333190918,25.1261005401611}, {496.57861328125,47.7324333190918,35.6259002685547}, {496.57861328125,47.7324333190918,24.3761005401611}, {511.07861328125,47.7324333190918,36.0009002685547}, {511.07861328125,47.7324333190918,24.0011005401611}, {514.70361328125,47.7324333190918,30.1885013580322}, {514.70361328125,47.7324333190918,35.8134994506836}, {514.70361328125,47.7324333190918,29.8135013580322}, {514.70361328125,47.7324333190918,24.1885013580322}, {496.57861328125,47.7324333190918,36.0009002685547}, {496.57861328125,47.7324333190918,24.0011005401611}, {510.238800048828,49.7324333190918,24.0011005401611}, {510.238800048828,49.7324333190918,36.0009002685547}, {514.70361328125,47.7324333190918,24.0011005401611}, {514.70361328125,47.7324333190918,36.0009002685547}, {496.158813476562,48.7324333190918,36.0009002685547}, {496.158813476562,48.7324333190918,24.0011005401611}, {502.808807373047,62.8258323669434,30.0011005401611}, {509.608795166016,51.2324333190918,24.0011005401611}, {509.608795166016,51.2324333190918,36.0009002685547}, {491.641204833984,50.8574333190918,24.0011005401611}, {495.423797607422,50.4824333190918,36.0009002685547}, {495.423797607422,50.4824333190918,24.0011005401611}, {491.641204833984,50.8574333190918,36.0009002685547}, {495.528594970703,50.2324333190918,24.0011005401611}, {492.0087890625,49.9824333190918,24.0011005401611}, {509.818786621094,50.7324333190918,24.0011005401611}, {495.948608398438,49.2324333190918,36.0009002685547}, {495.528594970703,50.2324333190918,36.0009002685547}, {495.948608398438,49.2324333190918,24.0011005401611}, {509.818786621094,50.7324333190918,36.0009002685547}, {492.0087890625,49.9824333190918,36.0009002685547}, {491.956207275391,50.1074333190918,24.0011005401611}, {491.956207275391,50.1074333190918,36.0009002685547}, {502.928588867188,81.599235534668,30.0011005401611}, {491.851013183594,50.3574333190918,36.0009002685547}, {491.851013183594,50.3574333190918,24.0011005401611}, {496.195404052734,54.7190322875977,30.0011005401611}, {509.361999511719,54.7190322875977,30.0011005401611}, {488.632598876953,51.7256317138672,30.0011005401611}, {488.632598876953,51.7256317138672,29.5091018676758}, {488.632598876953,51.7188339233398,24.0011005401611}, {488.632598876953,51.7256317138672,27.4929008483887}, {488.632598876953,51.7324333190918,30.0011005401611}, {488.632598876953,51.7324333190918,29.0175018310547}, {488.632598876953,51.7324333190918,24.9847011566162}, {488.632598876953,51.7324333190918,24.0011005401611}, {488.632598876953,51.7188339233398,30.0011005401611}, {488.632598876953,51.7176322937012,24.0011005401611}, {488.632598876953,51.7182312011719,30.0011005401611}, {488.632598876953,51.7176322937012,30.0011005401611}, {488.632598876953,51.715030670166,24.0011005401611}, {488.632598876953,51.7162322998047,30.0011005401611}, {488.632598876953,50.761833190918,24.0011005401611}, {488.632598876953,50.7578315734863,24.0011005401611}, {488.632598876953,50.7598342895508,30.0011005401611}, {488.632598876953,50.7522315979004,24.0011005401611}, {488.632598876953,49.7838325500488,24.0011005401611}, {488.632598876953,50.2680320739746,30.0011005401611}, {488.632598876953,51.7046318054199,24.0011005401611}, {488.632598876953,51.709831237793,30.0011005401611}, {488.632598876953,50.9120330810547,24.0011005401611}, {488.632598876953,50.8882331848145,24.0011005401611}, {488.632598876953,50.9002304077148,30.0011005401611}, {488.632598876953,47.7324333190918,24.0370998382568}, {488.632598876953,48.5612335205078,30.0011005401611}, {488.632598876953,47.7324333190918,24.0011005401611}, {488.632598876953,47.7324333190918,24.1091003417969}, {488.632598876953,48.5612335205078,30.0189018249512}, {488.632598876953,47.7324333190918,25.3211002349854}, {488.632598876953,48.5612335205078,30.0551013946533}, {488.632598876953,47.7324333190918,25.4651012420654}, {488.632598876953,48.5612335205078,30.6609001159668}, {488.632598876953,47.7324333190918,25.5371017456055}, {488.632598876953,48.5612335205078,30.7329006195068}, {488.632598876953,47.7324333190918,25.6091003417969}, {488.632598876953,48.5612335205078,30.7689018249512}, {488.632598876953,47.7324333190918,25.8971004486084}, {488.632598876953,48.5612335205078,30.8051013946533}, {488.632598876953,47.7324333190918,28.321102142334}, {488.632598876953,48.5612335205078,30.9491004943848}, {488.632598876953,47.7324333190918,28.4651012420654}, {488.632598876953,48.5612335205078,32.1609001159668}, {488.632598876953,47.7324333190918,28.5371017456055}, {488.632598876953,48.5612335205078,32.2329025268555}, {488.632598876953,47.7324333190918,28.6811008453369}, {488.632598876953,48.5612335205078,32.2689018249512}, {488.632598876953,47.7324333190918,31.1049003601074}, {488.632598876953,48.5612335205078,32.3411026000977}, {488.632598876953,47.7324333190918,31.3929004669189}, {488.632598876953,49.3900299072266,36.0009002685547}, {488.632598876953,47.7324333190918,31.536901473999}, {488.632598876953,47.7324333190918,31.6809005737305}, {488.632598876953,47.7324333190918,34.1049003601074}, {488.632598876953,47.7324333190918,34.3929023742676}, {488.632598876953,47.7324333190918,34.464900970459}, {488.632598876953,47.7324333190918,34.5369033813477}, {488.632598876953,47.7324333190918,34.6809005737305}, {488.632598876953,47.7324333190918,35.8929023742676}, {488.632598876953,47.7324333190918,35.964900970459}, {488.632598876953,47.7324333190918,36.0009002685547}, {488.632598876953,50.8816299438477,24.0011005401611}, {488.632598876953,50.8850326538086,30.0011005401611}, {488.632598876953,49.7480316162109,24.0011005401611}, {488.632598876953,49.7426300048828,24.0011005401611}, {488.632598876953,49.745231628418,30.0011005401611}, {488.632598876953,49.7592315673828,24.0011005401611}, {488.632598876953,49.7536315917969,30.0011005401611}, {488.632598876953,49.3900299072266,24.0011005401611}, {488.632598876953,49.5664329528809,30.0011005401611}, {488.632598876953,50.8786315917969,24.0011005401611}, {488.632598876953,50.7764320373535,24.0011005401611}, {488.632598876953,50.8274307250977,30.0011005401611}, {488.632598876953,50.7550315856934,30.0011005401611}, {488.632598876953,50.7692337036133,30.0011005401611}, {488.632598876953,50.9284324645996,24.0011005401611}, {488.632598876953,50.9202308654785,30.0011005401611}, {488.632598876953,51.1788330078125,24.0011005401611}, {488.632598876953,51.139232635498,24.0011005401611}, {488.632598876953,51.1590309143066,30.0011005401611}, {488.632598876953,51.2324333190918,24.0011005401611}, {488.632598876953,51.2056312561035,30.0011005401611}, {488.632598876953,51.4340324401855,24.0011005401611}, {488.632598876953,51.3946304321289,24.0011005401611}, {488.632598876953,51.4142303466797,30.0011005401611}, {488.632598876953,51.4498329162598,24.0011005401611}, {488.632598876953,51.5772323608398,30.0011005401611}, {488.632598876953,51.4418334960938,30.0011005401611}, {488.632598876953,51.3136329650879,30.0011005401611}, {488.632598876953,49.7714309692383,30.0011005401611}, {488.632598876953,51.0338325500488,30.0011005401611}, {488.632598876953,50.8816299438477,30.0011005401611}, {488.632598876953,50.8800315856934,30.0011005401611}, {488.632598876953,51.7188339233398,36.0009002685547}, {488.632598876953,51.7176322937012,36.0009002685547}, {488.632598876953,49.3900299072266,30.0011005401611}, {488.632598876953,50.7522315979004,30.0011005401611}, {488.632598876953,50.7522315979004,36.0009002685547}, {488.632598876953,49.7426300048828,30.0011005401611}, {488.632598876953,49.7426300048828,36.0009002685547}, {488.632598876953,49.7480316162109,30.0011005401611}, {488.632598876953,49.7480316162109,36.0009002685547}, {488.632598876953,51.715030670166,30.0011005401611}, {488.632598876953,51.715030670166,36.0009002685547}, {488.632598876953,50.7578315734863,30.0011005401611}, {488.632598876953,50.7578315734863,36.0009002685547}, {488.632598876953,50.761833190918,30.0011005401611}, {488.632598876953,50.761833190918,36.0009002685547}, {488.632598876953,50.8882331848145,30.0011005401611}, {488.632598876953,50.8882331848145,36.0009002685547}, {488.632598876953,49.7592315673828,30.0011005401611}, {488.632598876953,49.7592315673828,36.0009002685547}, {488.632598876953,51.1788330078125,30.0011005401611}, {488.632598876953,51.1788330078125,36.0009002685547}, {488.632598876953,50.9120330810547,30.0011005401611}, {488.632598876953,50.9120330810547,36.0009002685547}, {488.632598876953,51.4498329162598,30.0011005401611}, {488.632598876953,51.4498329162598,36.0009002685547}, {488.632598876953,51.7046318054199,30.0011005401611}, {488.632598876953,51.7046318054199,36.0009002685547}, {488.632598876953,51.2324333190918,30.0011005401611}, {488.632598876953,51.2324333190918,36.0009002685547}, {488.632598876953,51.3946304321289,30.0011005401611}, {488.632598876953,51.3946304321289,36.0009002685547}, {488.632598876953,51.4340324401855,30.0011005401611}, {488.632598876953,51.4340324401855,36.0009002685547}, {488.632598876953,49.7838325500488,30.0011005401611}, {488.632598876953,49.7838325500488,36.0009002685547}, {488.632598876953,50.7764320373535,30.0011005401611}, {488.632598876953,50.7764320373535,36.0009002685547}, {488.632598876953,51.139232635498,30.0011005401611}, {488.632598876953,51.139232635498,36.0009002685547}, {488.632598876953,50.9284324645996,30.0011005401611}, {488.632598876953,50.9284324645996,36.0009002685547}, {488.632598876953,50.8816299438477,36.0009002685547}, {488.632598876953,50.8786315917969,30.0011005401611}, {488.632598876953,50.8786315917969,36.0009002685547}, {488.632598876953,51.7324333190918,35.0173034667969}, {488.632598876953,51.7324333190918,36.0009002685547}, {488.632598876953,51.7324333190918,30.9847011566162}, {517.188415527344,51.7140884399414,24.0011005401611}, {517.188415527344,51.7140884399414,36.0009002685547}, {517.188415527344,50.4475173950195,24.0011005401611}, {517.188415527344,51.7324333190918,35.3734130859375}, {517.188415527344,51.7324333190918,36.0009002685547}, {517.188415527344,51.7324333190918,34.1185760498047}, {517.188415527344,51.7324333190918,31.88330078125}, {517.188415527344,51.7324333190918,30.0011005401611}, {517.188415527344,51.7324333190918,28.1187744140625}, {517.188415527344,51.7324333190918,25.8834266662598}, {517.188415527344,51.7324333190918,24.6285915374756}, {517.188415527344,51.7324333190918,24.0011005401611}, {517.188415527344,47.7324333190918,24.0600452423096}, {517.188415527344,47.7324333190918,24.0011005401611}, {517.188415527344,50.4475173950195,36.0009002685547}, {517.188415527344,47.7324333190918,24.1779975891113}, {517.188415527344,47.7324333190918,24.6498031616211}, {517.188415527344,47.7324333190918,28.7625770568848}, {517.188415527344,47.7324333190918,29.7061901092529}, {517.188415527344,47.7324333190918,29.9420928955078}, {517.188415527344,47.7324333190918,30.0600452423096}, {517.188415527344,47.7324333190918,30.2959480285645}, {517.188415527344,47.7324333190918,31.2395629882812}, {517.188415527344,47.7324333190918,35.3521995544434}, {517.188415527344,47.7324333190918,35.8240051269531}, {517.188415527344,47.7324333190918,35.9419555664062}, {517.188415527344,47.7324333190918,36.0009002685547} },
				{ {0,1,2}, {3,4,5}, {6,7,8}, {9,10,11}, {12,2,1}, {12,1,13}, {14,15,16}, {17,18,19}, {20,21,22}, {17,19,23}, {24,25,26}, {27,13,1}, {28,25,29}, {30,31,32}, {28,33,34}, {35,36,7}, {37,38,39}, {40,10,41}, {42,43,44}, {45,5,4}, {46,47,48}, {46,48,49}, {45,4,50}, {51,52,53}, {51,54,55}, {56,52,57}, {58,59,60}, {61,50,4}, {62,63,64}, {65,34,33}, {66,67,42}, {68,17,69}, {70,71,22}, {66,42,72}, {73,16,15}, {35,7,74}, {75,76,54}, {77,27,1}, {78,32,31}, {75,54,79}, {80,26,25}, {81,80,25}, {82,83,48}, {84,20,85}, {81,25,86}, {87,88,19}, {0,89,1}, {90,91,92}, {90,10,93}, {38,94,39}, {94,95,39}, {3,7,96}, {97,15,98}, {97,99,15}, {92,91,100}, {89,101,1}, {102,39,95}, {103,11,10}, {104,96,7}, {105,15,99}, {106,61,4}, {107,108,33}, {76,55,54}, {109,91,110}, {111,23,19}, {112,63,113}, {114,115,48}, {116,59,117}, {118,20,119}, {120,31,121}, {122,44,43}, {110,91,123}, {124,125,126}, {127,128,129}, {127,130,124}, {131,124,132}, {126,133,134}, {135,136,126}, {137,138,127}, {139,127,138}, {128,140,141}, {142,128,143}, {144,140,145}, {100,91,146}, {147,148,134}, {101,149,1}, {102,150,39}, {103,10,151}, {145,140,152}, {152,140,153}, {148,154,134}, {154,155,134}, {156,15,105}, {157,104,7}, {36,8,7}, {158,37,39}, {159,19,88}, {160,19,159}, {161,59,58}, {161,117,59}, {162,31,30}, {162,121,31}, {163,43,164}, {163,165,43}, {166,167,43}, {167,164,43}, {168,57,52}, {82,48,169}, {114,170,171}, {108,65,33}, {64,63,112}, {114,172,170}, {160,173,170}, {171,170,173}, {172,174,170}, {160,170,174}, {175,176,177}, {178,77,1}, {179,31,120}, {175,180,176}, {181,182,176}, {177,176,182}, {180,183,176}, {181,176,183}, {184,42,67}, {185,69,17}, {160,111,19}, {186,187,160}, {188,189,114}, {190,188,114}, {114,48,191}, {192,114,193}, {194,160,195}, {196,160,194}, {197,198,181}, {199,197,181}, {122,43,165}, {200,201,175}, {202,175,203}, {204,175,202}, {205,119,20}, {206,181,207}, {208,209,15}, {210,15,209}, {211,10,9}, {212,10,211}, {213,214,215}, {216,217,218}, {219,14,17}, {113,63,220}, {221,222,48}, {191,48,222}, {22,223,20}, {205,20,223}, {224,40,42}, {123,91,225}, {214,226,215}, {227,215,226}, {218,217,228}, {229,228,217}, {215,230,213}, {125,135,126}, {217,216,231}, {129,128,142}, {216,213,232}, {130,132,124}, {213,216,233}, {234,213,235}, {236,227,237}, {238,237,227}, {239,240,216}, {233,216,240}, {241,242,229}, {243,229,242}, {215,227,244}, {245,215,246}, {217,247,229}, {248,249,217}, {232,213,250}, {230,250,213}, {133,147,134}, {244,227,251}, {236,252,227}, {251,227,252}, {231,216,253}, {254,253,216}, {141,140,144}, {247,255,229}, {241,229,256}, {255,256,229}, {257,241,258}, {259,146,91}, {260,261,236}, {262,1,149}, {263,264,241}, {265,241,264}, {266,236,267}, {268,267,236}, {49,48,83}, {166,43,269}, {270,271,272}, {273,274,275}, {276,274,277}, {278,151,10}, {279,280,272}, {281,39,150}, {272,282,279}, {155,283,134}, {274,276,284}, {153,140,285}, {286,276,287}, {265,276,286}, {288,289,279}, {268,288,279}, {290,291,272}, {271,290,272}, {292,274,293}, {275,274,292}, {294,265,295}, {276,265,294}, {296,297,268}, {279,296,268}, {241,265,298}, {298,265,299}, {236,300,268}, {300,301,268}, {107,33,78}, {302,303,59}, {304,305,279}, {282,304,279}, {306,276,307}, {284,276,306}, {185,17,73}, {308,309,221}, {158,39,70}, {310,41,10}, {15,311,208}, {7,6,312}, {313,314,6}, {315,6,314}, {316,208,317}, {318,317,208}, {258,241,319}, {319,241,320}, {261,321,236}, {321,322,236}, {6,315,323}, {208,324,318}, {270,325,318}, {326,318,325}, {327,328,315}, {273,315,328}, {118,329,20}, {330,20,329}, {331,332,25}, {86,25,332}, {333,334,52}, {335,52,334}, {115,336,48}, {169,48,336}, {62,106,4}, {35,15,210}, {35,337,15}, {158,10,212}, {158,310,10}, {338,178,1}, {339,59,116}, {107,302,59}, {66,22,340}, {66,341,22}, {185,221,342}, {185,308,221}, {75,31,179}, {75,343,31}, {166,20,330}, {166,85,20}, {81,52,335}, {81,168,52}, {82,19,344}, {82,87,19}, {108,339,345}, {346,108,345}, {64,347,348}, {349,347,64}, {178,109,350}, {351,178,350}, {179,352,353}, {354,352,179}, {355,208,356}, {356,208,311}, {357,358,6}, {358,312,6}, {68,22,21}, {68,340,22}, {221,48,47}, {184,342,221}, {359,270,360}, {318,360,270}, {361,362,273}, {315,273,362}, {272,102,270}, {363,270,102}, {274,273,103}, {364,103,273}, {21,19,18}, {21,20,84}, {184,46,42}, {43,42,46}, {12,22,71}, {365,22,12}, {14,98,15}, {14,220,63}, {40,93,10}, {40,225,91}, {45,221,309}, {366,221,45}, {313,367,212}, {212,367,368}, {36,369,367}, {313,36,367}, {316,37,367}, {37,368,367}, {210,367,369}, {316,367,210}, {362,370,315}, {370,323,315}, {360,318,371}, {371,318,324}, {372,331,159}, {159,195,160}, {373,115,56}, {115,114,189}, {52,56,161}, {374,161,56}, {25,28,331}, {375,331,28}, {376,333,163}, {163,203,175}, {377,118,24}, {118,181,198}, {25,24,162}, {378,162,24}, {52,51,333}, {379,333,51}, {167,380,381}, {376,167,381}, {377,381,330}, {330,381,380}, {335,381,382}, {376,381,335}, {373,383,169}, {169,383,384}, {168,385,383}, {373,168,383}, {372,87,383}, {87,384,383}, {377,80,381}, {80,382,381}, {86,383,385}, {372,383,86}, {106,348,347}, {386,106,347}, {375,65,346}, {108,346,65}, {64,112,349}, {387,349,112}, {171,190,114}, {346,345,171}, {374,190,345}, {171,345,190}, {349,172,347}, {172,114,192}, {386,347,192}, {172,192,347}, {173,160,196}, {171,173,346}, {375,346,196}, {173,196,346}, {172,349,174}, {174,186,160}, {387,186,349}, {174,349,186}, {64,348,62}, {106,62,348}, {108,107,339}, {59,339,107}, {374,345,116}, {339,116,345}, {76,353,352}, {379,76,352}, {388,77,351}, {178,351,77}, {179,120,354}, {378,354,120}, {177,200,175}, {351,350,177}, {389,200,350}, {177,350,200}, {354,180,352}, {180,175,204}, {379,352,204}, {180,204,352}, {182,181,206}, {177,182,351}, {388,351,206}, {182,206,351}, {180,354,183}, {183,199,181}, {378,199,354}, {183,354,199}, {91,109,338}, {178,338,109}, {76,75,353}, {179,353,75}, {389,350,110}, {109,110,350}, {390,391,392}, {393,394,395}, {224,122,389}, {122,175,201}, {365,388,205}, {205,207,181}, {66,340,396}, {68,396,340}, {184,396,342}, {185,342,396}, {66,396,67}, {184,67,396}, {68,69,396}, {185,396,69}, {219,111,387}, {111,160,187}, {366,386,191}, {191,193,114}, {150,272,280}, {102,272,150}, {151,277,274}, {103,151,274}, {161,374,117}, {116,117,374}, {366,61,386}, {106,386,61}, {111,187,387}, {186,387,187}, {56,188,374}, {190,374,188}, {191,386,193}, {192,193,386}, {331,375,194}, {196,194,375}, {28,34,375}, {65,375,34}, {219,387,113}, {112,113,387}, {224,389,123}, {110,123,389}, {51,55,379}, {76,379,55}, {24,197,378}, {199,378,197}, {122,201,389}, {200,389,201}, {333,379,202}, {204,202,379}, {205,388,207}, {206,207,388}, {365,27,388}, {77,388,27}, {162,378,121}, {120,121,378}, {162,30,25}, {30,29,25}, {51,53,54}, {303,60,59}, {28,29,33}, {29,397,33}, {161,58,52}, {53,52,58}, {21,84,19}, {84,344,19}, {46,49,43}, {49,269,43}, {208,316,209}, {210,209,316}, {327,313,211}, {212,211,313}, {36,35,369}, {210,369,35}, {37,158,368}, {212,368,158}, {6,8,313}, {36,313,8}, {326,38,316}, {37,316,38}, {392,391,398}, {399,398,391}, {394,400,395}, {401,395,400}, {390,214,391}, {214,213,234}, {393,395,218}, {218,239,216}, {402,230,403}, {230,215,245}, {125,124,131}, {404,125,403}, {405,406,231}, {231,248,217}, {129,137,127}, {407,406,129}, {130,127,139}, {402,130,408}, {194,195,331}, {159,331,195}, {115,189,56}, {188,56,189}, {14,219,220}, {113,220,219}, {45,50,366}, {61,366,50}, {221,366,222}, {191,222,366}, {17,23,219}, {111,219,23}, {118,198,24}, {197,24,198}, {202,203,333}, {163,333,203}, {40,224,225}, {123,225,224}, {12,13,365}, {27,365,13}, {22,365,223}, {205,223,365}, {42,44,224}, {122,224,44}, {399,391,234}, {214,234,391}, {401,239,395}, {218,395,239}, {214,390,226}, {226,238,227}, {218,228,393}, {228,229,243}, {401,399,233}, {233,235,213}, {392,409,390}, {410,390,409}, {394,393,411}, {412,411,393}, {402,403,131}, {125,131,403}, {405,137,406}, {129,406,137}, {405,408,139}, {130,139,408}, {230,245,403}, {404,403,245}, {231,406,248}, {407,248,406}, {232,254,216}, {402,408,232}, {413,404,244}, {244,246,215}, {414,247,407}, {247,217,249}, {133,126,136}, {415,133,413}, {141,143,128}, {416,414,141}, {410,238,390}, {226,390,238}, {412,393,243}, {228,243,393}, {233,399,235}, {234,235,399}, {237,260,236}, {238,410,237}, {417,260,410}, {237,410,260}, {239,401,240}, {233,240,401}, {242,241,257}, {243,242,412}, {418,412,257}, {242,257,412}, {401,419,399}, {398,399,419}, {417,410,420}, {409,420,410}, {400,421,401}, {419,401,421}, {418,422,412}, {411,412,422}, {413,135,404}, {125,404,135}, {414,407,142}, {129,142,407}, {130,402,132}, {131,132,402}, {133,136,413}, {135,413,136}, {423,147,415}, {133,415,147}, {137,405,138}, {139,138,405}, {141,414,143}, {142,143,414}, {424,416,144}, {141,144,416}, {405,254,408}, {232,408,254}, {244,404,246}, {245,246,404}, {247,249,407}, {248,407,249}, {232,250,402}, {230,402,250}, {415,413,251}, {244,251,413}, {252,236,266}, {251,252,415}, {423,415,266}, {252,266,415}, {231,253,405}, {254,405,253}, {416,255,414}, {247,414,255}, {256,263,241}, {255,416,256}, {424,263,416}, {256,416,263}, {257,258,418}, {425,418,258}, {260,417,261}, {426,261,417}, {422,418,427}, {427,259,91}, {420,428,417}, {428,1,262}, {147,423,148}, {429,148,423}, {263,424,264}, {264,295,265}, {266,267,423}, {267,268,297}, {144,145,424}, {430,424,145}, {49,431,269}, {166,269,431}, {82,431,83}, {49,83,431}, {84,85,431}, {166,431,85}, {82,344,431}, {84,431,344}, {432,278,90}, {10,90,278}, {433,0,281}, {39,281,0}, {362,361,434}, {435,271,359}, {270,359,271}, {436,361,275}, {273,275,361}, {360,437,359}, {277,287,276}, {151,278,277}, {280,279,289}, {150,280,281}, {436,438,439}, {439,285,140}, {90,92,432}, {440,432,92}, {282,272,291}, {441,282,442}, {284,293,274}, {443,438,284}, {278,432,286}, {286,299,265}, {281,288,433}, {288,268,301}, {0,433,89}, {444,89,433}, {435,445,442}, {445,134,283}, {439,446,436}, {361,436,446}, {442,290,435}, {271,435,290}, {438,436,292}, {275,292,436}, {445,435,447}, {359,447,435}, {286,287,278}, {277,278,287}, {288,281,289}, {280,289,281}, {145,152,430}, {443,430,152}, {148,429,154}, {441,154,429}, {424,430,294}, {294,307,276}, {423,296,429}, {296,279,305}, {425,440,100}, {92,100,440}, {290,442,291}, {282,291,442}, {292,293,438}, {284,438,293}, {298,320,241}, {432,440,298}, {300,236,322}, {433,300,444}, {426,101,444}, {89,444,101}, {107,448,302}, {302,79,54}, {78,31,343}, {107,78,448}, {75,79,448}, {302,448,79}, {78,343,448}, {75,448,343}, {427,418,259}, {425,259,418}, {428,262,417}, {426,417,262}, {437,449,359}, {447,359,449}, {434,361,450}, {446,450,361}, {32,33,397}, {78,33,32}, {53,303,54}, {302,54,303}, {152,153,443}, {438,443,153}, {429,304,441}, {282,441,304}, {430,443,306}, {284,306,443}, {154,441,155}, {442,155,441}, {298,299,432}, {286,432,299}, {300,433,301}, {288,301,433}, {185,451,308}, {308,74,7}, {73,15,337}, {185,73,451}, {35,74,451}, {308,451,74}, {73,337,451}, {35,451,337}, {158,452,310}, {310,72,42}, {70,22,341}, {158,70,452}, {66,72,452}, {310,452,72}, {70,341,452}, {66,452,341}, {313,327,314}, {315,314,327}, {316,317,326}, {318,326,317}, {15,156,311}, {356,311,156}, {7,312,157}, {358,157,312}, {211,9,327}, {364,327,9}, {38,326,94}, {363,94,326}, {294,295,424}, {264,424,295}, {296,423,297}, {267,297,423}, {262,149,426}, {101,426,149}, {258,319,425}, {440,425,319}, {261,426,321}, {444,321,426}, {259,425,146}, {100,146,425}, {306,307,430}, {294,430,307}, {304,429,305}, {296,305,429}, {319,320,440}, {298,440,320}, {321,444,322}, {300,322,444}, {445,283,442}, {155,442,283}, {439,438,285}, {153,285,438}, {17,68,18}, {21,18,68}, {46,184,47}, {221,47,184}, {102,95,363}, {94,363,95}, {9,11,364}, {103,364,11}, {6,323,357}, {370,357,323}, {371,324,355}, {208,355,324}, {270,363,325}, {326,325,363}, {327,364,328}, {273,328,364}, {0,2,39}, {12,39,2}, {90,93,91}, {40,91,93}, {14,16,17}, {73,17,16}, {45,309,7}, {308,7,309}, {12,71,39}, {70,39,71}, {40,41,42}, {310,42,41}, {97,98,63}, {14,63,98}, {3,5,7}, {45,7,5}, {118,377,329}, {330,329,377}, {331,372,332}, {86,332,372}, {333,376,334}, {335,334,376}, {115,373,336}, {169,336,373}, {167,166,380}, {330,380,166}, {80,81,382}, {335,382,81}, {86,385,81}, {168,81,385}, {169,384,82}, {87,82,384}, {159,88,372}, {87,372,88}, {163,164,376}, {167,376,164}, {24,26,377}, {80,377,26}, {56,57,373}, {168,373,57}, {32,397,30}, {29,30,397}, {58,60,53}, {303,53,60}, {205,181,119}, {118,119,181}, {163,175,165}, {122,165,175}, {453,454,455}, {454,456,455}, {457,455,456}, {458,455,457}, {459,455,458}, {460,455,459}, {461,462,463}, {464,465,466}, {467,468,469}, {470,471,472}, {465,473,474}, {475,476,477}, {478,479,480}, {481,482,478}, {483,484,481}, {485,486,483}, {487,488,485}, {489,490,487}, {491,492,489}, {493,494,491}, {495,496,493}, {497,498,495}, {499,500,497}, {501,502,499}, {503,504,501}, {505,504,503}, {506,504,505}, {507,504,506}, {508,504,507}, {509,504,508}, {510,504,509}, {511,504,510}, {512,504,511}, {513,504,512}, {514,504,513}, {476,515,516}, {517,518,519}, {520,517,521}, {518,522,523}, {522,480,479}, {524,525,526}, {468,470,527}, {525,467,528}, {529,475,530}, {531,532,533}, {534,531,535}, {536,537,538}, {473,539,540}, {539,536,541}, {537,534,542}, {471,520,543}, {532,529,544}, {545,524,546}, {453,461,547}, {463,464,548}, {523,549,504}, {527,550,551}, {519,552,553}, {521,554,555}, {466,556,557}, {469,558,559}, {528,560,561}, {477,562,563}, {543,564,565}, {535,566,567}, {530,568,569}, {540,570,571}, {474,572,573}, {542,574,575}, {538,576,577}, {541,578,579}, {472,580,581}, {526,582,583}, {533,584,585}, {544,586,587}, {516,545,588}, {588,589,590}, {455,460,4}, {591,592,63}, {462,455,4}, {592,547,63}, {547,548,63}, {465,462,4}, {548,557,63}, {127,124,501}, {127,501,499}, {505,503,124}, {124,126,507}, {124,507,506}, {509,508,126}, {126,134,512}, {126,512,511}, {510,509,126}, {128,127,493}, {128,493,491}, {497,495,127}, {489,487,128}, {140,128,483}, {140,483,481}, {487,485,128}, {478,480,140}, {480,522,140}, {514,513,134}, {504,514,134}, {551,581,437}, {471,470,434}, {445,447,555}, {445,555,553}, {134,445,553}, {134,553,504}, {446,439,518}, {446,518,517}, {439,140,522}, {439,522,518}, {515,476,358}, {563,588,356}, {557,573,63}, {473,465,4}, {437,360,559}, {437,559,551}, {360,371,561}, {360,561,559}, {362,434,470}, {362,470,468}, {370,362,468}, {370,468,467}, {499,497,127}, {506,505,124}, {495,493,127}, {513,512,134}, {481,478,140}, {447,449,565}, {447,565,555}, {450,446,517}, {450,517,520}, {356,156,569}, {356,569,563}, {157,358,476}, {157,476,475}, {357,370,467}, {357,467,525}, {371,355,583}, {371,583,561}, {460,459,4}, {63,62,593}, {63,593,591}, {62,4,459}, {62,459,458}, {532,531,104}, {531,534,104}, {567,585,105}, {575,567,105}, {4,3,539}, {4,539,473}, {536,539,3}, {97,63,573}, {97,573,571}, {571,579,97}, {99,97,579}, {99,579,577}, {105,99,577}, {105,577,575}, {96,104,534}, {96,534,537}, {3,96,537}, {3,537,536}, {503,501,124}, {508,507,126}, {491,489,128}, {511,510,126}, {485,483,128}, {434,450,520}, {434,520,471}, {449,437,581}, {449,581,565}, {156,105,585}, {156,585,587}, {587,569,156}, {104,157,529}, {104,529,532}, {475,529,157}, {590,583,355}, {355,356,588}, {355,588,590}, {358,357,524}, {358,524,515}, {525,524,357}, {458,457,62}, {457,593,62}, {479,478,482}, {479,504,549}, {479,482,504}, {482,481,484}, {472,551,550}, {581,551,472}, {482,484,504}, {484,483,486}, {523,553,552}, {504,553,523}, {540,573,572}, {571,573,540}, {544,585,584}, {587,585,544}, {542,577,576}, {575,577,542}, {526,590,589}, {583,590,526}, {535,575,574}, {567,575,535}, {533,567,566}, {585,567,533}, {538,579,578}, {577,579,538}, {543,581,580}, {565,581,543}, {477,569,568}, {563,569,477}, {530,587,586}, {569,587,530}, {541,571,570}, {579,571,541}, {528,583,582}, {561,583,528}, {591,453,592}, {547,592,453}, {521,565,564}, {555,565,521}, {474,557,556}, {573,557,474}, {516,563,562}, {588,563,516}, {519,555,554}, {553,555,519}, {527,559,558}, {551,559,527}, {469,561,560}, {559,561,469}, {462,461,455}, {453,455,461}, {461,463,547}, {548,547,463}, {465,464,462}, {463,462,464}, {464,466,548}, {557,548,466}, {469,560,467}, {528,467,560}, {472,550,470}, {527,470,550}, {474,556,465}, {466,465,556}, {477,568,475}, {530,475,568}, {516,562,476}, {477,476,562}, {519,554,517}, {521,517,554}, {521,564,520}, {543,520,564}, {523,552,518}, {519,518,552}, {479,549,522}, {523,522,549}, {526,589,524}, {589,546,524}, {527,558,468}, {469,468,558}, {528,582,525}, {526,525,582}, {530,586,529}, {544,529,586}, {533,566,531}, {535,531,566}, {535,574,534}, {542,534,574}, {538,578,536}, {541,536,578}, {540,572,473}, {474,473,572}, {541,570,539}, {540,539,570}, {542,576,537}, {538,537,576}, {543,580,471}, {472,471,580}, {544,584,532}, {533,532,584}, {524,545,515}, {516,515,545}, {545,546,588}, {589,588,546}, {453,591,454}, {593,454,591}, {484,486,504}, {486,485,488}, {486,488,504}, {488,487,490}, {488,490,504}, {490,489,492}, {490,492,504}, {492,491,494}, {492,494,504}, {494,493,496}, {494,496,504}, {496,495,498}, {496,498,504}, {498,497,500}, {498,500,504}, {500,499,502}, {500,502,504}, {501,504,502}, {454,593,456}, {457,456,593}, {594,595,596}, {597,598,594}, {599,597,594}, {600,599,594}, {601,600,594}, {602,601,594}, {603,602,594}, {604,603,594}, {605,604,594}, {606,607,608}, {609,606,608}, {610,609,608}, {611,610,608}, {612,611,608}, {613,612,608}, {614,613,608}, {615,614,608}, {616,615,608}, {617,616,608}, {618,617,608}, {619,618,608}, {620,619,608}, {596,608,607}, {595,594,598}, {608,596,595}, {605,594,91}, {91,338,602}, {91,602,603}, {598,597,1}, {594,596,91}, {608,595,1}, {595,598,1}, {616,617,392}, {610,611,394}, {419,421,613}, {419,613,614}, {422,427,607}, {422,607,606}, {427,91,596}, {427,596,607}, {428,420,619}, {428,619,620}, {1,428,620}, {1,620,608}, {420,409,618}, {420,618,619}, {411,422,606}, {411,606,609}, {398,419,614}, {398,614,615}, {421,400,612}, {421,612,613}, {409,392,617}, {409,617,618}, {394,411,609}, {394,609,610}, {604,605,91}, {338,1,599}, {338,599,600}, {392,398,615}, {392,615,616}, {400,394,611}, {400,611,612}, {603,604,91}, {601,602,338}, {597,599,1}, {600,601,338} });
            break;
        case TestMesh::gt2_teeth:
            mesh = TriangleMesh(
            	{ {15.8899993896484,19.444055557251,2.67489433288574}, {15.9129991531372,19.1590557098389,2.67489433288574}, {15.9039993286133,19.1500549316406,2.67489433288574}, {15.9489994049072,19.2490558624268,2.67489433288574},
                  {15.9579992294312,19.3570556640625,2.67489433288574}, {15.8819999694824,18.690055847168,2.67489433288574}, {15.8319997787476,17.7460556030273,2.67489433288574}, {15.8489999771118,18.819055557251,2.67489433288574},
                  {15.8589992523193,17.7190551757812,2.67489433288574}, {15.8769998550415,19.0490550994873,2.67489433288574}, {15.7529993057251,17.8080558776855,2.67489433288574}, {15.7869997024536,19.5010547637939,2.67489433288574},
                  {14.0329990386963,18.7170543670654,2.67489433288574}, {13.9599990844727,18.7460556030273,2.67489433288574}, {13.9869995117188,20.2840557098389,2.67489433288574}, {14.2029991149902,20.149055480957,2.67489433288574},
                  {14.1939992904663,19.9560546875,2.67489433288574}, {14.1939992904663,20.1670551300049,2.67489433288574}, {14.2119998931885,20.0590553283691,2.67489433288574}, {12.1899995803833,19.1840553283691,2.67489433288574},
                  {12.096999168396,19.1950550079346,2.67489433288574}, {12.1099996566772,20.6690559387207,2.67489433288574}, {11.382999420166,19.9750556945801,2.67489433288574}, {11.2599992752075,19.2490558624268,2.67489433288574}, 
                  {11.2369995117188,19.9320545196533,2.67489433288574}, {11.5349998474121,20.0640544891357,2.67489433288574}, {11.6259994506836,20.1550559997559,2.67489433288574}, {11.6829986572266,20.2390556335449,2.67489433288574},
                  {11.7369995117188,20.3570556640625,2.67489433288574}, {11.8449993133545,20.645055770874,2.67489433288574}, {11.7729988098145,20.4640560150146,2.67489433288574}, {11.7799987792969,20.5370559692383,9.41389465332031}, 
                  {11.7639999389648,20.4470558166504,2.67489433288574}, {11.9559993743896,20.6810550689697,2.67489433288574}, {12.3079996109009,20.6020545959473,2.67489433288574}, {12.1959991455078,19.1860542297363,2.67489433288574},
                  {12.2059993743896,20.6540546417236,2.67489433288574}, {12.3489990234375,20.3740558624268,2.67489433288574}, {12.3579998016357,20.2750549316406,2.67489433288574}, {12.3669996261597,20.266056060791,2.67489433288574},
                  {12.3849992752075,20.1670551300049,2.67489433288574}, {12.4269990921021,20.0680541992188,2.67489433288574}, {12.5029993057251,19.9540557861328,2.67489433288574}, {12.6169996261597,19.8550548553467,2.67489433288574},
                  {12.7449989318848,19.7800559997559,2.67489433288574}, {12.7629995346069,19.7800559997559,2.67489433288574}, {12.8799991607666,19.7350559234619,2.67489433288574}, {13.0369997024536,19.7250556945801,2.67489433288574}, 
                  {13.0149993896484,19.0340557098389,2.67489433288574}, {11.1699991226196,19.2580547332764,2.67489433288574}, {11.0959987640381,19.2580547332764,2.67489433288574}, {11.1209993362427,19.9230556488037,2.67489433288574}, 
                  {13.0599994659424,19.024055480957,2.67489433288574}, {14.9049997329712,18.3170547485352,2.67489433288574}, {14.8779993057251,18.3400554656982,2.67489433288574}, {14.8779993057251,19.149055480957,2.67489433288574},
                  {13.3039989471436,19.77805519104,2.67489433288574}, {13.1589994430542,18.9890556335449,2.67489433288574}, {13.1559991836548,19.7350559234619,2.67489433288574}, {13.4269990921021,19.8600559234619,2.67489433288574},
                  {13.5339994430542,19.9700546264648,2.67389440536499}, {13.6359996795654,20.1220550537109,2.67489433288574}, {13.6359996795654,20.1400547027588,2.67489433288574}, {13.6719989776611,20.2210559844971,2.67489433288574},
                  {13.6899995803833,20.2300548553467,2.67489433288574}, {13.7509994506836,20.3010559082031,2.67489433288574}, {13.8539991378784,20.3180541992188,2.67489433288574}, {14.8329992294312,18.3580551147461,2.67489433288574},
                  {14.1849994659424,19.8530559539795,2.67489433288574}, {14.0769996643066,18.7000541687012,2.67489433288574}, {14.1099996566772,20.2400550842285,2.67489433288574}, {14.2009992599487,19.6230545043945,2.67489433288574},
                  {14.2729997634888,19.4670543670654,2.67489433288574}, {14.3379993438721,19.3790550231934,2.67489433288574}, {14.4549999237061,19.2770557403564,2.67489433288574}, {14.5899991989136,19.2040557861328,2.67489433288574},
                  {14.6079998016357,19.2040557861328,2.67489433288574}, {14.7209997177124,19.1600551605225,2.67489433288574}, {15.1379995346069,19.210054397583,2.67489433288574}, {14.9949998855591,18.2680549621582,2.67489433288574},
                  {15.0029993057251,19.1580543518066,2.67489433288574}, {15.2369995117188,19.2760543823242,2.67489433288574}, {15.3779993057251,19.4060554504395,2.67489433288574}, {15.4539995193481,19.520055770874,2.67489433288574},
                  {15.471999168396,19.52805519104,2.67489433288574}, {15.5449991226196,19.5830554962158,2.67489433288574}, {15.6529998779297,19.573055267334,2.67489433288574}, {15.7059993743896,17.8360557556152,2.67489433288574},
                  {15.9449996948242,18.5560550689697,2.67489433288574}, {15.8589992523193,18.9380550384521,2.67489433288574}, {14.9589996337891,18.2950553894043,2.67489433288574}, {15.7779998779297,19.5100555419922,2.67489433288574},
                  {14.0049991607666,20.2750549316406,2.67489433288574}, {12.3489990234375,20.5000553131104,2.67489433288574}, {13.0689992904663,19.0150547027588,2.67489433288574}, {13.0999994277954,19.0100555419922,2.67489433288574},
                  {15.9489994049072,19.3670558929443,9.41489505767822}, {15.9489994049072,19.2490558624268,9.41489505767822}, {15.75,17.8080558776855,9.41489505767822}, {15.6639995574951,19.5710544586182,9.41489505767822},
                  {15.5709991455078,17.9260559082031,9.41489505767822}, {15.8769998550415,18.690055847168,9.41489505767822}, {15.8499994277954,18.8170547485352,9.41489505767822}, {15.9459991455078,18.5520553588867,9.41489505767822},
                  {15.914999961853,17.6890544891357,9.41489505767822}, {15.3999996185303,19.4290542602539,9.41489505767822}, {15.3099994659424,19.339054107666,9.41489505767822}, {15.3729991912842,18.0440559387207,9.41489505767822},
                  {15.4579992294312,19.5170555114746,9.41489505767822}, {15.5469999313354,19.5820541381836,9.41489505767822}, {13.2309989929199,19.7610549926758,9.41489505767822}, {13.168999671936,19.7360553741455,9.41489505767822},
                  {13.096999168396,19.0140552520752,9.41489505767822}, {13.1999988555908,18.9870548248291,9.41489505767822}, {15.1399993896484,19.2080554962158,9.41489505767822}, {15.0159997940063,19.1600551605225,9.41489505767822},
                  {14.9859991073608,18.2770557403564,9.41489505767822}, {15.1749992370605,18.1690559387207,9.41489505767822}, {15.9039993286133,19.1320552825928,9.41489505767822}, {15.8949995040894,19.4460544586182,9.41489505767822},
                  {15.8769998550415,19.0420551300049,9.41489505767822}, {12.2169990539551,20.6500549316406,9.41489505767822}, {11.9379997253418,20.6810550689697,9.41489505767822}, {11.8629989624023,19.2130546569824,9.41489505767822}, {12.096999168396,19.1950550079346,9.41489505767822}, {14.1669998168945,18.6640548706055,9.41489505767822}, {14.1039991378784,20.2460556030273,9.41489505767822}, {13.9849996566772,18.7360553741455,9.41489505767822}, {14.7349996566772,19.1590557098389,9.41489505767822}, {14.5849990844727,19.2050552368164,9.41489505767822}, {14.5719995498657,18.4850559234619,9.41489505767822}, {14.1939992904663,19.6760559082031,9.41489505767822}, {14.1849994659424,19.9330558776855,9.41489505767822}, {14.1759996414185,18.6640548706055,9.41489505767822}, {14.261999130249,19.4890556335449,9.41489505767822}, {14.3539991378784,19.3610553741455,9.41489505767822}, {14.3559989929199,18.5830554962158,9.41489505767822}, {11.6039991378784,20.1250553131104,9.41489505767822}, {11.5209999084473,20.0520553588867,9.41489505767822}, {11.4209995269775,19.2480545043945,9.41489505767822}, {11.6989994049072,20.2690544128418,9.41389465332031}, {11.7609996795654,20.4310550689697,9.41489505767822}, {11.8359994888306,19.2130546569824,9.41489505767822}, {14.1889991760254,20.1710548400879,9.41489505767822}, {13.9689998626709,20.2840557098389,9.41489505767822}, {13.8739995956421,20.315055847168,9.41489505767822}, {13.7799997329712,18.8080558776855,9.41489505767822}, {13.9869995117188,20.2750549316406,9.41489505767822}, {12.3129997253418,20.5980548858643,9.41489505767822}, {12.3399991989136,20.5090560913086,9.41489505767822}, {12.3489990234375,20.3830547332764,9.41489505767822}, {12.3599996566772,20.2680549621582,9.41489505767822}, {12.3849992752075,20.1850547790527,9.41489505767822}, {12.3849992752075,20.1670551300049,9.41489505767822}, {12.4249992370605,20.065055847168,9.41489505767822}, {12.4729995727539,19.1350555419922,9.41489505767822}, {14.4399995803833,19.2900543212891,9.41489505767822}, {14.3649997711182,18.5740547180176,9.41489505767822}, {13.5729999542236,20.0310554504395,9.41489505767822}, {13.4889993667603,19.9140548706055,9.41489505767822}, {13.5639991760254,18.8710556030273,9.41489505767822}, {13.6389999389648,20.1310558319092,9.41489505767822}, {13.6719989776611,20.2130546569824,9.41489505767822}, {13.75,20.3020553588867,9.41489505767822}, {12.7399997711182,19.7810554504395,9.41489505767822}, {12.6189994812012,19.8520545959473,9.41489505767822}, {12.5799999237061,19.1200542449951,9.41489505767822}, {12.8349990844727,19.069055557251,9.41489505767822}, {11.2669992446899,19.9350547790527,9.41489505767822}, {11.1029987335205,19.9230556488037,9.41489505767822}, {11.0209999084473,19.2600555419922,9.41489505767822}, {11.3819999694824,19.9710559844971,9.41489505767822}, {13.418999671936,19.8530559539795,9.41489505767822}, {13.4329996109009,18.9160556793213,9.41489505767822}, {11.8399991989136,20.6430549621582,9.41489505767822}, {13.3119993209839,19.7800559997559,9.41489505767822}, {15.2189998626709,19.2600555419922,9.41489505767822}, {15.1839990615845,18.1600551605225,9.41489505767822}, {15.3639993667603,18.0520553588867,9.41489505767822}, {13.0189990997314,19.7250556945801,9.41489505767822}, {12.8949995040894,19.7350559234619,9.41489505767822}, {15.9039993286133,19.1500549316406,9.41489505767822}, {15.7699995040894,19.5140552520752,9.41489505767822}, {15.8589992523193,18.9340553283691,9.41489505767822}, {14.1939992904663,19.9510555267334,9.41489505767822}, {14.2119998931885,20.0630550384521,9.41489505767822}, {14.8589992523193,19.149055480957,9.41489505767822}, {14.8159999847412,18.3670558929443,9.41489505767822}, {14.8959999084473,18.3220558166504,9.41489505767822}, {12.5189990997314,19.9360542297363,9.41489505767822}, {11.0209999084473,19.9290542602539,9.41489505767822}, {11.0209999084473,19.2530555725098,2.67489433288574}, {11.0209999084473,19.9300556182861,2.67489433288574}, {15.9799995422363,18.505931854248,5.58724021911621}, {15.9799995422363,18.5044555664062,9.41489505767822}, {15.9799995422363,18.5041732788086,2.67489433288574}, {15.9799995422363,18.1684837341309,2.67489433288574}, {15.9799995422363,18.1288299560547,9.41489505767822}, {15.9799995422363,17.9876575469971,2.67489433288574}, {15.9799995422363,17.6247596740723,3.91620373725891}, {15.9799995422363,17.6247596740723,2.67489433288574}, {15.9799995422363,17.6254329681396,4.32245063781738}, {15.9799995422363,17.8920269012451,9.41489505767822}, {15.9799995422363,17.8795108795166,2.67489433288574}, {15.9799995422363,17.629810333252,4.58585262298584}, {15.9799995422363,17.6336059570312,5.27938556671143}, {15.9799995422363,17.8311748504639,2.67489433288574}, {15.9799995422363,17.638355255127,9.41489505767822}, {15.9799995422363,17.6346111297607,5.98653984069824}, {15.9799995422363,17.8728256225586,2.67489433288574}, {15.9799995422363,18.2221603393555,2.67489433288574} },
				{ {0,1,2}, {0,3,1}, {0,4,3}, {5,6,7}, {8,6,5}, {2,9,0}, {6,10,11}, {12,13,14}, {15,16,17}, {18,16,15}, {19,20,21}, {22,23,24}, {25,23,22}, {26,23,25}, {27,23,26}, {28,23,27}, {29,30,31}, {29,32,30}, {29,28,32}, {33,28,29}, {33,23,28}, {21,23,33}, {20,23,21}, {34,35,36}, {37,35,34}, {38,35,37}, {39,35,38}, {40,35,39}, {41,35,40}, {42,35,41}, {43,35,42}, {44,35,43}, {45,35,44}, {46,35,45}, {47,35,46}, {48,35,47}, {49,50,51}, {52,48,47}, {23,49,24}, {53,54,55}, {56,57,58}, {59,57,56}, {60,57,59}, {61,57,60}, {62,57,61}, {63,57,62}, {64,57,63}, {65,57,64}, {66,57,65}, {13,57,66}, {54,67,55}, {68,69,70}, {71,69,68}, {72,69,71}, {73,69,72}, {74,69,73}, {75,69,74}, {76,69,75}, {77,69,76}, {67,69,77}, {70,16,68}, {70,17,16}, {78,79,80}, {81,79,78}, {82,79,81}, {83,79,82}, {84,79,83}, {85,79,84}, {86,79,85}, {87,79,86}, {88,8,5}, {11,7,6}, {11,89,7}, {11,9,89}, {11,0,9}, {55,90,53}, {55,79,90}, {55,80,79}, {91,11,10}, {92,69,12}, {92,70,69}, {34,93,37}, {47,94,52}, {47,95,94}, {47,57,95}, {47,58,57}, {51,24,49}, {21,35,19}, {21,36,35}, {14,92,12}, {86,10,87}, {86,91,10}, {77,55,67}, {66,14,13}, {96,97,4}, {98,99,100}, {101,102,98}, {103,101,98}, {104,103,98}, {105,106,107}, {108,105,107}, {109,108,107}, {100,109,107}, {110,111,112}, {113,110,112}, {114,115,116}, {117,114,116}, {118,119,120}, {121,122,123}, {124,121,123}, {125,126,127}, {128,129,130}, {131,132,133}, {71,131,133}, {134,71,133}, {135,134,133}, {136,135,133}, {137,138,139}, {140,137,139}, {141,140,139}, {142,31,141}, {142,141,139}, {143,126,132}, {144,145,146}, {147,144,146}, {127,147,146}, {148,121,124}, {149,148,124}, {150,149,124}, {151,150,124}, {152,151,124}, {153,152,124}, {154,153,124}, {155,154,124}, {129,156,157}, {130,129,157}, {158,159,160}, {161,158,160}, {162,161,160}, {163,162,160}, {146,163,160}, {164,165,166}, {167,164,166}, {168,169,170}, {171,168,170}, {139,171,170}, {159,172,173}, {123,174,142}, {175,110,113}, {173,175,113}, {106,176,177}, {178,106,177}, {179,180,167}, {112,179,167}, {175,173,172}, {119,118,181}, {119,181,97}, {119,97,96}, {182,98,102}, {182,102,183}, {182,183,120}, {182,120,119}, {143,132,184}, {184,185,143}, {147,127,126}, {174,123,122}, {159,173,160}, {126,125,133}, {126,133,132}, {186,187,188}, {186,188,116}, {186,116,115}, {99,98,182}, {109,100,99}, {106,178,107}, {114,117,177}, {114,177,176}, {128,130,187}, {128,187,186}, {135,136,157}, {135,157,156}, {163,146,145}, {164,167,180}, {179,112,111}, {171,139,138}, {189,155,166}, {189,166,165}, {149,150,93}, {154,155,189}, {31,142,174}, {114,176,78}, {81,78,176}, {7,89,183}, {89,9,120}, {89,120,183}, {78,80,114}, {176,106,81}, {88,5,103}, {183,102,7}, {118,120,9}, {9,2,181}, {9,181,118}, {115,114,80}, {82,81,106}, {101,103,5}, {102,101,5}, {5,7,102}, {97,181,2}, {2,1,97}, {1,3,97}, {80,55,115}, {172,159,59}, {59,56,172}, {3,4,97}, {4,0,96}, {105,108,82}, {186,115,55}, {82,106,105}, {83,82,108}, {60,59,159}, {175,172,56}, {119,96,0}, {0,11,119}, {108,109,84}, {84,83,108}, {55,77,186}, {56,58,110}, {56,110,175}, {60,159,158}, {11,91,182}, {182,119,11}, {91,86,182}, {85,84,109}, {86,85,99}, {128,186,77}, {58,111,110}, {158,161,60}, {26,25,137}, {138,137,25}, {99,182,86}, {109,99,85}, {77,76,128}, {58,47,111}, {61,60,161}, {137,140,26}, {27,26,140}, {25,22,138}, {129,128,76}, {76,75,129}, {75,74,129}, {74,73,156}, {73,72,135}, {68,16,184}, {68,184,132}, {16,18,185}, {161,162,62}, {62,61,161}, {179,111,47}, {171,138,22}, {156,129,74}, {135,156,73}, {134,135,72}, {72,71,134}, {68,132,131}, {185,184,16}, {18,15,185}, {63,62,162}, {28,27,140}, {22,24,171}, {71,68,131}, {15,17,143}, {15,143,185}, {17,70,143}, {70,92,126}, {162,163,64}, {64,63,162}, {180,179,47}, {47,46,180}, {140,141,28}, {168,171,24}, {126,143,70}, {92,14,147}, {147,126,92}, {14,66,144}, {14,144,147}, {65,64,163}, {66,65,145}, {46,45,180}, {32,28,141}, {24,51,168}, {145,144,66}, {163,145,65}, {164,180,45}, {45,44,164}, {44,43,164}, {43,42,165}, {38,37,151}, {150,151,37}, {37,93,150}, {141,31,30}, {30,32,141}, {169,168,51}, {165,164,43}, {189,165,42}, {42,41,189}, {40,39,152}, {40,152,153}, {151,152,39}, {39,38,151}, {93,34,149}, {154,189,41}, {153,154,41}, {41,40,153}, {148,149,34}, {34,36,148}, {36,21,121}, {31,174,29}, {121,148,36}, {21,33,122}, {21,122,121}, {33,29,122}, {174,122,29}, {116,188,53}, {104,98,10}, {87,10,98}, {98,100,87}, {79,87,100}, {79,100,107}, {90,79,107}, {90,107,178}, {178,177,90}, {53,90,177}, {53,177,117}, {117,116,53}, {54,53,188}, {54,188,187}, {67,54,187}, {67,187,130}, {69,67,130}, {69,130,157}, {12,69,157}, {12,157,136}, {136,133,12}, {12,133,125}, {125,127,12}, {13,12,127}, {127,146,13}, {57,13,146}, {57,146,160}, {95,57,160}, {95,160,173}, {173,113,95}, {94,95,113}, {113,112,94}, {52,94,112}, {48,52,112}, {112,167,48}, {35,48,167}, {35,167,166}, {19,35,166}, {139,170,50}, {50,49,139}, {166,155,19}, {20,19,155}, {155,124,20}, {23,20,124}, {23,124,123}, {49,23,123}, {49,123,142}, {142,139,49}, {190,191,170}, {192,191,190}, {191,192,51}, {191,51,50}, {170,169,190}, {169,51,192}, {169,192,190}, {170,191,50}, {193,194,195}, {196,197,198}, {199,200,201}, {198,202,203}, {204,201,200}, {205,204,200}, {206,207,208}, {206,208,205}, {206,205,200}, {207,206,209}, {207,209,203}, {207,203,202}, {202,198,197}, {197,196,210}, {197,210,195}, {197,195,194}, {8,88,195}, {8,195,210}, {210,196,8}, {196,198,8}, {198,203,8}, {203,209,8}, {209,206,8}, {206,200,8}, {202,197,104}, {207,202,104}, {103,104,197}, {103,197,194}, {193,195,88}, {88,103,194}, {88,194,193}, {200,199,8}, {199,201,8}, {204,205,6}, {6,8,201}, {6,201,204}, {10,6,205}, {10,205,208}, {104,10,208}, {104,208,207} });
            break;
        case TestMesh::pyramid:
            mesh = TriangleMesh(
            	{ {10,10,40}, {0,0,0}, {20,0,0}, {20,20,0}, {0,20,0} },
            	{ {0,1,2}, {0,3,4}, {3,1,4}, {1,3,2}, {3,0,2}, {4,1,0} });
            break;
        case TestMesh::two_hollow_squares:
            mesh = TriangleMesh(
            	{ {66.7133483886719f,104.286666870117f,0}, {66.7133483886719f,95.7133331298828f,0}, {65.6666870117188f,94.6666717529297f,0}, {75.2866821289062f,95.7133331298828f,0}, {76.3333435058594f,105.333335876465f,0},
                  {76.3333435058594f,94.6666717529297f,0}, {65.6666870117188f,105.33332824707f,0}, {75.2866821289062f,104.286666870117f,0}, {71.1066818237305f,104.58666229248f,2.79999995231628f}, {66.4133529663086f,104.58666229248f,2.79999995231628f},
                  {75.5866851806641f,104.58666229248f,2.79999995231628f}, {66.4133529663086f,99.8933334350586f,2.79999995231628f}, {66.4133529663086f,95.4133377075195f,2.79999995231628f}, {71.1066818237305f,95.4133377075195f,2.79999995231628f}, 
                  {75.5866851806641f,95.4133377075195f,2.79999995231628f}, {75.5866851806641f,100.106666564941f,2.79999995231628f}, {74.5400161743164f,103.540000915527f,2.79999995231628f}, {70.0320129394531f,103.540000915527f,2.79999995231628f},
                  {67.4600067138672f,103.540000915527f,2.79999995231628f}, {67.4600067138672f,100.968002319336f,2.79999995231628f}, {67.4600067138672f,96.4599990844727f,2.79999995231628f}, {74.5400161743164f,99.0319976806641f,2.79999995231628f},
                  {74.5400161743164f,96.4599990844727f,2.79999995231628f}, {70.0320129394531f,96.4599990844727f,2.79999995231628f}, {123.666717529297f,94.6666717529297f,0}, {134.333312988281f,94.6666717529297f,0}, {124.413360595703f,95.4133377075195f,2.79999995231628f}, 
                  {129.106674194336f,95.4133377075195f,2.79999995231628f}, {133.586669921875f,95.4133377075195f,2.79999995231628f}, {123.666717529297f,105.33332824707f,0}, {124.413360595703f,104.58666229248f,2.79999995231628f}, 
                  {124.413360595703f,99.8933334350586f,2.79999995231628f}, {134.333312988281f,105.33332824707f,0}, {129.106674194336f,104.58666229248f,2.79999995231628f}, {133.586669921875f,104.58666229248f,2.79999995231628f},
                  {133.586669921875f,100.106666564941f,2.79999995231628f}, {124.713317871094f,104.286666870117f,0}, {124.713317871094f,95.7133331298828f,0}, {133.286712646484f,95.7133331298828f,0}, {133.286712646484f,104.286666870117f,0},
                  {132.540023803711f,103.540000915527f,2.79999995231628f}, {128.032028198242f,103.540008544922f,2.79999995231628f}, {125.460006713867f,103.540000915527f,2.79999995231628f}, {125.460006713867f,100.968002319336f,2.79999995231628f},
                  {125.460006713867f,96.4599990844727f,2.79999995231628f}, {132.540023803711f,99.0319976806641f,2.79999995231628f}, {132.540023803711f,96.4599990844727f,2.79999995231628f}, {128.032028198242f,96.4599990844727f,2.79999995231628f} },
	            { {0,1,2}, {3,4,5}, {6,4,0}, {6,0,2}, {2,1,5}, {7,4,3}, {1,3,5}, {0,4,7}, {4,6,8}, {6,9,8}, {4,8,10}, {6,2,9}, {2,11,9}, {2,12,11}, {2,5,12}, {5,13,12}, {5,14,13}, {4,10,15}, {5,4,14}, {4,15,14}, {7,16,17}, {0,7,18}, {7,17,18}, {1,19,20}, {1,0,19}, {0,18,19}, {7,3,21}, {3,22,21}, {7,21,16}, {3,23,22}, {3,1,23}, {1,20,23}, {24,25,26}, {25,27,26}, {25,28,27}, {29,24,30}, {24,31,30}, {24,26,31}, {32,29,33}, {29,30,33}, {32,33,34}, {32,34,35}, {25,32,28}, {32,35,28}, {36,37,24}, {38,32,25}, {29,32,36}, {29,36,24}, {24,37,25}, {39,32,38}, {37,38,25}, {36,32,39}, {39,40,41}, {36,39,42}, {39,41,42}, {37,43,44}, {37,36,43}, {36,42,43}, {39,38,45}, {38,46,45}, {39,45,40}, {38,47,46}, {38,37,47}, {37,44,47}, {16,8,9}, {16,10,8}, {10,16,15}, {15,16,21}, {22,15,21}, {15,22,14}, {22,23,14}, {23,20,14}, {17,16,9}, {18,17,9}, {19,18,9}, {19,9,11}, {19,11,20}, {13,14,20}, {20,11,12}, {13,20,12}, {41,40,30}, {42,41,30}, {43,42,30}, {43,30,31}, {43,31,44}, {27,28,44}, {44,31,26}, {27,44,26}, {40,33,30}, {40,34,33}, {34,40,35}, {35,40,45}, {46,35,45}, {35,46,28}, {46,47,28}, {47,44,28} });
            break;
        case TestMesh::small_dorito:
            mesh = TriangleMesh(
            	{ {6.00058937072754f,-22.9982089996338f,0}, {22.0010242462158f,-49.9998741149902f,0}, {-9.99957847595215f,-49.999870300293f,0}, {6.00071382522583f,-32.2371635437012f,28.0019245147705f},
                  {11.1670551300049f,-37.9727020263672f,18.9601669311523f}, {6.00060224533081f,-26.5392456054688f,10.7321853637695f} },
            	{ {0,1,2}, {3,4,5}, {2,1,4}, {2,4,3}, {2,3,5}, {2,5,0}, {5,4,1}, {5,1,0} });
            break;
        case TestMesh::bridge:
            mesh = TriangleMesh(
            	{ {75,84.5f,8}, {125,84.5f,8}, {75,94.5f,8}, {120,84.5f,5}, {125,94.5f,8}, {75,84.5f,0}, {80,84.5f,5}, {125,84.5f,0}, {125,94.5f,0}, {80,94.5f,5}, {75,94.5f,0}, {120,94.5f,5}, {120,84.5f,0}, {80,94.5f,0}, {80,84.5f,0}, {120,94.5f,0} },
            	{ {0,1,2}, {1,0,3}, {2,1,4}, {2,5,0}, {0,6,3}, {1,3,7}, {1,8,4}, {4,9,2}, {10,5,2}, {5,6,0}, {6,11,3}, {3,12,7}, {7,8,1}, {4,8,11}, {4,11,9}, {9,10,2}, {10,13,5}, {14,6,5}, {9,11,6}, {11,12,3}, {12,8,7}, {11,8,15}, {13,10,9}, {5,13,14}, {14,13,6}, {6,13,9}, {15,12,11}, {15,8,12} });
            break;
        case TestMesh::bridge_with_hole:
            mesh = TriangleMesh(
            	{ {75,69.5f,8}, {80,76.9091644287109f,8}, {75,94.5f,8}, {125,69.5f,8}, {120,76.9091644287109f,8}, {120,87.0908355712891f,8}, {80,87.0908355712891f,8}, {125,94.5f,8}, {80,87.0908355712891f,5}, {120,87.0908355712891f,5}, {125,94.5f,0}, {120,69.5f,0}, {120,94.5f,0}, {125,69.5f,0}, {120,94.5f,5}, {80,94.5f,5}, {80,94.5f,0}, {75,94.5f,0}, {80,69.5f,5}, {80,69.5f,0}, {80,76.9091644287109f,5}, {120,69.5,5}, {75,69.5f,0}, {120,76.9091644287109f,5} },
            	{ {0,1,2}, {1,0,3}, {1,3,4}, {4,3,5}, {2,6,7}, {6,2,1}, {7,6,5}, {7,5,3}, {5,8,9}, {8,5,6}, {10,11,12}, {11,10,13}, {14,8,15}, {8,14,9}, {2,16,17}, {16,2,15}, {15,2,14}, {14,10,12}, {10,14,7}, {7,14,2}, {16,18,19}, {18,16,20}, {20,16,1}, {1,16,8}, {8,16,15}, {6,1,8}, {3,11,13}, {11,3,21}, {21,3,18}, {18,22,19}, {22,18,0}, {0,18,3}, {16,22,17}, {22,16,19}, {2,22,0}, {22,2,17}, {5,23,4}, {23,11,21}, {11,23,12}, {12,23,9}, {9,23,5}, {12,9,14}, {23,18,20}, {18,23,21}, {10,3,13}, {3,10,7}, {1,23,20}, {23,1,4} });
            break;
        case TestMesh::step:
            mesh = TriangleMesh(
            	{ {0,20,5}, {0,20,0}, {0,0,5}, {0,0,0}, {20,0,0}, {20,0,5}, {1,19,5}, {1,1,5}, {19,1,5}, {20,20,5}, {19,19,5}, {20,20,0}, {19,19,10}, {1,19,10}, {1,1,10}, {19,1,10} },
            	{ {0,1,2}, {1,3,2}, {3,4,5}, {2,3,5}, {6,0,2}, {6,2,7}, {5,8,7}, {5,7,2}, {9,10,8}, {9,8,5}, {9,0,6}, {9,6,10}, {9,11,1}, {9,1,0}, {3,1,11}, {4,3,11}, {5,11,9}, {5,4,11}, {12,10,6}, {12,6,13}, {6,7,14}, {13,6,14}, {7,8,15}, {14,7,15}, {15,8,10}, {15,10,12}, {12,13,14}, {12,14,15} });
            break;
        case TestMesh::slopy_cube:
            mesh = TriangleMesh(
            	{ {-10,-10,0}, {-10,-10,20}, {-10,10,0}, {-10,10,20}, {0,-10,10}, {10,-10,0}, {2.92893f,-10,10}, {10,-10,2.92893f}, {0,-10,20}, {10,10,0}, {0,10,10}, {0,10,20}, {2.92893f,10,10}, {10,10,2.92893f} },
            	{ {0,1,2}, {2,1,3}, {4,0,5}, {4,1,0}, {6,4,7}, {7,4,5}, {4,8,1}, {0,2,5}, {5,2,9}, {2,10,9}, {10,3,11}, {2,3,10}, {9,10,12}, {13,9,12}, {3,1,8}, {11,3,8}, {10,11,8}, {4,10,8}, {6,12,10}, {4,6,10}, {7,13,12}, {6,7,12}, {7,5,9}, {13,7,9} });
            break;
        default:
        	throw Slic3r::InvalidArgument("Slic3r::Test::mesh(): called with invalid mesh ID");
            break;        
    }

    return mesh;
}

TriangleMesh mesh(TestMesh min, Vec3d translate, Vec3d scale)
{
    TriangleMesh m = mesh(min);
    m.translate(translate.cast<float>());
    m.scale(scale.cast<float>());
    return m;
}

TriangleMesh mesh(TestMesh m, Vec3d translate, double scale)
{
    return mesh(m, translate, Vec3d(scale, scale, scale));
}

static bool verbose_gcode() 
{
    const char *v = std::getenv("SLIC3R_TESTS_GCODE");
    if (v == nullptr)
    	return false;
    std::string s(v);
    return s == "1" || s == "on" || s == "yes";
}

void init_print(std::vector<TriangleMesh> &&meshes, Slic3r::Print &print, Slic3r::Model &model, const DynamicPrintConfig &config_in, bool comments)
{
	DynamicPrintConfig config = DynamicPrintConfig::full_print_config();
    config.apply(config_in);

    if (verbose_gcode())
        config.set_key_value("gcode_comments", new ConfigOptionBool(true));

    for (const TriangleMesh &t : meshes) {
		ModelObject *object = model.add_object();
		object->name += "object.stl";
		object->add_volume(std::move(t));
		object->add_instance();
	}
    arrange_objects(model, arr2::to_arrange_bed(get_bed_shape(config)), arr2::ArrangeSettings{}.set_distance_from_objects(min_object_distance(config)));
    model.center_instances_around_point({100, 100});
	for (ModelObject *mo : model.objects) {
        mo->ensure_on_bed();
		print.auto_assign_extruders(mo);
    }

	print.apply(model, config);
    print.validate();
    print.set_status_silent();
}

void init_print(std::initializer_list<TestMesh> test_meshes, Slic3r::Print &print, Slic3r::Model &model, const Slic3r::DynamicPrintConfig &config_in, bool comments)
{
	std::vector<TriangleMesh> triangle_meshes;
	triangle_meshes.reserve(test_meshes.size());
	for (const TestMesh test_mesh : test_meshes)
		triangle_meshes.emplace_back(mesh(test_mesh));
	init_print(std::move(triangle_meshes), print, model, config_in, comments);
}

void init_print(std::initializer_list<TriangleMesh> input_meshes, Slic3r::Print &print, Slic3r::Model &model, const DynamicPrintConfig &config_in, bool comments)
{
	std::vector<TriangleMesh> triangle_meshes;
	triangle_meshes.reserve(input_meshes.size());
	for (const TriangleMesh &input_mesh : input_meshes)
		triangle_meshes.emplace_back(input_mesh);
	init_print(std::move(triangle_meshes), print, model, config_in, comments);
}

void init_print(std::initializer_list<TestMesh> meshes, Slic3r::Print &print, Slic3r::Model &model, std::initializer_list<Slic3r::ConfigBase::SetDeserializeItem> config_items, bool comments)
{
	Slic3r::DynamicPrintConfig config = Slic3r::DynamicPrintConfig::full_print_config();
	config.set_deserialize_strict(config_items);
	init_print(meshes, print, model, config, comments);
}

void init_print(std::initializer_list<TriangleMesh> meshes, Slic3r::Print &print, Slic3r::Model &model, std::initializer_list<Slic3r::ConfigBase::SetDeserializeItem> config_items, bool comments)
{
	Slic3r::DynamicPrintConfig config = Slic3r::DynamicPrintConfig::full_print_config();
	config.set_deserialize_strict(config_items);
	init_print(meshes, print, model, config, comments);
}

void init_and_process_print(std::initializer_list<TestMesh> meshes, Slic3r::Print &print, const DynamicPrintConfig &config, bool comments)
{
	Slic3r::Model model;
	init_print(meshes, print, model, config, comments);
	print.process();
}

void init_and_process_print(std::initializer_list<TriangleMesh> meshes, Slic3r::Print &print, const DynamicPrintConfig &config, bool comments)
{
	Slic3r::Model model;
	init_print(meshes, print, model, config, comments);
	print.process();
}

void init_and_process_print(std::initializer_list<TestMesh> meshes, Slic3r::Print &print, std::initializer_list<Slic3r::ConfigBase::SetDeserializeItem> config_items, bool comments)
{
	Slic3r::Model model;
	init_print(meshes, print, model, config_items, comments);
	print.process();
}

void init_and_process_print(std::initializer_list<TriangleMesh> meshes, Slic3r::Print &print, std::initializer_list<Slic3r::ConfigBase::SetDeserializeItem> config_items, bool comments)
{
	Slic3r::Model model;
	init_print(meshes, print, model, config_items, comments);
	print.process();
}

std::string gcode(Print & print)
{
	boost::filesystem::path temp = boost::filesystem::unique_path();
    print.set_status_silent();
    print.process();
    print.export_gcode(temp.string(), nullptr, nullptr);
    boost::nowide::ifstream t(temp.string());
	std::string str((std::istreambuf_iterator<char>(t)), std::istreambuf_iterator<char>());
	boost::nowide::remove(temp.string().c_str());
	return str;
}

Slic3r::Model model(const std::string &model_name, TriangleMesh &&_mesh)
{
    Slic3r::Model result;
	ModelObject *object = result.add_object();
	object->name += model_name + ".stl";
    object->add_volume(_mesh);
    object->add_instance();
    return result;
}

std::string slice(std::initializer_list<TestMesh> meshes, const DynamicPrintConfig &config, bool comments)
{
	Slic3r::Print print;
	Slic3r::Model model;
	init_print(meshes, print, model, config, comments);
	return gcode(print);
}

std::string slice(std::initializer_list<TriangleMesh> meshes, const DynamicPrintConfig &config, bool comments)
{
	Slic3r::Print print;
	Slic3r::Model model;
	init_print(meshes, print, model, config, comments);
	return gcode(print);
}

std::string slice(std::initializer_list<TestMesh> meshes, std::initializer_list<Slic3r::ConfigBase::SetDeserializeItem> config_items, bool comments)
{
	Slic3r::Print print;
	Slic3r::Model model;
	init_print(meshes, print, model, config_items, comments);
	return gcode(print);
}

std::string slice(std::initializer_list<TriangleMesh> meshes, std::initializer_list<Slic3r::ConfigBase::SetDeserializeItem> config_items, bool comments)
{
	Slic3r::Print print;
	Slic3r::Model model;
	init_print(meshes, print, model, config_items, comments);
	return gcode(print);
}

bool contains(const std::string &data, const std::string &pattern)
{
    return data.find(pattern) != data.npos;    
}

bool contains_regex(const std::string &data, const std::string &pattern)
{
    boost::regex re(pattern);
    return boost::regex_match(data, re);
}

} } // namespace Slic3r::Test

#include <catch2/catch.hpp>

SCENARIO("init_print functionality", "[test_data]") {
	GIVEN("A default config") {
		Slic3r::DynamicPrintConfig config = Slic3r::DynamicPrintConfig::full_print_config();
		WHEN("init_print is called with a single mesh.") {
			Slic3r::Model model;
			Slic3r::Print print;
			Slic3r::Test::init_print({ Slic3r::Test::TestMesh::cube_20x20x20 }, print, model, config, true);
			THEN("One mesh/printobject is in the resulting Print object.") {
				REQUIRE(print.objects().size() == 1);
			}
			THEN("print.process() doesn't crash.") {
				REQUIRE_NOTHROW(print.process());
			}
			THEN("Export gcode functions outputs text.") {
				REQUIRE(! Slic3r::Test::gcode(print).empty());
			}
#if 0
			THEN("Embedded meshes exported") {
				std::string path = "C:\\data\\temp\\embedded_meshes\\";
				for (auto kvp : Slic3r::Test::mesh_names) {
					Slic3r::TriangleMesh m = mesh(kvp.first);
					std::string name = kvp.second;
					REQUIRE(Slic3r::store_stl((path + name + ".stl").c_str(), &m, true) == true);
					REQUIRE(Slic3r::store_obj((path + name + ".obj").c_str(), &m) == true);
				}
			}
#endif
		}
	}
}
