/*
 * --------------------------------------------------------------------------
 * Interface between Serial chip emulator and real terminal
 * 
 * (C) 2006 Jochen Karrer 
 *
 * State: working, not well tested 
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * --------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <unistd.h>
#include <string.h>
#include <sgstring.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <termios.h>
#include <sys/ioctl.h>

#include <serial.h>
#include <fio.h>
#include <configfile.h>

#if 0
#define dbgprintf(x...) { fprintf(stderr,x); }
#else
#define dbgprintf(x...)
#endif

typedef struct FileUart {
	SerialDevice serdev;
	int fd;
	FILE *logfile;
        FIO_FileHandler input_fh;
        FIO_FileHandler output_fh;
        int ifh_is_active;
        int ofh_is_active;
	struct termios termios;
} FileUart;

typedef struct NullUart {
	SerialDevice serdev;
	int tx_enabled;
} NullUart;

/*
 * ------------------------------------------------------
 * File descriptor based serial emulator backend 
 * ------------------------------------------------------
 */

static int
file_input(void *cd,int mask) {
	SerialDevice *serdev = (SerialDevice*) cd;
	Uart_RxEvent(serdev);
	return 0;
}

static int
file_output(void *cd,int mask) {
	SerialDevice *serdev = (SerialDevice*) cd;
	Uart_TxEvent(serdev);
	return 0;
}

static void
file_enable_rx(SerialDevice *serial_device) {
	FileUart *fuart = serial_device->owner;
        if((fuart->fd >= 0) && !(fuart->ifh_is_active)) {
                FIO_AddFileHandler(&fuart->input_fh,fuart->fd,FIO_READABLE,file_input,serial_device);
        	fuart->ifh_is_active=1;
        }
}

static void
file_disable_rx(SerialDevice *serial_device) {
	FileUart *fuart = serial_device->owner;
        if(fuart->ifh_is_active) {
                FIO_RemoveFileHandler(&fuart->input_fh);
        	fuart->ifh_is_active=0;
        }
}
static void
file_enable_tx(SerialDevice *serial_device) {
	FileUart *fuart = serial_device->owner;
        if((fuart->fd>=0) && !(fuart->ofh_is_active)) {
                FIO_AddFileHandler(&fuart->output_fh,fuart->fd,FIO_WRITABLE,file_output,serial_device);
        	fuart->ofh_is_active=1;
        }
}
static void
file_disable_tx(SerialDevice *serial_device) {
	FileUart *fuart = serial_device->owner;
        if(fuart->ofh_is_active) {
                FIO_RemoveFileHandler(&fuart->output_fh);
        	fuart->ofh_is_active=0;
        }
}

static void
file_close(SerialDevice *serial_device) {
	FileUart *fuart = serial_device->owner;	
	file_disable_tx(serial_device);
	file_disable_rx(serial_device);
	close(fuart->fd);
	fuart->fd=-1;
}

static int 
file_uart_read(SerialDevice *serial_device,uint8_t *buf,int maxlen) 
{
	FileUart *fuart = serial_device->owner;	
	int count;
	if(fuart->fd<0) {
		return 0;
	}
	count = read(fuart->fd,buf,maxlen);
	if(count>0)
		return count;
	if((count<=0) && (errno != EAGAIN)) {
		fprintf(stderr,"Read error\n");
		file_close(serial_device);
		return -1;
	}
	return 0;
}

static int 
file_uart_write(SerialDevice *serial_device,const uint8_t *buf,int len) 
{
	FileUart *fuart = serial_device->owner;	
	int count;
	if(fuart->fd<0) {
		return 0;
	}
	count = write(fuart->fd,buf,len);
	if(count>0) {
		if(fuart->logfile) {
			if(fwrite(buf,1,count,fuart->logfile)!=count) {
				fprintf(stderr,"Writing to logfile failed\n");
			}
		}
		return count;
	}
	if((count<=0) && (errno != EAGAIN)) {
		fprintf(stderr,"Write error\n");
		file_close(serial_device);
		return -1;
	}
	return 0;
}

static void 
file_uart_flush_termios(FileUart *fuart) 
{
        /* Shit: TCSADRAIN would be better but tcsettattr seems to block */
        if(tcsetattr(fuart->fd,TCSANOW,&fuart->termios)<0) {
                //perror("Can not  set terminal attributes");
                return;
        }

}

static void
file_uart_set_baudrate(FileUart *fuart,int rx_baudrate) 
{
	speed_t rate;
	if((rx_baudrate > 440000) && (rx_baudrate < 480000)) {
                rate=B460800;
        } else if((rx_baudrate > 220000) && (rx_baudrate < 240000)) {
                rate=B230400;
        } else if((rx_baudrate > 110000) && (rx_baudrate < 120000)) {
                rate=B115200;
        } else if((rx_baudrate > 55000) && (rx_baudrate < 60000)) {
                rate=B57600;
        } else if((rx_baudrate > 36571 ) && (rx_baudrate < 40320)) {
                rate=B38400;
        } else if((rx_baudrate > 18285 ) && (rx_baudrate < 20160)) {
                rate=B19200;
        } else if((rx_baudrate > 9142 ) && (rx_baudrate < 10080)) {
                rate=B9600;
        } else if((rx_baudrate > 4571 ) && (rx_baudrate < 5040)) {
                rate=B4800;
        } else if((rx_baudrate > 2285 ) && (rx_baudrate < 2520)) {
                rate=B2400;
        } else if((rx_baudrate > 1890 ) && (rx_baudrate < 1714)) {
                rate=B1800;
        } else if((rx_baudrate > 1142 ) && (rx_baudrate < 1260)) {
                rate=B1200;
        } else if((rx_baudrate > 570 ) && (rx_baudrate < 630)) {
                rate=B600;
        } else if((rx_baudrate > 285 ) && (rx_baudrate < 315)) {
                rate=B300;
        } else if((rx_baudrate > 190 ) && (rx_baudrate < 210)) {
                rate=B200;
        } else if((rx_baudrate > 142 ) && (rx_baudrate < 158)) {
                rate=B150;
        } else if((rx_baudrate > 128 ) && (rx_baudrate < 141)) {
                rate=B134;
        } else if((rx_baudrate > 105 ) && (rx_baudrate < 116)) {
                rate=B110;
        } else if((rx_baudrate > 71 ) && (rx_baudrate < 79)) {
                rate=B75;
        } else if((rx_baudrate > 47 ) && (rx_baudrate < 53)) {
                rate=B50;
        } else {
                fprintf(stderr,"Serial emulator: Can not handle Baudrate %d\n",rx_baudrate);
                rate=B0;
        }
	if(cfsetispeed ( &fuart->termios, rate)<0) {
                fprintf(stderr,"Can not change Baudrate\n");
        } else {
		dbgprintf("Changed speed to %d, rate %d\n",rx_baudrate,rate);
	}
        if(cfsetospeed ( &fuart->termios, rate)<0) {
                fprintf(stderr,"Can not change Baudrate\n");
        }
	file_uart_flush_termios(fuart);
}

static void
file_uart_set_csize(FileUart *fuart,int csize) 
{
	tcflag_t bits;
	switch(csize) {
                case 5:
                        bits=CS5; break;
                case 6:
                        bits=CS6; break;
                case  7:
                        bits=CS7; break;
                case  8:
                        bits=CS8; break;
                default:
			fprintf(stderr,"Illegal word length of %d\n",csize);
                        bits=CS8; break;
        }
	fuart->termios.c_cflag &= ~(CSIZE);
        fuart->termios.c_cflag |= bits;
	file_uart_flush_termios(fuart);
}

static int 
file_uart_cmd(SerialDevice *serial_device,UartCmd *cmd) 
{
	FileUart *fuart = serial_device->owner;	
	uint32_t arg = cmd->arg;
	unsigned int host_status;
	cmd->retval = 0;
	switch(cmd->opcode) {
		case UART_OPC_GET_DCD:
			if(isatty(fuart->fd) && (ioctl(fuart->fd,TIOCMGET,&host_status)>=0)) {
				cmd->retval = !!(host_status & TIOCM_CAR);
			}
			break;

		case UART_OPC_GET_RI:
			if(isatty(fuart->fd) && (ioctl(fuart->fd,TIOCMGET,&host_status)>=0)) {
				cmd->retval = !!(host_status & TIOCM_RNG);
			}
			break;

		case UART_OPC_GET_DSR:
			if(isatty(fuart->fd) && (ioctl(fuart->fd,TIOCMGET,&host_status)>=0)) {
				cmd->retval = !!(host_status & TIOCM_DSR);
			}
			break;

		case UART_OPC_GET_CTS:
			if(isatty(fuart->fd) && (ioctl(fuart->fd,TIOCMGET,&host_status)>=0)) {
				cmd->retval = !!(host_status & TIOCM_CTS);
			}
			break;

		case UART_OPC_SET_RTS:
			if(!isatty(fuart->fd)) {
				return 0;
			}
			if(arg) {
				ioctl(fuart->fd,TIOCMSET,TIOCM_RTS);
			} else {
				ioctl(fuart->fd,TIOCMBIC,TIOCM_RTS);
			}
			break;
		case UART_OPC_SET_DTR:
			if(!isatty(fuart->fd)) {
				return 0;
			}
			if(arg) {
				ioctl(fuart->fd,TIOCMSET,TIOCM_DTR);
			} else {
				ioctl(fuart->fd,TIOCMBIC,TIOCM_DTR);
			}
			break;

		case UART_OPC_SET_BAUDRATE:
			if(!isatty(fuart->fd)) {
                		return 0;
        		}
			file_uart_set_baudrate(fuart,arg);
			break;		
			
		case UART_OPC_SET_CSIZE: 
			if(!isatty(fuart->fd)) {
                		return 0;
        		}
			file_uart_set_csize(fuart,arg);	
			break;

		case UART_OPC_CRTSCTS: 
			
			if(!isatty(fuart->fd)) {
                		return 0;
        		}
			if(arg) {
        			fuart->termios.c_cflag |= CRTSCTS;
			} else {
				fuart->termios.c_cflag &= ~(CRTSCTS);
			}
			fuart->termios.c_cflag &= ~(CRTSCTS);
			file_uart_flush_termios(fuart);
			break;

		case UART_OPC_PAREN: 
			if(!isatty(fuart->fd)) {
                		return 0;
        		}
			if(arg) {
        			fuart->termios.c_cflag |= PARENB;
			} else {
				fuart->termios.c_cflag &= ~(PARENB);
			}
			file_uart_flush_termios(fuart);
			break;

		case UART_OPC_PARODD:
			if(!isatty(fuart->fd)) {
                		return 0;
        		}
			if(arg) {
        			fuart->termios.c_cflag |= PARODD;
			} else {
				fuart->termios.c_cflag &= ~(PARODD);
			}
			file_uart_flush_termios(fuart);
			break;

		default:
			break;
	}
	return 0;
}



static void
TerminalInit(FileUart *fuart) {
	 if(tcgetattr(fuart->fd,&fuart->termios)<0) {
                fprintf(stderr,"Can not  get terminal attributes\n");
                return;
        }
        if(fuart->fd > 0) {
                cfmakeraw(&fuart->termios);
        } else if(fuart->fd == 0){
		fuart->termios.c_lflag &= ~(ECHO | ICANON | ISIG | IEXTEN);
        }

        if(tcsetattr(fuart->fd,TCSAFLUSH,&fuart->termios)<0) {
                perror("can't set terminal settings");
                return;
        }
}

static void
TerminalRestore(int fd) {
        struct termios term;
        if(tcgetattr(fd,&term)<0) {
                perror("can't restore terminal settings\n");
                return;
        }
        term.c_lflag |= (ECHO|ECHONL|ICANON|ISIG|IEXTEN);
        if(tcsetattr(fd,TCSAFLUSH,&term)<0) {
                perror("can't restore terminal settings");
                return;
        } else {
		fprintf(stderr,"Restored terminal settings\n");
	}
}

static void
TerminalExit(void) {
        TerminalRestore(0);
}

/*
 * --------------------------------------------
 * The Template for a File serial device
 * --------------------------------------------
 */
static SerialDevice file_uart = {
	.stop_tx = file_disable_tx,
	.start_tx = file_enable_tx,
	.stop_rx = file_disable_rx,
	.start_rx = file_enable_rx,
	.uart_cmd = file_uart_cmd,
	.write = file_uart_write,
	.read = file_uart_read,
};

SerialDevice *
FileUart_New(const char *uart_name,const char *filename)  {
	FileUart *fiua = sg_calloc(FileUart);
	char *logfilename;
	fiua->serdev = file_uart; /* Copy from template */
	fiua->serdev.owner = fiua;
	fiua->logfile = NULL;	
	logfilename = Config_ReadVar(uart_name,"logfile");
	if(logfilename) {
		fiua->logfile = fopen(logfilename,"w+");
	}
	if(!strcmp(filename,"stdin")) {
		fiua->fd = 0;
	} else {
		fiua->fd = open(filename,O_RDWR);
	}
	if(fiua->fd<0) {
		fprintf(stderr,"%s: Cannot open %s\n",uart_name,filename);
		sleep(3);
		return &fiua->serdev;
	} else {
		fcntl(fiua->fd,F_SETFL,O_NONBLOCK);
		fprintf(stderr,"Uart \"%s\" Connected to %s\n",uart_name,filename);
	}
	TerminalInit(fiua);	
	atexit(TerminalExit);
	return &fiua->serdev;
}

static void
null_enable_tx(SerialDevice *serial_device) {
	NullUart *nua = serial_device->owner;
	nua->tx_enabled = 1;
	while(nua->tx_enabled) {
		Uart_TxEvent(serial_device);
	}
}

static int 
null_write(SerialDevice *serial_device,const uint8_t *buf,int len) 
{
	return len;
}

static int 
null_read(SerialDevice *serial_device,uint8_t *buf,int len) 
{
	return 0;
}

static void
null_disable_tx(SerialDevice *serial_device) {
	NullUart *nua = serial_device->owner;
	nua->tx_enabled = 0;
}

static SerialDevice null_uart =  {
	.stop_tx = null_disable_tx,
	.start_tx = null_enable_tx,
	.stop_rx = NULL,
	.start_rx = NULL,
	.write = null_write,
	.read = null_read,
};

static SerialDevice * 
NullUart_New(const char *uart_name,const char *filename)  {
	NullUart *nua = sg_calloc(NullUart);
	nua->serdev = null_uart; /* copy from template */
	nua->serdev.owner = nua;
	nua->tx_enabled = 0;
	return &nua->serdev;
}

typedef struct SerialModule_ListEntry {
	char *type;
	SerialDevice_Constructor *constructor;
	struct SerialModule_ListEntry *next;
} SerialModule_ListEntry;

static SerialModule_ListEntry *modListHead = NULL;
/*
 * --------------------------------------------------------
 * Register new Serial Device emulators
 * --------------------------------------------------------
 */
void
SerialModule_Register(const char *modname,SerialDevice_Constructor *newSerdev) 
{	
	SerialModule_ListEntry *le = sg_calloc(*le);
	le->type = sg_strdup(modname);
	le->next = modListHead;
	le->constructor = newSerdev;
	modListHead = le;	
}

UartPort *
Uart_New(const char *uart_name,UartRxEventProc *rxEventProc,UartTxEventProc *txEventProc,UartStatChgProc *statproc,void *owner) 
{
	const char *filename=Config_ReadVar(uart_name,"file");
	const char *type=Config_ReadVar(uart_name,"type");
	UartPort *port = (UartPort*) malloc(sizeof(UartPort));
	port->owner = owner;
	port->rxEventProc = rxEventProc;
        port->txEventProc = txEventProc;
        port->statProc = statproc;
	if(type) {
		SerialModule_ListEntry *le;
		for(le=modListHead;le;le=le->next) {
			if(!strcmp(le->type,type)) {
				port->serial_device = le->constructor(uart_name);
				break;
			}
		}
		if(!le) {
			fprintf(stderr,"No serial emulator of type \"%s\" found\n",type);
			exit(1);
		}
	} else {
		/* Old scheme */
		if(filename) {
			port->serial_device =  FileUart_New(uart_name,filename);
		} else {
			port->serial_device = NullUart_New(uart_name,filename);
			fprintf(stderr,"%s connected to nowhere\n",uart_name);
		}
	}
	port->serial_device->uart = port;
	return port;		
}
