/* ====================================================================
 * Copyright (c) 2008          Martin Hauner
 *                             http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "config.h"
#include "RpViewEntry.h"
#include "svn/DirEntry.h"


RpViewEntry::RpViewEntry( svn::DirEntryPtr entry )
  : _entry(entry)
{
}

svn::DirEntryPtr RpViewEntry::entry() const
{
  return _entry;
}

bool RpViewEntry::isDir() const
{
  return _entry->isDir();
}

const sc::String& RpViewEntry::path() const
{
  return _entry->getName();
}



enum RpViewEntryColumns
{
  ColName,
  ColRevision,
  ColSize,
  ColProperties,
  ColAuthor,
  ColTime,
  ColEmpty,
  ColMax
};


RpViewEntryData::RpViewEntryData()
{
}

QString RpViewEntryData::data( int column, const RpViewItem* item ) const
{
  svn::DirEntryPtr entry = static_cast<const RpViewEntry*>(item)->entry();

  if( column != ColName && column != ColEmpty && entry->isNop() )
  {
    return "-";
  }

  switch(column)
  {
  case ColName:
    {
      return QString::fromUtf8( entry->getName().getStr() );
    }
  case ColRevision:
    {
      return QString().sprintf( "%10ld", (unsigned long)entry->getLastRevision() );
    }
  case ColSize:
    {
      if( entry->isDir() )
        return _q("folder");
      else
      {
        char buf[5];
        apr_strfsize( entry->getSize(), buf );
        return buf;
      }
    }
  case ColProperties:
    {
      return entry->hasProperties() ? _q("yes") : _q("no");
    }
  case ColAuthor:
    {
      return QString::fromUtf8(entry->getLastAuthor().getStr());
    }
  case ColTime:
    {
      //char time[APR_RFC822_DATE_LEN] = {};
      //apr_rfc822_date( time, _entry->getLastDate() );
      char time[APR_CTIME_LEN] = {};
      apr_ctime( time, entry->getLastDate() );
      return time;
    }
  case ColEmpty:
    {
      return "";
    }
  default:
    {
      return "?";
    }
  }
}

QString RpViewEntryData::header( int column ) const
{
  switch(column)
  {
  case ColName: 
    return _q("repository file/folder");
  case ColRevision:
    return _q("last rev");
  case ColSize:
    return _q("size");
  case ColProperties:
    return _q("props");
  case ColAuthor:
    return _q("last author");
  case ColTime:
    return _q("last time");
  case ColEmpty:
    return "";
  default:
    return "?";
  }
}

Qt::Alignment RpViewEntryData::alignment( int column ) const
{
  switch(column)
  {
  case ColName:
    return Qt::AlignLeft | Qt::AlignVCenter;
  case ColRevision:
  case ColSize:
  case ColTime:
    return Qt::AlignRight | Qt::AlignVCenter;
  default:
    return Qt::AlignCenter;
  }
}

int RpViewEntryData::columns() const
{
  return ColMax;
}
