/*
 * Copyright 2009-2019 The VOTCA Development Team (http://www.votca.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
#define BOOST_TEST_MAIN

#define BOOST_TEST_MODULE bsecoupling_test
#include <boost/test/unit_test.hpp>
#include <votca/xtp/bsecoupling.h>

using namespace votca::xtp;
using namespace votca;

BOOST_AUTO_TEST_SUITE(bsecoupling_test)

Eigen::MatrixXd ReadMatrixFromString(const std::string& matrix) {
  votca::tools::Tokenizer lines(matrix, "\n");

  std::vector<double> entries;
  Index cols = 0;
  Index rows = 0;
  for (auto line : lines) {
    if (line[0] == '#') {
      continue;
    }
    votca::tools::Tokenizer entries_tok(line, " ");
    std::vector<std::string> temp = entries_tok.ToVector();
    cols = Index(temp.size());
    rows++;
    for (const auto& s : temp) {
      entries.push_back(std::stod(s));
    }
  }

  return Eigen::Map<Eigen::MatrixXd>(entries.data(), rows, cols);
}

BOOST_AUTO_TEST_CASE(coupling_test) {

  std::ofstream xyzfile("molecule.xyz");
  xyzfile << " 5" << std::endl;
  xyzfile << " methane" << std::endl;
  xyzfile << " C            .000000     .000000     .000000" << std::endl;
  xyzfile << " H            .629118     .629118     .629118" << std::endl;
  xyzfile << " H           -.629118    -.629118     .629118" << std::endl;
  xyzfile << " H            .629118    -.629118    -.629118" << std::endl;
  xyzfile << " H           -.629118     .629118    -.629118" << std::endl;
  xyzfile.close();

  std::ofstream basisfile("3-21G.xml");
  basisfile << "<basis name=\"3-21G\">" << std::endl;
  basisfile << "  <element name=\"H\">" << std::endl;
  basisfile << "    <shell scale=\"1.0\" type=\"S\">" << std::endl;
  basisfile << "      <constant decay=\"5.447178e+00\">" << std::endl;
  basisfile << "        <contractions factor=\"1.562850e-01\" type=\"S\"/>"
            << std::endl;
  basisfile << "      </constant>" << std::endl;
  basisfile << "      <constant decay=\"8.245470e-01\">" << std::endl;
  basisfile << "        <contractions factor=\"9.046910e-01\" type=\"S\"/>"
            << std::endl;
  basisfile << "      </constant>" << std::endl;
  basisfile << "    </shell>" << std::endl;
  basisfile << "    <shell scale=\"1.0\" type=\"S\">" << std::endl;
  basisfile << "      <constant decay=\"1.831920e-01\">" << std::endl;
  basisfile << "        <contractions factor=\"1.000000e+00\" type=\"S\"/>"
            << std::endl;
  basisfile << "      </constant>" << std::endl;
  basisfile << "    </shell>" << std::endl;
  basisfile << "  </element>" << std::endl;
  basisfile << "  <element name=\"C\">" << std::endl;
  basisfile << "    <shell scale=\"1.0\" type=\"S\">" << std::endl;
  basisfile << "      <constant decay=\"1.722560e+02\">" << std::endl;
  basisfile << "        <contractions factor=\"6.176690e-02\" type=\"S\"/>"
            << std::endl;
  basisfile << "      </constant>" << std::endl;
  basisfile << "      <constant decay=\"2.591090e+01\">" << std::endl;
  basisfile << "        <contractions factor=\"3.587940e-01\" type=\"S\"/>"
            << std::endl;
  basisfile << "      </constant>" << std::endl;
  basisfile << "      <constant decay=\"5.533350e+00\">" << std::endl;
  basisfile << "        <contractions factor=\"7.007130e-01\" type=\"S\"/>"
            << std::endl;
  basisfile << "      </constant>" << std::endl;
  basisfile << "    </shell>" << std::endl;
  basisfile << "    <shell scale=\"1.0\" type=\"SP\">" << std::endl;
  basisfile << "      <constant decay=\"3.664980e+00\">" << std::endl;
  basisfile << "        <contractions factor=\"-3.958970e-01\" type=\"S\"/>"
            << std::endl;
  basisfile << "        <contractions factor=\"2.364600e-01\" type=\"P\"/>"
            << std::endl;
  basisfile << "      </constant>" << std::endl;
  basisfile << "      <constant decay=\"7.705450e-01\">" << std::endl;
  basisfile << "        <contractions factor=\"1.215840e+00\" type=\"S\"/>"
            << std::endl;
  basisfile << "        <contractions factor=\"8.606190e-01\" type=\"P\"/>"
            << std::endl;
  basisfile << "      </constant>" << std::endl;
  basisfile << "    </shell>" << std::endl;
  basisfile << "    <shell scale=\"1.0\" type=\"SP\">" << std::endl;
  basisfile << "      <constant decay=\"1.958570e-01\">" << std::endl;
  basisfile << "        <contractions factor=\"1.000000e+00\" type=\"S\"/>"
            << std::endl;
  basisfile << "        <contractions factor=\"1.000000e+00\" type=\"P\"/>"
            << std::endl;
  basisfile << "      </constant>" << std::endl;
  basisfile << "    </shell> " << std::endl;
  basisfile << "  </element>" << std::endl;
  basisfile << "</basis>" << std::endl;
  basisfile.close();

  Orbitals A;
  A.setDFTbasisName("3-21G.xml");
  A.QMAtoms().LoadFromFile("molecule.xyz");
  A.setBasisSetSize(17);
  A.setNumberOfAlphaElectrons(5);
  A.setNumberOfOccupiedLevels(5);
  A.MOs().eigenvalues() = Eigen::VectorXd::Zero(17);
  A.MOs().eigenvalues() << -19.8117, -6.22408, -6.14094, -6.14094, -6.14094,
      -3.72889, -3.72889, -3.72889, -3.64731, -3.09048, -3.09048, -3.09048,
      -2.63214, -2.08206, -2.08206, -2.08206, -2.03268;

  A.MOs().eigenvectors() = Eigen::MatrixXd::Zero(17, 17);
  A.MOs().eigenvectors() << -0.996559, -0.223082, 4.81443e-15, 2.21045e-15,
      -6.16146e-17, -3.16966e-16, 5.46703e-18, -1.09681e-15, -0.0301914,
      6.45993e-16, 1.05377e-16, 3.41154e-16, -0.102052, -5.79826e-16,
      9.38593e-16, -4.69346e-15, -0.111923, -0.0445146, 0.88316, -1.94941e-14,
      -8.67388e-15, -7.26679e-16, 1.16326e-14, -3.35886e-15, 2.37877e-14,
      0.866126, 3.2068e-15, 3.80914e-15, 3.24563e-15, -0.938329, -6.4404e-15,
      1.10811e-14, -5.5056e-14, -1.28767, 8.15798e-17, 2.30849e-14, 1.04169,
      0.117804, 0.0951759, 0.179467, 0.147031, 0.39183, -1.02927e-14, 0.32699,
      -0.223689, -0.130009, 1.0375e-15, -0.0940179, 0.126956, 0.0122904,
      1.41709e-15, 4.60157e-17, -7.1203e-15, 0.143338, -0.980459, -0.355251,
      0.41611, -0.10826, -0.149964, 2.41546e-16, 0.12214, -0.0512447, 0.39537,
      1.1054e-15, -0.0996828, -0.0636092, -0.105478, 5.10746e-15, -5.25872e-18,
      4.8424e-15, 0.0488925, 0.364515, -0.9863, 0.0447336, 0.417155, -0.177023,
      5.76117e-15, -0.228081, -0.348136, 0.0253377, -1.05286e-15, 0.079576,
      0.0703157, -0.117608, 5.31327e-15, 0.0472716, 0.235837, -3.58018e-15,
      -1.68354e-15, 2.3989e-15, -9.86879e-15, 4.52519e-15, -1.6106e-14,
      -0.599523, -1.31237e-14, -8.63443e-15, -8.61196e-15, 1.8333, 2.05275e-14,
      -3.9562e-14, 1.89874e-13, 4.24316, -2.74184e-16, -1.53939e-15, -0.162416,
      -0.0183675, -0.0148395, -0.151162, -0.123842, -0.330032, 1.10084e-15,
      -1.45092, 0.992556, 0.576875, -3.82954e-15, 0.604373, -0.816111,
      -0.0790061, -8.89474e-15, -2.24862e-16, 3.23655e-15, -0.0223487, 0.152869,
      0.0553894, -0.350483, 0.0911859, 0.126313, -5.48468e-15, -0.541961,
      0.227383, -1.75434, -3.89443e-15, 0.640788, 0.408897, 0.67804,
      -3.17156e-14, -2.81346e-17, -1.09423e-15, -0.00762313, -0.0568338,
      0.15378, -0.0376785, -0.351364, 0.149104, -4.94425e-15, 1.01204, 1.54475,
      -0.112429, 8.50653e-15, -0.511536, -0.452008, 0.756019, -3.3547e-14,
      -0.00106227, 0.0237672, 0.00345981, -0.00139675, -0.00349474, -0.597906,
      -0.425733, -0.0605479, -0.343823, 0.162103, -0.45692, 0.21318, -0.600309,
      0.310843, -0.36406, 0.574148, 0.0554949, -0.00928842, -0.0414346,
      0.0619999, -0.0250297, -0.0626259, 0.00227746, 0.00162164, 0.00023063,
      -0.0301047, 0.273177, -0.770004, 0.359253, 0.0095153, -0.8783, 1.02867,
      -1.62228, -1.24406, -0.00106227, 0.0237672, 0.00238182, 0.00205737,
      0.00402848, 0.262742, 0.151145, -0.671213, -0.343823, 0.317484, 0.12884,
      -0.40386, -0.600309, 0.201313, -0.327527, -0.641099, 0.0554949,
      -0.00928842, -0.0414346, 0.0426822, 0.0368682, 0.0721904, -0.0010008,
      -0.000575719, 0.00255669, -0.0301047, 0.535026, 0.217123, -0.680588,
      0.0095153, -0.568818, 0.925441, 1.81145, -1.24406, -0.00106227, 0.0237672,
      -0.00318563, 0.0034409, -0.00203628, 0.514364, -0.353326, 0.391148,
      -0.343823, -0.496623, -0.0536813, -0.176018, -0.600309, -0.744328,
      -0.01898, 0.0665156, 0.0554949, -0.00928842, -0.0414346, -0.0570866,
      0.0616609, -0.0364902, -0.00195924, 0.00134584, -0.0014899, -0.0301047,
      -0.836913, -0.0904642, -0.296627, 0.0095153, 2.10313, 0.0536287,
      -0.187943, -1.24406, -0.00106227, 0.0237672, -0.002656, -0.00410152,
      0.00150255, -0.1792, 0.627913, 0.340613, -0.343823, 0.0170366, 0.38176,
      0.366698, -0.600309, 0.232172, 0.710567, 0.000435528, 0.0554949,
      -0.00928842, -0.0414346, -0.0475955, -0.0734994, 0.0269257, 0.000682583,
      -0.00239176, -0.00129742, -0.0301047, 0.0287103, 0.643346, 0.617962,
      0.0095153, -0.656011, -2.00774, -0.0012306, -1.24406;

  A.setBSEindices(0, 16);
  A.setTDAApprox(true);
  A.setGWindices(0, 16);
  Eigen::MatrixXd spsi_ref = Eigen::MatrixXd::Zero(60, 3);
  spsi_ref << -0.00178316, -0.0558332, 0.0151767, 0.00568291, 0.0149417,
      0.0556358, 0.05758, -0.00320371, -0.00502105, 0.00379328, -0.00232255,
      -0.00817518, -0.00848959, -0.000618633, -0.00376334, -0.000395809,
      -0.00899117, 0.0023708, 7.13955e-08, 3.47498e-08, -1.08537e-08,
      0.000420413, 0.011896, -0.00320024, -0.00288487, 0.00320821, 0.0115465,
      0.0119767, 0.000355172, 0.00289343, -2.55565e-08, 1.91684e-08,
      3.01647e-09, 6.84051e-09, 2.79592e-10, -1.35767e-09, 0.00420618, 0.092451,
      -0.0239374, 0.0036276, 0.0113951, 0.0471896, 0.0465325, -0.00398807,
      -0.00484251, 0.0051614, -0.0031325, -0.0112447, -0.010968, -0.000896935,
      -0.00488789, 0.000951266, 0.0239709, -0.00630323, 0.000419006, 0.0201651,
      -0.00573095, -0.00118124, -0.0269275, 0.00700955, -0.00345217, 0.00356488,
      0.0134361, 0.013156, 9.58532e-05, 0.00315613, -2.58268e-05, -0.00124098,
      0.000352706, -1.80679e-06, -8.71959e-05, 2.47799e-05, -0.0150073,
      0.0114186, 0.0443012, 0.0180327, -0.026287, 0.0734351, -0.0643994,
      0.0257628, 0.0132084, -0.0123339, 0.0092297, -0.0148779, 0.0122493,
      -0.00246837, -0.0125735, -0.00375172, 0.00294872, 0.0112899, 0.00648758,
      -0.0055755, -0.0191436, 0.00433063, -0.00332619, -0.0128321, 0.0111166,
      -0.00969272, 0.0189659, -0.0160119, 0.00458659, 0.0107104, -0.000399315,
      0.000343129, 0.00117813, -2.80525e-05, 2.41086e-05, 8.2778e-05,
      -0.0450479, -0.00034974, -0.015063, 0.0655838, 0.0115163, -0.022358,
      0.0220978, 0.0583236, 0.0513097, -0.0119156, 0.00490159, 0.0116429,
      -0.0132479, -0.0146227, -0.00863565, -0.0118978, -0.000282044,
      -0.00400272, 0.0199347, 0.00139057, 0.00635067, 0.0131991, 0.000163177,
      0.00441453, 0.0159091, -0.00241207, -0.0110696, 0.0123057, 0.0171503,
      0.0119626, -0.00122682, -8.55716e-05, -0.00039083, -8.62007e-05,
      -6.0128e-06, -2.746e-05, -0.0304688, -0.954049, 0.259333, 0.0971056,
      0.255313, 0.950672, 0.983887, -0.0547431, -0.0857965, 0.0170489,
      -0.0104387, -0.036743, -0.0381557, -0.00278036, -0.0169143, -0.00177889,
      -0.04041, 0.0106552, -2.23782e-07, 2.40738e-07, 1.03401e-07, -0.000182535,
      -0.00516415, 0.00138942, 0.00125201, -0.00139237, -0.00501195,
      -0.00519809, -0.000154171, -0.00125602, 4.03664e-08, -6.04796e-08,
      -4.6768e-08, -2.38233e-09, 2.31605e-09, 1.35922e-09;

  A.BSESinglets().eigenvectors() = spsi_ref;

  Orbitals B = A;
  B.QMAtoms().Translate(4 * Eigen::Vector3d::UnitX());

  Orbitals AB;
  AB.QMAtoms() = A.QMAtoms();
  AB.QMAtoms().AddContainer(B.QMAtoms());
  AB.MOs().eigenvalues().resize(34);
  AB.MOs().eigenvalues() << -10.1341, -10.1337, -0.808607, -0.665103, -0.474928,
      -0.455857, -0.455857, -0.365971, -0.365971, -0.263259, 0.140444, 0.154745,
      0.168775, 0.168775, 0.223948, 0.231217, 0.26323, 0.26323, 0.713478,
      0.713478, 0.793559, 0.885998, 0.944915, 0.944915, 1.01169, 1.04977,
      1.04977, 1.08863, 1.10318, 1.17822, 1.18094, 1.18094, 1.69037, 1.91046;
  std::string mos_ref_string =
      " -0.697309 -0.697192 -0.138562 0.154639 0.0435428 1.73438e-12 "
      "2.13093e-13 -8.65189e-14 -3.43449e-14 0.0581478 -0.13454 -0.0376033 "
      "5.46382e-12 8.3231e-14 0.111183 -0.00798105 1.59258e-13 1.63619e-12 "
      "2.3882e-12 -1.87201e-14 0.00654486 0.048026 3.94367e-12 -5.11935e-14 "
      "-0.0789143 3.6718e-14 1.52055e-12 -0.0134363 -0.00206854 -0.0770242 "
      "4.59464e-11 -2.01607e-13 -0.00162496 -0.0367745\n"
      " -0.0783921 -0.0769772 0.116956 -0.142166 -0.0554301 -3.50499e-12 "
      "-1.86741e-13 5.36568e-14 5.11008e-13 -0.0529852 0.0398041 0.00322307 "
      "-1.70454e-12 -1.13104e-13 -0.0504198 0.0144859 7.16041e-14 3.37842e-12 "
      "1.55705e-11 -9.16012e-14 0.110589 0.00602313 3.50438e-12 1.49731e-13 "
      "-0.0786935 7.1578e-13 8.77919e-12 -0.0741183 0.0994097 0.480567 "
      "-2.86802e-10 8.58629e-13 -1.41349 -1.37617\n"
      "-1.31663e-14 5.79687e-15 -9.12567e-13 -1.21281e-12 -9.79314e-12 "
      "0.179808 -0.175214 0.202197 0.194016 7.90197e-12 2.10679e-12 "
      "1.26699e-11 0.188156 0.116443 -7.73171e-12 -1.56645e-11 -0.255098 "
      "0.153207 -0.530812 0.119984 5.20064e-11 2.27813e-11 -0.194538 -0.39235 "
      "-3.45222e-11 -0.667081 -0.369257 3.03131e-11 -1.41125e-11 2.6854e-10 "
      "0.426545 0.234686 -1.21861e-12 7.38641e-13\n"
      "-1.97257e-14 -8.71107e-16 -6.32194e-13 -1.70457e-12 -1.12352e-11 "
      "0.180436 0.174567 -0.202601 0.193595 7.92911e-12 2.22435e-12 "
      "1.26533e-11 0.188465 -0.115943 -8.11243e-12 -1.54494e-11 0.256652 "
      "0.150589 -0.531118 -0.118621 5.18854e-11 2.30201e-11 -0.184528 0.397156 "
      "-3.33442e-11 0.667869 -0.36783 3.10521e-11 -1.40613e-11 2.68602e-10 "
      "0.4268 -0.234223 -9.00545e-13 9.31133e-13\n"
      " -0.00115489 0.000164776 0.0578394 0.0454767 0.251996 1.19171e-11 "
      "6.58841e-13 -2.64913e-13 1.44576e-11 -0.294909 -0.00802487 0.199834 "
      "-2.06998e-11 -1.42604e-13 0.151363 -0.218658 2.00335e-13 -2.66865e-12 "
      "3.61889e-11 -7.66537e-14 0.620011 -0.624288 -1.35313e-11 6.45206e-13 "
      "-0.0930979 -6.0734e-13 3.55302e-11 0.476225 0.450576 -0.323109 "
      "2.04055e-10 -7.22344e-13 0.0223133 -0.1443\n"
      " 0.060534 0.0561821 0.415836 -0.487318 -0.114633 1.54029e-12 "
      "-5.10868e-13 2.51948e-14 -2.28281e-12 -0.287976 1.82438 0.51819 "
      "-7.75715e-11 -1.07587e-12 -1.6604 0.033244 -2.25681e-12 -2.86385e-11 "
      "-2.19985e-12 1.54692e-13 0.1436 -0.365953 -3.10497e-11 2.09328e-14 "
      "0.237819 -1.55317e-12 -9.07554e-12 0.0534822 -0.272317 -0.708544 "
      "4.23309e-10 -9.43029e-13 2.71501 2.80085\n"
      " 2.08366e-13 -1.03736e-14 -1.14485e-12 1.42553e-13 -1.42216e-11 "
      "0.125879 -0.173468 0.166021 0.168907 -3.38267e-12 -4.79722e-12 "
      "5.09993e-11 0.639391 0.580945 -2.31268e-11 -4.28666e-11 -0.844352 "
      "0.75214 0.849064 -0.215142 -1.07708e-10 -3.4986e-11 -0.256404 1.12602 "
      "5.1508e-11 1.26525 0.754312 -6.50804e-11 3.10304e-11 -4.78706e-10 "
      "-0.741441 -0.00363983 1.77776e-12 -4.23163e-12\n"
      " 2.11528e-13 -6.42308e-15 -9.3007e-13 -2.46589e-13 -1.55798e-11 "
      "0.126502 0.173014 -0.166372 0.16856 -3.42581e-12 -5.1542e-12 "
      "5.02221e-11 0.640932 -0.579245 -2.38678e-11 -4.16714e-11 0.852002 "
      "0.743462 0.849613 0.212963 -1.07194e-10 -3.59559e-11 -0.284869 -1.11916 "
      "4.93541e-11 -1.26686 0.751605 -6.64254e-11 3.05998e-11 -4.76759e-10 "
      "-0.741444 0.00283525 1.41929e-12 -4.41113e-12\n"
      " 0.00325132 0.00072711 0.0753503 0.00950181 0.228655 1.62908e-11 "
      "5.88682e-13 -2.68345e-13 3.87226e-12 -0.28074 0.144232 0.869449 "
      "-8.4971e-11 -8.8691e-13 0.415771 -1.00835 4.58046e-13 -4.59974e-11 "
      "-1.24163e-10 3.99175e-13 -1.58015 0.920689 2.69425e-11 -1.12376e-12 "
      "0.183046 5.98902e-13 -2.88527e-11 -0.380313 -0.581711 0.578926 "
      "-3.6625e-10 1.2043e-12 -0.0303405 0.329309\n"
      " -0.0011464 0.00141574 0.102123 -0.062761 0.0756644 0.175473 "
      "-0.000315324 -0.00015246 0.14636 -0.14431 -0.0206219 -0.0846146 "
      "-0.0777281 -0.000103235 0.0556423 0.0507764 -0.000268375 -0.0524635 "
      "-0.511888 0.000656729 0.208213 -0.0190293 0.352659 -0.00447108 "
      "-0.391351 -0.000339021 0.317159 -0.515189 -0.458517 -0.471794 -0.629635 "
      "-0.000341627 0.0265973 -0.362435\n"
      " -0.013307 -0.00957248 -0.00732072 -0.00310248 0.0344914 0.171006 "
      "-0.000307297 -0.000150334 0.144318 -0.179084 -0.712591 -0.830754 "
      "-0.806581 -0.00107126 0.596627 0.752064 -0.00677785 -1.32497 0.0988994 "
      "-0.000126883 -0.474187 -0.145716 0.519372 -0.0065847 0.348342 "
      "0.00135561 -1.2682 0.790399 0.710069 0.136107 1.34459 0.000729549 "
      "-0.543167 -0.623166\n"
      " 8.55501e-05 0.000986043 0.0536401 -0.105273 -0.128709 -0.000224935 "
      "-0.125172 0.182036 0.000189623 0.101776 -0.00109152 0.0317062 "
      "0.000173788 -0.130849 0.00264037 -0.0712429 -0.0375076 0.000191869 "
      "-6.64731e-06 -0.00518125 -0.201684 0.377977 -0.00805124 -0.635046 "
      "-0.544273 0.358244 0.000382938 0.322411 0.418971 -0.334379 0.00033114 "
      "-0.610306 -0.0454904 -0.194439\n"
      " -0.0091205 -0.0117179 0.00848738 -0.0268224 -0.085757 -0.000140587 "
      "-0.0782345 0.184258 0.000191937 0.16027 -0.556338 0.388959 0.00126037 "
      "-0.948964 0.908403 -0.807446 1.13292 -0.00579543 0.000121528 0.0947251 "
      "-0.306129 0.0971475 -0.00206653 -0.162999 0.298931 -1.1321 -0.00121013 "
      "-0.454898 -0.615171 0.499217 -0.000372134 0.685859 -0.622467 -0.343531\n"
      " -0.0011464 0.00141574 0.102123 -0.062761 0.0756644 -0.175473 "
      "0.000315324 0.00015246 -0.14636 -0.14431 -0.0206219 -0.0846146 "
      "0.0777281 0.000103235 0.0556423 0.0507764 0.000268375 0.0524635 "
      "0.511888 -0.000656729 0.208213 -0.0190293 -0.352659 0.00447108 "
      "-0.391351 0.000339021 -0.317159 -0.515189 -0.458517 -0.471794 0.629635 "
      "0.000341627 0.0265973 -0.362435\n"
      " -0.013307 -0.00957248 -0.00732072 -0.00310248 0.0344914 -0.171006 "
      "0.000307297 0.000150334 -0.144318 -0.179084 -0.712591 -0.830754 "
      "0.806581 0.00107126 0.596627 0.752064 0.00677785 1.32497 -0.0988994 "
      "0.000126883 -0.474187 -0.145716 -0.519372 0.0065847 0.348342 "
      "-0.00135561 1.2682 0.790399 0.710069 0.136107 -1.34459 -0.000729549 "
      "-0.543167 -0.623166\n"
      " 8.55501e-05 0.000986043 0.0536401 -0.105273 -0.128709 0.000224935 "
      "0.125172 -0.182036 -0.000189623 0.101776 -0.00109152 0.0317062 "
      "-0.000173788 0.130849 0.00264037 -0.0712429 0.0375076 -0.000191869 "
      "6.6473e-06 0.00518125 -0.201684 0.377977 0.00805124 0.635046 -0.544273 "
      "-0.358244 -0.000382937 0.322411 0.418971 -0.334379 -0.00033114 0.610306 "
      "-0.0454904 -0.194439\n"
      " -0.0091205 -0.0117179 0.00848738 -0.0268224 -0.085757 0.000140587 "
      "0.0782345 -0.184258 -0.000191937 0.16027 -0.556338 0.388959 -0.00126037 "
      "0.948964 0.908403 -0.807446 -1.13292 0.00579543 -0.000121528 -0.0947251 "
      "-0.306129 0.0971475 0.00206653 0.162999 0.298931 1.1321 0.00121013 "
      "-0.454898 -0.615171 0.499217 0.000372133 -0.685859 -0.622467 -0.343531\n"
      " -0.697309 0.697192 -0.138562 -0.154639 0.0435428 1.49203e-12 "
      "9.77213e-14 1.24841e-14 1.25141e-12 -0.0581478 -0.13454 0.0376033 "
      "-7.21201e-12 7.1008e-14 -0.111183 -0.00798105 -9.94282e-15 -6.00058e-13 "
      "2.88443e-13 1.04188e-14 -0.00654486 0.048026 -1.03258e-12 -3.53311e-14 "
      "0.0789143 -1.02089e-13 -7.69953e-12 0.0134363 -0.00206854 -0.0770242 "
      "4.60228e-11 -1.45926e-13 0.00162496 -0.0367745\n"
      " -0.0783921 0.0769772 0.116956 0.142166 -0.0554301 -5.20584e-12 "
      "-1.00581e-13 1.64705e-14 -3.0152e-12 0.0529852 0.0398041 -0.00322307 "
      "2.08288e-13 -1.90771e-14 0.0504198 0.0144859 3.37325e-14 -1.30319e-13 "
      "-2.38237e-11 1.44013e-13 -0.110589 0.00602313 -3.17329e-12 -1.06756e-14 "
      "0.0786935 1.78234e-13 2.00784e-11 0.0741183 0.0994097 0.480567 "
      "-2.8943e-10 1.1105e-12 1.41349 -1.37617\n"
      "-1.94715e-14 1.93153e-14 -1.63987e-12 1.9805e-12 -5.25726e-12 0.174567 "
      "-0.180436 -0.193595 -0.202601 -6.02172e-12 5.04867e-12 -1.17883e-12 "
      "-0.115943 -0.188465 2.56675e-12 -2.29827e-12 -0.150589 0.256652 "
      "0.118621 -0.531118 -1.87996e-12 -1.40338e-11 0.397156 0.184528 "
      "-3.11885e-11 -0.36783 -0.667869 3.25477e-11 1.46489e-11 -1.16403e-10 "
      "-0.234223 -0.4268 5.13688e-12 1.85941e-12\n"
      "-2.76824e-14 2.71818e-14 -1.50687e-12 2.10656e-12 -6.03834e-12 0.175214 "
      "0.179808 0.194016 -0.202197 -6.09358e-12 5.43735e-12 -9.64377e-13 "
      "-0.116443 0.188156 2.44188e-12 -2.23432e-12 0.153207 0.255098 0.119984 "
      "0.530812 -1.64208e-12 -1.40718e-11 0.39235 -0.194538 -3.09928e-11 "
      "0.369257 -0.667081 3.30204e-11 1.49297e-11 -1.18997e-10 -0.234686 "
      "0.426545 5.05575e-12 1.94355e-12\n"
      " 0.00115489 0.000164776 -0.0578394 0.0454767 -0.251996 -5.69911e-12 "
      "-7.97142e-13 5.68508e-14 1.3035e-11 -0.294909 0.00802487 0.199834 "
      "7.61611e-12 -9.8441e-14 0.151363 0.218658 -5.89857e-14 8.58175e-12 "
      "1.10677e-10 -3.79358e-13 0.620011 0.624288 -7.96112e-12 -1.08888e-13 "
      "-0.0930979 1.20145e-13 3.63482e-11 0.476225 -0.450576 0.323109 "
      "-2.09343e-10 7.74471e-13 0.0223133 0.1443\n"
      " 0.060534 -0.0561821 0.415836 0.487318 -0.114633 4.73838e-12 "
      "-2.00065e-13 -3.44901e-14 -4.54885e-12 0.287976 1.82438 -0.51819 "
      "1.02866e-10 -8.89747e-13 1.6604 0.033244 2.01243e-13 4.8244e-12 "
      "-5.98145e-12 -2.17907e-13 -0.1436 -0.365953 1.00174e-11 2.37019e-13 "
      "-0.237819 -2.93644e-13 -1.26674e-11 -0.0534822 -0.272317 -0.708544 "
      "4.1826e-10 -1.73009e-12 -2.71501 2.80085\n"
      " 2.66098e-13 -3.10494e-13 5.25454e-13 -3.01832e-13 -8.47312e-12 "
      "0.173014 -0.126502 -0.16856 -0.166372 -2.37598e-12 -2.19896e-12 "
      "-3.16763e-11 -0.579245 -0.640932 1.7174e-11 1.07442e-11 -0.743462 "
      "0.852002 -0.212963 0.849613 -8.91957e-12 3.46585e-11 -1.11916 0.284869 "
      "5.32058e-11 0.751605 1.26686 -6.82198e-11 -9.26745e-12 -4.60515e-11 "
      "0.00283525 0.741444 -9.92385e-12 -1.24539e-12\n"
      " 2.71095e-13 -3.13867e-13 6.16964e-13 -2.6292e-13 -9.09244e-12 0.173468 "
      "0.125879 0.168907 -0.166021 -2.49777e-12 -9.54976e-13 -3.07486e-11 "
      "-0.580945 0.639391 1.64204e-11 1.15619e-11 0.75214 0.844352 -0.215142 "
      "-0.849064 -9.09168e-12 3.41739e-11 -1.12602 -0.256404 5.25065e-11 "
      "-0.754312 1.26525 -6.90363e-11 -9.8764e-12 -4.12883e-11 0.00363983 "
      "-0.741441 -9.93678e-12 -1.37646e-12\n"
      " -0.00325132 0.00072711 -0.0753503 0.00950181 -0.228655 -1.83426e-11 "
      "-6.23026e-13 1.16744e-13 -1.52124e-12 -0.28074 -0.144232 0.869449 "
      "-8.0255e-12 -3.3796e-13 0.415771 1.00835 -2.11614e-13 8.62638e-13 "
      "-2.5409e-10 9.75851e-13 -1.58015 -0.920689 2.99441e-11 -5.07104e-14 "
      "0.183046 -4.50746e-13 -4.51192e-11 -0.380313 0.581711 -0.578926 "
      "3.56562e-10 -1.33803e-12 -0.0303405 -0.329309\n"
      " 8.55501e-05 -0.000986043 0.0536401 0.105273 -0.128709 0.125172 "
      "-0.000224935 0.000189623 -0.182036 -0.101776 -0.00109152 -0.0317062 "
      "0.130849 0.000173788 -0.00264037 -0.0712429 0.000191869 0.0375076 "
      "-0.00518125 6.64731e-06 0.201684 0.377977 0.635046 -0.00805124 0.544273 "
      "-0.000382938 0.358244 -0.322411 0.418971 -0.334379 0.610306 0.00033114 "
      "0.0454904 -0.194439\n"
      " -0.0091205 0.0117179 0.00848738 0.0268224 -0.085757 0.0782345 "
      "-0.000140587 0.000191937 -0.184258 -0.16027 -0.556338 -0.388959 "
      "0.948964 0.00126037 -0.908403 -0.807446 -0.00579543 -1.13292 0.0947251 "
      "-0.000121528 0.306129 0.0971475 0.162999 -0.00206653 -0.298931 "
      "0.00121013 -1.1321 0.454898 -0.615171 0.499217 -0.685859 -0.000372134 "
      "0.622467 -0.343531\n"
      " -0.0011464 -0.00141574 0.102123 0.062761 0.0756644 -0.000315324 "
      "-0.175473 -0.14636 -0.00015246 0.14431 -0.0206219 0.0846146 "
      "-0.000103235 0.0777281 -0.0556423 0.0507764 0.0524635 -0.000268375 "
      "-0.000656729 -0.511888 -0.208213 -0.0190293 -0.00447108 -0.352659 "
      "0.391351 0.317159 0.000339021 0.515189 -0.458517 -0.471794 -0.000341627 "
      "0.629635 -0.0265973 -0.362435\n"
      " -0.013307 0.00957248 -0.00732072 0.00310248 0.0344914 -0.000307297 "
      "-0.171006 -0.144318 -0.000150334 0.179084 -0.712591 0.830754 "
      "-0.00107126 0.806581 -0.596627 0.752064 1.32497 -0.00677785 0.000126883 "
      "0.0988994 0.474187 -0.145716 -0.0065847 -0.519372 -0.348342 -1.2682 "
      "-0.00135561 -0.790399 0.710069 0.136107 0.000729549 -1.34459 0.543167 "
      "-0.623166\n"
      " 8.55501e-05 -0.000986043 0.0536401 0.105273 -0.128709 -0.125172 "
      "0.000224935 -0.000189623 0.182036 -0.101776 -0.00109152 -0.0317062 "
      "-0.130849 -0.000173788 -0.00264037 -0.0712429 -0.000191869 -0.0375076 "
      "0.00518125 -6.64731e-06 0.201684 0.377977 -0.635046 0.00805124 0.544273 "
      "0.000382938 -0.358244 -0.322411 0.418971 -0.334379 -0.610306 "
      "-0.00033114 0.0454904 -0.194439\n"
      " -0.0091205 0.0117179 0.00848738 0.0268224 -0.085757 -0.0782345 "
      "0.000140587 -0.000191937 0.184258 -0.16027 -0.556338 -0.388959 "
      "-0.948964 -0.00126037 -0.908403 -0.807446 0.00579543 1.13292 -0.0947251 "
      "0.000121528 0.306129 0.0971475 -0.162999 0.00206653 -0.298931 "
      "-0.00121013 1.1321 0.454898 -0.615171 0.499217 0.685859 0.000372134 "
      "0.622467 -0.343531\n"
      " -0.0011464 -0.00141574 0.102123 0.062761 0.0756644 0.000315324 "
      "0.175473 0.14636 0.00015246 0.14431 -0.0206219 0.0846146 0.000103235 "
      "-0.0777281 -0.0556423 0.0507764 -0.0524635 0.000268375 0.000656729 "
      "0.511888 -0.208213 -0.0190293 0.00447108 0.352659 0.391351 -0.317159 "
      "-0.000339021 0.515189 -0.458517 -0.471794 0.000341628 -0.629635 "
      "-0.0265973 -0.362435\n"
      " -0.013307 0.00957248 -0.00732072 0.00310248 0.0344914 0.000307297 "
      "0.171006 0.144318 0.000150334 0.179084 -0.712591 0.830754 0.00107126 "
      "-0.806581 -0.596627 0.752064 -1.32497 0.00677785 -0.000126883 "
      "-0.0988994 0.474187 -0.145716 0.0065847 0.519372 -0.348342 1.2682 "
      "0.00135561 -0.790399 0.710069 0.136107 -0.000729549 1.34459 0.543167 "
      "-0.623166";

  AB.setBasisSetSize(34);
  AB.setNumberOfAlphaElectrons(10);
  AB.setNumberOfOccupiedLevels(10);
  AB.setDFTbasisName(A.getDFTbasisName());
  AB.setAuxbasisName(A.getDFTbasisName());
  AB.setRPAindices(0, 33);
  AB.setBSEindices(0, 33);
  AB.setGWindices(0, 33);
  AB.MOs().eigenvectors() = ReadMatrixFromString(mos_ref_string);
  AB.setNumberOfAlphaElectrons(10);
  AB.setNumberOfOccupiedLevels(10);

  std::ofstream opt("bsecoupling.xml");
  opt << "<bsecoupling>" << std::endl;
  opt << "        <spin>singlet</spin>" << std::endl;
  opt << "       <moleculeA>" << std::endl;
  opt << "                <states>1</states>" << std::endl;
  opt << "                <occLevels>3</occLevels>" << std::endl;
  opt << "                <unoccLevels>3</unoccLevels>" << std::endl;
  opt << "        </moleculeA>" << std::endl;
  opt << "        <moleculeB>" << std::endl;
  opt << "                <states>1</states>" << std::endl;
  opt << "                <occLevels>3</occLevels>" << std::endl;
  opt << "                <unoccLevels>3</unoccLevels>" << std::endl;
  opt << "         </moleculeB>" << std::endl;
  opt << "</bsecoupling>" << std::endl;
  opt.close();
  votca::tools::Property prop;
  prop.LoadFromXML("bsecoupling.xml");
  BSECoupling coup;
  Logger log;
  log.setCommonPreface("\n... ...");
  coup.setLogger(&log);

  AB.QPdiag().eigenvalues().resize(34);
  AB.QPdiag().eigenvalues() << -10.504, -10.5038, -0.923616, -0.775673,
      -0.549084, -0.530193, -0.530193, -0.430293, -0.430293, -0.322766,
      0.267681, 0.307809, 0.326961, 0.326961, 0.36078, 0.381947, 0.414845,
      0.414845, 0.906609, 0.906609, 0.993798, 1.09114, 1.14639, 1.14639, 1.1966,
      1.25629, 1.25629, 1.27991, 1.29122, 1.35945, 1.36705, 1.36705, 1.93286,
      2.11739;

  std::string Hqp_input_string =
      "-0.999989 -1.5451e-09 0.00372795 -2.38522e-11 -0.000933189 -1.22943e-13 "
      "3.53006e-15 -7.76284e-15 -2.5356e-16 7.36926e-12 0.00235453 4.89519e-12 "
      "1.46771e-16 -1.76571e-14 -1.09615e-11 0.000163686 -7.03225e-14 "
      "-5.33295e-15 8.9466e-14 -2.25942e-15 1.46429e-12 -0.00062374 "
      "-1.4634e-13 7.8008e-16 -6.45351e-12 -3.03299e-14 -9.73079e-15 "
      "-6.88974e-13 9.83765e-05 0.0010822 7.20777e-14 -1.08121e-15 "
      "-4.25014e-13 -0.000246801\n"
      " 1.5451e-09 -0.999989 -2.24884e-11 -0.00393612 5.84299e-12 2.46391e-14 "
      "-2.86419e-17 9.31348e-14 -3.85249e-15 0.00124745 -1.41626e-11 "
      "0.000817314 -1.26996e-15 9.91057e-14 -0.00181263 -1.02145e-12 "
      "4.71046e-14 -2.60884e-16 5.12342e-14 3.48249e-15 0.000204578 "
      "3.52763e-12 1.10526e-13 -3.48859e-15 -0.00112107 3.12729e-14 "
      "1.88679e-14 -7.51946e-05 -7.24006e-13 -6.28809e-12 4.01043e-14 "
      "-6.53798e-15 -7.92482e-05 1.38198e-12\n"
      " 0.00372932 -1.69262e-11 0.999162 5.99257e-11 0.0351359 4.77947e-12 "
      "-3.6751e-13 -1.10823e-12 1.33202e-13 3.78631e-11 0.0175803 9.67852e-12 "
      "2.1705e-14 -1.60824e-13 8.51535e-12 -0.0101988 6.34651e-13 1.15307e-13 "
      "-5.02319e-13 1.24342e-14 -1.86242e-11 -0.000772021 2.10444e-13 "
      "4.38956e-15 5.8451e-13 -1.036e-12 -6.11664e-13 -1.24689e-11 0.00149291 "
      "-0.00237699 2.54818e-13 -1.43923e-14 -3.50306e-12 0.0021897\n"
      "-1.83205e-11 -0.00398209 -6.40837e-11 0.999692 1.06666e-10 -2.46919e-13 "
      "3.79346e-13 -3.3749e-12 -5.57557e-13 -0.0022922 -1.94491e-12 -0.0045807 "
      "-1.06975e-13 -9.86615e-13 0.0200591 -2.22577e-11 -9.54711e-13 "
      "-1.01642e-13 1.21591e-12 2.97119e-14 -0.00288919 -3.17718e-11 "
      "2.73013e-13 -3.10979e-14 0.0023512 1.09619e-12 6.34163e-13 -0.000602002 "
      "2.57649e-11 -1.64942e-11 -9.49244e-13 3.23744e-14 0.0125641 "
      "1.02804e-11\n"
      " -0.00108549 5.4775e-12 -0.035344 -1.09719e-10 0.998599 7.58752e-11 "
      "3.0362e-12 -1.19749e-11 -7.46719e-13 -6.55113e-11 -0.00844963 "
      "-2.19447e-11 -9.39782e-14 -2.23764e-12 -2.12282e-11 -0.0365615 "
      "1.4498e-12 -9.36067e-14 -3.27523e-12 1.91968e-14 4.10558e-11 "
      "-0.00717143 1.82389e-12 -2.72526e-14 -1.3786e-11 1.15424e-12 "
      "6.76643e-13 3.71707e-11 0.00662974 -0.0024409 -8.26991e-13 4.06132e-14 "
      "3.86272e-12 -0.00632589\n"
      " 6.71094e-14 -2.35237e-14 2.12448e-12 -2.68406e-13 7.61678e-11 "
      "-0.999379 -0.000888538 0.00733481 -2.16906e-05 3.08943e-12 1.72853e-12 "
      "1.14416e-12 -7.2077e-05 -0.00872183 6.94414e-13 2.36836e-13 -0.0315708 "
      "-0.000632677 -0.00398823 1.50229e-05 -1.80609e-12 2.82112e-12 "
      "-0.00587235 3.05112e-05 9.16606e-13 0.00701601 0.00380514 -8.90359e-13 "
      "-2.68379e-12 2.06425e-12 0.000354593 -2.14337e-05 -4.12271e-13 "
      "-1.97349e-12\n"
      "-8.20874e-15 -1.53436e-15 -4.80136e-13 3.79419e-13 2.95106e-12 "
      "0.000888538 -0.999379 -2.16906e-05 -0.00733481 -1.91279e-13 5.81924e-14 "
      "-1.72157e-14 0.00872183 -7.2077e-05 -5.24087e-14 -5.69276e-14 "
      "-0.000632677 0.0315708 1.50229e-05 0.00398823 3.95501e-14 -2.90234e-14 "
      "-3.05112e-05 -0.00587235 5.78015e-15 0.00380514 -0.00701601 "
      "-8.98288e-15 4.73243e-14 -1.06583e-15 -2.14337e-05 -0.000354593 "
      "-6.65604e-15 6.69116e-15\n"
      " 1.49396e-15 5.57347e-15 1.63419e-13 -5.74417e-13 -7.92216e-13 "
      "-9.40995e-07 0.00703423 -0.00220003 -0.999155 -2.99257e-13 -5.57734e-14 "
      "-8.50978e-15 -0.0379469 0.000342257 7.39195e-14 1.72487e-14 2.37624e-05 "
      "-0.00114266 2.58274e-05 0.0085763 -1.10182e-14 -3.06282e-14 "
      "-4.91143e-05 -0.00825301 -9.18751e-15 -0.00100694 0.00185329 2.3706e-14 "
      "-3.25676e-14 -3.45312e-15 -0.000409902 -0.00686742 2.16647e-14 "
      "1.95942e-15\n"
      "-1.88578e-14 1.0893e-13 7.10193e-13 3.42394e-12 1.15947e-11 0.00703423 "
      "9.40995e-07 0.999155 -0.00220003 2.53084e-11 -1.39783e-12 2.85823e-12 "
      "0.000342257 0.0379469 3.47592e-12 1.38347e-12 0.00114266 2.37624e-05 "
      "-0.0085763 2.58274e-05 -4.61787e-12 6.57237e-13 -0.00825301 4.91143e-05 "
      "4.33757e-13 -0.00185329 -0.00100694 -4.08242e-12 -7.66775e-13 "
      "9.36645e-13 0.00686742 -0.000409902 -3.9064e-13 -8.93727e-13\n"
      " -5.1976e-12 -0.00124624 4.14538e-11 -0.00216962 -6.5007e-11 "
      "-3.01848e-12 1.86476e-13 2.55454e-11 2.4833e-13 -0.999008 1.65956e-12 "
      "-0.0395362 -8.80982e-14 -5.40678e-13 -0.0136391 2.41435e-11 2.25749e-12 "
      "4.75754e-14 -5.77805e-12 5.56285e-14 0.0111238 2.09176e-11 -1.25159e-12 "
      "3.75023e-14 0.00222786 -2.51397e-14 -6.86836e-15 0.0099029 -2.93272e-11 "
      "2.49245e-11 -2.76428e-13 -4.45294e-15 -4.7204e-05 -7.71887e-12\n"
      " 0.00230565 -1.02845e-11 -0.0178163 4.55167e-14 0.00789831 2.60615e-12 "
      "1.01752e-13 1.58465e-12 -4.98211e-14 1.16215e-12 0.999587 -1.07339e-11 "
      "-5.5817e-13 -7.40608e-12 -1.04323e-11 0.00321105 -6.1131e-12 "
      "2.68013e-13 -3.07816e-13 2.51601e-14 1.49259e-12 -0.00960228 "
      "8.06839e-13 7.93067e-15 -1.41701e-12 5.70903e-12 2.98743e-12 "
      "1.98691e-13 0.00469998 0.0177193 1.80929e-12 -5.93327e-14 9.10244e-13 "
      "0.00121306\n"
      " 2.50831e-12 0.000575921 -1.0887e-11 0.0024431 2.10546e-11 1.13166e-12 "
      "1.90254e-16 -4.64874e-12 2.92354e-15 -0.0410343 1.17053e-11 0.99371 "
      "9.69588e-13 6.29652e-11 0.0982449 -3.78327e-11 -1.87343e-11 "
      "-9.78746e-14 4.40994e-12 -4.24759e-14 -0.00153937 -9.87357e-13 "
      "2.51256e-13 4.0783e-14 -0.000437221 3.41288e-12 1.84851e-12 -0.0345352 "
      "9.88116e-12 -1.40205e-11 2.55703e-12 -1.49202e-13 0.00182007 "
      "-7.13441e-13\n"
      " 6.60775e-15 -9.22101e-15 4.053e-13 7.82213e-15 4.31088e-12 -0.00715341 "
      "1.59955e-05 -0.0380437 -6.38925e-06 6.967e-13 6.98384e-12 -6.70468e-11 "
      "0.0113553 0.997003 2.65434e-11 4.86379e-11 -0.0574615 -0.00133118 "
      "-0.0102801 6.59617e-06 -2.12785e-12 5.53642e-12 0.0018611 -1.54863e-05 "
      "-1.06339e-13 -0.00321642 -0.00175746 2.47457e-13 -4.11948e-12 "
      "-4.47323e-13 0.030459 -0.0017456 -4.98476e-13 -1.28487e-12\n"
      "-1.67242e-15 2.94931e-15 2.28802e-14 -6.6276e-14 9.20727e-16 "
      "-1.59955e-05 -0.00715341 -6.38925e-06 0.0380437 8.20798e-14 "
      "-4.93198e-13 2.42184e-13 -0.997003 0.0113553 -4.36445e-14 3.95048e-13 "
      "-0.00133118 0.0574615 6.59617e-06 0.0102801 5.58551e-15 1.29608e-14 "
      "1.54863e-05 0.00186111 1.92241e-14 -0.00175746 0.00321642 8.31444e-15 "
      "2.67237e-16 2.83709e-14 -0.0017456 -0.030459 3.07856e-14 -1.61097e-14\n"
      "-8.34839e-12 -0.00184616 -8.31532e-12 -0.0207277 1.22627e-11 "
      "7.86697e-13 -6.8981e-14 -1.7309e-12 9.98819e-14 -0.0098459 9.74065e-12 "
      "-0.0992103 -4.54327e-13 -3.30151e-11 0.994129 -1.3494e-10 3.1847e-12 "
      "-2.54992e-13 4.04396e-12 2.34126e-14 -0.0137533 5.10711e-12 -1.9934e-12 "
      "-1.46431e-14 0.0263529 6.64298e-13 4.79646e-13 -0.0153298 8.95804e-12 "
      "-9.75042e-12 1.44529e-12 -1.44311e-13 0.0146984 -1.88644e-12\n"
      " 0.000154896 -1.00496e-12 0.00900831 1.72852e-11 0.0371703 2.83715e-12 "
      "3.39466e-14 -7.63463e-14 -1.8015e-14 2.05689e-11 -0.00323179 "
      "2.48339e-11 -1.83551e-13 -4.7589e-11 1.38077e-10 0.998318 2.36548e-11 "
      "-7.99499e-14 5.94604e-12 -4.48454e-14 3.26666e-12 0.000666957 "
      "-2.39037e-12 -2.98327e-14 1.63429e-11 -2.63416e-12 -1.49026e-12 "
      "-5.34484e-12 -0.0429308 0.00535215 2.92935e-13 2.1944e-14 6.38761e-12 "
      "0.00387241\n"
      "-4.71745e-15 -2.88924e-15 -7.64108e-14 6.03629e-14 4.05836e-14 "
      "0.000195433 0.0324457 1.23363e-05 -0.00311897 -1.06893e-15 -4.21262e-13 "
      "-4.41414e-13 0.0568027 -0.000862115 3.02765e-13 6.80818e-13 -0.0268515 "
      "0.996115 3.37016e-05 -0.0107138 1.80815e-14 1.75274e-15 -0.000199331 "
      "-0.0164618 1.61364e-14 0.0234398 -0.042515 -3.11586e-14 6.9636e-14 "
      "5.09807e-14 -6.61382e-05 -0.00123597 -4.06351e-14 3.58609e-14\n"
      " 5.89067e-14 -6.31069e-14 8.88852e-13 -1.13824e-12 -3.5576e-13 "
      "0.0324457 -0.000195433 0.00311897 1.23363e-05 -1.76099e-12 -6.92939e-12 "
      "-1.55804e-11 -0.000862115 -0.0568027 -6.14802e-13 2.07864e-11 -0.996115 "
      "-0.0268515 0.0107138 3.37016e-05 -6.70981e-12 1.24245e-12 -0.0164618 "
      "0.000199331 1.63367e-12 0.042515 0.0234398 -3.51725e-12 -8.33984e-13 "
      "1.08218e-13 0.00123597 -6.61382e-05 -8.63107e-13 8.39857e-13\n"
      " 7.25865e-14 2.67292e-14 4.01175e-13 -1.38864e-12 3.95715e-12 "
      "-0.00464296 1.01721e-05 0.00852102 1.04591e-06 -6.74958e-12 9.03238e-13 "
      "-5.48799e-12 0.000137926 0.0121582 -6.29987e-12 -6.54417e-12 0.00817104 "
      "0.000188925 0.998424 -0.000685764 -1.1491e-10 4.84238e-11 0.0204144 "
      "-0.000168946 8.11452e-12 -0.0244273 -0.0133457 -2.52236e-11 "
      "-1.67194e-11 1.84818e-11 -0.0405378 0.00232505 -5.38097e-12 "
      "-6.38328e-12\n"
      "-2.41302e-15 3.9553e-15 -1.21124e-14 -2.53937e-14 -2.35179e-14 "
      "1.01721e-05 0.00464296 -1.04591e-06 0.00852102 5.66772e-14 -2.66394e-14 "
      "3.91526e-14 0.0121582 -0.000137926 -1.42398e-14 4.42405e-14 "
      "-0.000188925 0.00817104 0.000685764 0.998424 3.75585e-13 -1.87935e-13 "
      "-0.000168946 -0.0204144 -2.62571e-14 0.0133457 -0.0244273 2.35468e-14 "
      "6.2738e-14 3.64967e-14 -0.00232505 -0.0405378 6.66107e-14 3.0702e-14\n"
      "-1.34726e-12 -0.000189882 -2.0427e-11 -0.00267468 4.21541e-11 "
      "2.04654e-12 -3.26125e-14 -5.27349e-12 2.61258e-14 -0.0114524 "
      "3.32253e-12 -0.00295641 -3.25668e-14 -2.87567e-12 -0.0156854 "
      "7.69252e-12 6.14226e-12 1.94165e-13 -1.17043e-10 4.58893e-13 -0.997134 "
      "1.36769e-10 1.92835e-11 -2.05444e-13 0.0121217 7.95136e-12 4.37457e-12 "
      "-0.0720012 7.88588e-11 -9.88252e-11 -1.56511e-12 1.29337e-13 "
      "-0.000383389 -7.92573e-12\n"
      "-0.000565136 1.97519e-12 0.000538042 3.34063e-11 0.0076554 3.98467e-12 "
      "-9.51335e-15 -1.21018e-12 -3.66603e-14 2.41808e-11 0.00917663 "
      "-2.76805e-12 -6.17273e-14 -6.32796e-12 -1.30255e-11 -0.00460675 "
      "1.03244e-12 5.36006e-14 -5.01869e-11 2.4367e-13 1.44637e-10 0.995663 "
      "-1.37468e-11 7.25191e-13 2.80858e-10 -5.25453e-12 -2.76091e-12 "
      "-2.08222e-10 -0.0812771 0.0421835 4.08304e-12 -1.81714e-13 1.59472e-11 "
      "-0.0102723\n"
      " -1.6932e-13 8.45765e-14 9.48149e-15 -3.93464e-13 -1.39628e-12 "
      "-0.0046866 -5.5162e-05 0.0086769 -0.000120076 -5.84394e-13 -1.07225e-12 "
      "-5.643e-13 1.10273e-06 -0.000421415 1.82385e-12 3.51648e-12 -0.0130633 "
      "-0.00011962 -0.020585 0.000301536 2.24314e-11 1.45274e-11 0.994865 "
      "0.0056557 2.35249e-11 0.0684075 0.0361432 2.12923e-12 9.09677e-12 "
      "-6.69577e-12 -0.0593002 0.00423248 -1.72083e-12 2.42722e-12\n"
      " 1.66089e-15 -4.5156e-15 -1.20364e-14 3.49767e-14 4.12227e-14 "
      "5.5162e-05 -0.0046866 -0.000120076 -0.0086769 3.19374e-14 -2.20128e-14 "
      "-4.74726e-14 0.000421415 1.10273e-06 8.44509e-15 1.0755e-14 -0.00011962 "
      "0.0130633 0.000301536 0.020585 -3.3572e-13 -7.95014e-13 -0.0056557 "
      "0.994865 -4.99338e-14 0.0361432 -0.0684075 -1.93202e-13 4.37548e-14 "
      "-1.2039e-14 0.00423248 0.0593002 -8.71549e-14 2.45432e-14\n"
      " 4.16885e-12 0.00105341 8.70692e-13 0.00192003 -1.24182e-11 "
      "-1.51601e-12 -3.18957e-15 7.08208e-13 6.31693e-15 -0.00228822 "
      "2.17435e-12 -0.00174832 -5.23138e-14 -1.50568e-12 0.0269109 2.29916e-11 "
      "-5.79689e-12 -1.62093e-13 4.66037e-12 -2.54207e-14 -0.0155554 "
      "3.1453e-10 2.95991e-11 -3.69461e-14 -0.998571 -6.69648e-11 -3.74212e-11 "
      "0.0418692 2.18585e-10 -7.27867e-11 -2.89758e-12 4.95038e-13 -0.0111587 "
      "2.04024e-11\n"
      "-5.35849e-15 -2.53342e-15 -2.97709e-14 2.0459e-14 6.99945e-15 "
      "1.31511e-05 0.00665261 -1.30698e-07 -0.00143182 -5.1832e-15 -1.098e-13 "
      "1.27814e-14 -0.00759724 8.45586e-05 7.36904e-14 -6.32567e-14 0.00114113 "
      "-0.0498156 -2.53466e-05 -0.0281356 -5.42449e-14 1.6193e-13 -0.000622099 "
      "-0.0771663 7.58368e-13 0.476951 -0.873432 3.52224e-13 4.84779e-13 "
      "6.70545e-13 0.000996504 0.0173094 -1.23506e-13 1.13446e-13\n"
      " 6.87343e-14 -1.12854e-13 -1.28664e-12 9.98637e-13 1.50195e-12 "
      "-0.00665261 1.31511e-05 -0.00143182 1.30698e-07 -5.02122e-13 "
      "5.23634e-12 1.56324e-12 -8.45586e-05 -0.00759724 -1.87876e-12 "
      "-1.137e-12 -0.0498156 -0.00114113 -0.0281356 2.53466e-05 7.68618e-13 "
      "-6.0523e-12 0.0771663 -0.000622099 7.61936e-11 -0.873432 -0.476951 "
      "-6.08936e-11 2.04535e-11 3.10467e-11 0.0173094 -0.000996504 4.70933e-12 "
      "-1.45696e-12\n"
      " 6.78169e-13 0.000131953 -1.20395e-11 -5.4882e-05 2.89732e-11 "
      "9.72917e-13 1.66693e-14 -3.4173e-12 -1.39163e-14 -0.00762804 "
      "3.78962e-13 -0.0331848 -4.10327e-14 -1.30957e-12 -0.0165186 2.93945e-12 "
      "6.30538e-12 9.67312e-14 -1.49564e-11 -1.27078e-15 0.0718244 "
      "-1.90562e-10 -2.71853e-12 -1.65395e-13 -0.0431741 5.07871e-11 "
      "2.73112e-11 -0.995746 9.66585e-11 -2.01485e-10 7.16198e-12 -2.88322e-13 "
      "-0.0058238 -2.52523e-11\n"
      "-5.03803e-05 -1.47486e-13 -0.000970771 -2.26788e-11 -0.00470274 "
      "-2.8694e-12 2.05923e-14 1.01876e-12 -2.89068e-14 -2.52583e-11 "
      "-0.00233842 -5.76157e-12 5.99828e-14 3.22621e-12 -9.1205e-12 0.043258 "
      "9.44681e-13 -5.54223e-14 1.31621e-11 -5.72939e-14 9.60207e-11 0.0849754 "
      "-1.47374e-11 4.72602e-14 2.53925e-10 1.25731e-11 7.36282e-12 "
      "9.52067e-11 0.99122 -0.0912456 -4.68636e-11 2.33171e-12 -2.9977e-11 "
      "-0.00612544\n"
      " 0.00107035 -4.53031e-12 0.00244279 1.21399e-11 0.00146631 2.1099e-12 "
      "-1.82329e-16 -4.22108e-12 -2.29972e-14 1.78901e-11 -0.0183782 "
      "6.39907e-12 -2.9107e-13 -1.89701e-11 7.4852e-12 -0.00139693 3.80889e-12 "
      "7.04533e-14 1.2794e-11 1.14886e-13 -8.34596e-11 -0.0342301 4.29757e-11 "
      "-4.53035e-13 -6.93816e-11 4.14605e-11 2.35369e-11 -1.89318e-10 "
      "0.0945585 0.994746 6.68913e-10 -3.46217e-11 2.91379e-11 0.00433428\n"
      "-6.60833e-13 4.71631e-14 -1.9179e-12 8.59146e-13 -6.47531e-13 "
      "0.000133318 -1.93376e-07 -0.00484862 2.99465e-06 -6.2472e-13 "
      "1.00653e-11 -4.28477e-13 -0.0003192 -0.0301025 -2.46096e-12 1.1186e-12 "
      "0.00341965 7.65337e-05 0.0402955 -5.75106e-05 -5.35859e-12 2.56676e-11 "
      "0.0588489 -0.000443451 -2.22385e-12 0.0183088 0.0099853 7.46672e-12 "
      "-1.69149e-11 -6.7311e-10 0.995085 -0.0578125 -5.01451e-13 -6.1312e-12\n"
      " 1.57318e-15 4.26184e-15 8.77151e-15 -1.36523e-14 9.26493e-15 "
      "1.93376e-07 0.000133318 2.99465e-06 0.00484862 1.76391e-14 -1.76465e-14 "
      "-1.17391e-14 0.0301025 -0.0003192 -5.11735e-14 5.30757e-15 7.65337e-05 "
      "-0.00341965 -5.75106e-05 -0.0402955 -4.88528e-14 -1.51085e-13 "
      "0.000443451 0.0588489 -3.05335e-13 0.0099853 -0.0183088 -1.21247e-13 "
      "2.36626e-14 4.26735e-12 -0.0578125 -0.995085 1.15763e-13 -7.54793e-15\n"
      " 9.71473e-14 9.26568e-06 4.46688e-12 -0.0122437 -6.04994e-12 "
      "-4.0779e-13 -9.46969e-15 5.0583e-13 2.27289e-14 0.000126831 "
      "-5.42585e-13 -0.000508673 4.01134e-14 9.01939e-13 -0.0148497 "
      "-2.0982e-12 -6.70484e-13 1.99513e-14 5.32734e-12 -6.84731e-14 "
      "0.000104246 -9.96658e-12 1.81611e-12 8.19651e-14 -0.0118088 3.7299e-12 "
      "1.87532e-12 -0.00506458 3.08663e-11 -3.39156e-11 1.18599e-13 "
      "1.26549e-13 0.999732 -4.06386e-11\n"
      " 0.000276012 -1.13672e-12 0.00243508 1.1633e-11 -0.0061305 1.56017e-12 "
      "-2.30061e-14 -7.94773e-13 4.16571e-15 8.5302e-12 0.00113194 9.17348e-13 "
      "1.36087e-15 -1.44523e-12 9.57373e-13 0.00385543 -6.58094e-13 "
      "1.09904e-14 -6.05954e-12 3.12691e-14 2.9577e-12 -0.010863 2.2124e-12 "
      "2.28071e-14 -2.35353e-11 1.71015e-12 7.96531e-13 2.72019e-11 "
      "-0.00502671 0.00448979 -2.90263e-12 2.01326e-13 -4.04695e-11 -0.999888";
  AB.QPdiag().eigenvectors() = ReadMatrixFromString(Hqp_input_string);

  coup.Initialize(prop);
  log.setReportLevel(Log::error);
  coup.CalculateCouplings(A, B, AB);
  votca::tools::Property output;
  coup.Addoutput(output, A, B);
  double diag_J_ref = 32.67651;
  double pert_J_ref = 4.434018;

  double diag_j =
      output.get("bsecoupling.singlet.coupling").getAttribute<double>("j_diag");
  double pert_j =
      output.get("bsecoupling.singlet.coupling").getAttribute<double>("j_pert");

  BOOST_CHECK_CLOSE(diag_J_ref, diag_j, 1e-4);
  BOOST_CHECK_CLOSE(pert_J_ref, pert_j, 1e-4);
}
BOOST_AUTO_TEST_SUITE_END()
