/*
  Copyright (C) 2000-2008

  Code contributed by Greg Collecutt, Joseph Hope, Andrew Reid and Paul Cochrane

  This file is part of xmds.

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/*
  $Id: xsil2graphics.cc 1993 2008-04-16 23:59:10Z andyferris $
*/

/*! @file xsil2graphics.cc
  @brief the xsil2graphics utility program classes and methods

  More detailed explanation...
*/

// This program converts xmds field data objects in XSIL
// format to various other input formats

#include <config.h>
#include <xmds_common.h>
#include <kissdom.h>
#include <xml_parser.h>
#include <xsil_field.h>
#include <getopt_xmds.h>
#include <iostream>
#include <cstring>

// **********************************************************************

//! Display the usage of xsil2graphics
void display_usage() {
  cout << "\n";
  cout << "xsil2graphics utility supplied with xmds version ";
  cout << VERSION;
  cout << "\n"
          "\n"
          "Usage: xsil2graphics [options] infile(s)\n"
          "Options:\n"
          "  infile(s):        required, the input xsil file or files\n"
          "  -h/--help:        optional, display this information\n"
          "  -m/--matlab:      optional, produce matlab/octave output (default)\n"
          "  -s/--scilab:      optional, produce scilab output\n"
          "  -a/--mathematica: optional, produce mathematica 5.x output\n"
          "  -e/--mathematica: optional, produce mathematica output\n"
          "  -g/--gnuplot:     optional, produce gnuplot output\n"
          "  -r/--R:           optional, produce R output\n"
          "  -p/--plot:        optional, generate plotting output (matlab/octave)\n"
          "  -o/--outfile:     optional, alternate output file name (one input file only)\n"
          "  -v/--verbose:     optional, verbose output\n"
          "\n"
          "For further help, please see http://www.xmds.org\n";
}

bool debugFlag = 0;
bool xmlDebugFlag = 0;

// **********************************************************************

//! The main routine of xsil2graphics
/*!
  @param argc The number of arguments to the program
  @param argv The "vector" of arguments to the program
*/
int main(
         int argc,
         char **argv) {

  bool verbose = 0;
  XMLString inFileName;
  XMLString * inFileNames;
  XMLString outFileName = "";
  XMLString outFileNameBase;
  outputFormatEnum outFormat = FORMAT_NONE;
  bool plotFlag = false;

  // process arguments
  int resp;
  while (1) {
    static struct option long_options[] =
      {
        {"help", no_argument, 0, 'h'},
        {"verbose", no_argument, 0, 'v'},
        {"matlab", no_argument, 0, 'm'},
        {"scilab", no_argument, 0, 's'},
        {"mathematica5", no_argument, 0, 'a'},
        {"mathematica", no_argument, 0, 'e'},
        {"gnuplot", no_argument, 0, 'g'},
        {"R", no_argument, 0, 'r'},
        {"plot", no_argument, 0, 'p'},
        {"outfile", required_argument, 0, 'o'},
        {"debug", no_argument, 0, 'd'},
        {"xmldebug", no_argument, 0, 'x'},
        {0, 0, 0, 0}
      };
    int option_index = 0;
    resp = getopt_xmds_long(argc, argv, "hvdxmsaegrpo:", long_options, &option_index);
    if (resp == -1) {
      break;
    }
    switch (resp) {
    case 'h':
      display_usage();
      return 0;
    case 'v':
      verbose = 1;
      break;
    case 'd':
      debugFlag = 1;
      break;
    case 'x':
      xmlDebugFlag = 1;
    case 'm':
      outFormat = FORMAT_MATLAB;
      break;
    case 's':
      outFormat = FORMAT_SCILAB;
      break;
    case 'a':
      outFormat = FORMAT_MATHEMATICA5;
      break;
    case 'e':
      outFormat = FORMAT_MATHEMATICA;
      break;
    case 'g':
      outFormat = FORMAT_GNUPLOT;
      break;
    case 'r':
      outFormat = FORMAT_R;
      break;
    case 'p':
      plotFlag = true;
      break;
    case 'o':
      outFileName = optarg;
      break;
    default:
      display_usage();
      return 0;
    }
  }

  int fnameCount;
  // process non-option command line elements
  if (optind_xmds < argc) {
    fnameCount = argc - optind_xmds;
    inFileNames = new XMLString[fnameCount];
    int i = 0;
    while (optind_xmds < argc) {
      inFileNames[i] = argv[optind_xmds];
      optind_xmds++;
      i++;
    }
  }
  else {
    // error, no input file was specified
    cerr << "Error: no input file specified.\nExiting.\n";
    display_usage();
    return 1;
  }

  if (fnameCount > 1 && outFileName.length() != 0) {
        // error, input file name already exists
      cerr << "Error: multiple input files not allowed when specifying output filename\n\n";
      display_usage();
      return 0;
  }

  // set matlab to be the default output
  if (outFormat == FORMAT_NONE) {
    cout << "Output file format defaulting to matlab.\n";
    outFormat = FORMAT_MATLAB;
  }

  // If multiple files are specified then we repeat for each one  
  for (int j=0; j<fnameCount; j++) {
    inFileName = inFileNames[j];
    // create the XMLParser
    XMLParser myXMLParser;

    // now load xmds script and results into the DOMImplementation

    Document* theDocument = 0;

    if (verbose) {
      cout << "Parsing file '" << inFileName.c_str() << "' ...\n";
    }

    try {
      theDocument = myXMLParser.parseFromFile(inFileName.c_str());
    }
    catch(XMLParserException XMLRoutinesErr) {
      cerr << "Could not load Document\n"
           << "due to the following XMLParserException:\n"
           << XMLRoutinesErr.getError()
           << "Exiting.\n";
      return 1;
    }

    if (outFileName.length() == 0) {
      // Create default output file name
      unsigned long lastdot = 0;
      for (unsigned long i=0; i<inFileName.length(); i++) {
        if (inFileName.data(i) == '.') {
          lastdot = i;
        }
      }

      if (lastdot > 0) {
        inFileName.subString(outFileName, 0, lastdot);
      }
      else {
        outFileName = inFileName;
      }

      // keep the base file name for use in xsilfield
      outFileNameBase = outFileName;

      if (outFormat == FORMAT_MATLAB) {
        cout << "Generating output for matlab\n";
        outFileName += ".m";
      }
      else if (outFormat == FORMAT_SCILAB) {
        cout << "Generating output for scilab\n";
        outFileName += ".sci";
      }
      else if (outFormat == FORMAT_MATHEMATICA5) {
        cout << "Generating output for Mathematica 5.x\n";
        outFileName += ".nb";
      }
      else if (outFormat == FORMAT_MATHEMATICA) {
        cout << "Generating output for Mathematica\n";
        outFileName += ".nb";
      }
      else if (outFormat == FORMAT_GNUPLOT) {
        cout << "Generating output for Gnuplot\n";
        outFileName += ".gnu";
      }
      else if (outFormat == FORMAT_R) {
        cout << "Generating output for R\n";
        outFileName += ".R";
      }
      else {
        cerr << "Error: unspecified output format\n";
        cerr << "I got " << outFileName.c_str() << " as the format\n";
      }

      cout << "Output file name defaulting to '" << outFileName.c_str() << "'\n";
    }
    else {
      // display what the output filename is
      cout << "Output file name set to '" << outFileName.c_str() << "'\n";
    }


    // ************************************
    // find and process xsil children
    // ************************************

    const NodeList* candidateElements;
    list<const Node*> xsilNodeList;

    if (*theDocument->documentElement()->nodeName() == "simulation") {

      candidateElements = theDocument->documentElement()->getElementsByTagName("XSIL", 0);

      if (candidateElements->length() < 1) {
        cerr << "Error: no <xsil> elements from within <simulation> element.\n"
          "Exiting.\n";
        return 1;
      }

      for (unsigned long i=0; i<candidateElements->length(); i++) {
        xsilNodeList.push_back(candidateElements->item(i));
      }
    }
    else if (*theDocument->documentElement()->nodeName() == "XSIL") {
      xsilNodeList.push_back(theDocument->documentElement());
    }
    else {
      cerr << "Error: expecting root element to be <XSIL> or <simulation>.\n"
        "Exiting.\n";
      return 1;
    }

    list<const Node*>::const_iterator ppNode = xsilNodeList.begin();

    FILE* outfile = fopen(outFileName.c_str(), "w");

    if (outfile == 0) {
      cerr << "Error: Could not open file '" << outFileName.c_str()
           << "' for writing.\nExiting.\n";
      return 1;
    }

    for (unsigned long i=0; i<xsilNodeList.size(); i++) {

      const Element* nextElement = dynamic_cast<const Element*> (*ppNode);

      const DOMString *mgName = nextElement->getAttribute("Name");
      int mgNumber = i+1;
      if (std::strcmp("breakpoint", mgName->c_str()) == 0) {
        mgNumber = 1;
      }
      else {
        sscanf(mgName->c_str(), "moment_group_%i", &mgNumber);
      }

      cout << "Processing xsil data container " << mgNumber << " ...\n";

      xsilField myxsilField;

      try {
        myxsilField.processElement(nextElement);
      }
      catch(xmdsException xmdsErr) {
        cerr << "Could not load data container\n"
             << "due to the following xsilException:\n"
             << xmdsErr.getError()
             << "Exiting.\n";
        fclose(outfile);
        return 1;
      }

      cout << "Writing data container " << i+1 << " to file ...\n";

      myxsilField.writeAsFormat(outfile, outFormat, mgNumber, outFileNameBase.c_str(), plotFlag);

      ppNode++;
    }

    fclose(outfile);
    outFileName = ""; // Clear this to generate the default output filename for the next input file
  }

  return 0;
}

/*
 * Local variables:
 * c-indentation-style: bsd
 * c-basic-offset: 2
 * indent-tabs-mode: nil
 * End:
 *
 * vim: tabstop=2 expandtab shiftwidth=2:
 */
